import logging
import multiprocessing
import os
import re
import sys
import socket
from pathlib import Path

from peacepie.assist import dir_opers, version
from peacepie.assist.auxiliaries import is_testing


instance = None
test_instance = None


def init_params(path, test_params):
    global instance
    global test_instance
    test_instance = test_params
    try:
        with open(path) as f:
            pass
    except (FileNotFoundError, TypeError):
        path = None
    if path is None:
        path = deploy_environment()
    params = []
    try:
        with open(path) as f:
            params = [line.strip().split('#')[0] for line in f.readlines()]
    except BaseException as bex:
        print(bex)
    res = {}
    for param in params:
        if params == '':
            continue
        lst = param.strip().split('=', 1)
        if len(lst) == 2 and not lst[0].strip() == '' and not lst[1].strip() == '':
            name = lst[0].strip()
            value = lst[1].strip()
            if name == 'developing_mode' or name == 'separate_log_per_process':
                value = value.lower() == 'true'
            elif name == 'inter_port':
                value = int(value)
            elif name == 'intra_port':
                value = int(value)
            elif name == 'package_dir':
                value = normalize(value)
            elif name == 'plugin_dir':
                value = normalize(value)
            elif name == 'log_dir':
                value = normalize(value)
            res[name] = value
    res['source_path'] = f'{res.get("package_dir")}/source'
    res['ip'] = get_ip()
    ver = version.version_from_string(f'{sys.version_info.major}.{sys.version_info.minor}.{sys.version_info.micro}')
    ver[version.MINOR_LEVEL] = 12
    res['python_version'] = ver
    if res.get('process_name') is None:
        res['process_name'] = multiprocessing.current_process().name
    res['peacepie_version'] = get_version()
    instance = res


def normalize(value):
    return value[:-1] if value.endswith('/') else value


def get_ip():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.settimeout(0)
    try:
        s.connect(('8.8.8.8', 1))
        res = s.getsockname()[0]
    except OSError:
        res = '127.0.0.1'
    finally:
        s.close()
    return res


def deploy_environment():
    if is_testing():
        return None
    src = f'{os.path.dirname(__file__)}/resources/config/'
    dst = f'{os.getcwd()}/config/'
    dir_opers.rem_dir(dst)
    dir_opers.copy_dir(src, dst)
    return dst + 'peacepie.cfg'


def get_version():
    try:
        base_path = Path(__file__).resolve().parent.parent
        pyproject_path = base_path / 'pyproject.toml'
        if pyproject_path.exists():
            return read_version_from_pyproject(pyproject_path)
        dist_info_dirs = list(base_path.glob('peacepie*.dist-info'))
        if dist_info_dirs:
            metadata_path = dist_info_dirs[0] / 'METADATA'
            if metadata_path.exists():
                return read_version_from_metadata(metadata_path)
    except Exception as e:
        logging.exception(e)
    return 'unknown'


def read_version_from_pyproject(pyproject_path):
    with open(pyproject_path, 'r', encoding='utf-8') as f:
        content = f.read()
    version_match = re.search(r'version\s*=\s*["\']([^"\']+)["\']', content, re.IGNORECASE)
    if version_match:
        return version_match.group(1)

    raise ValueError('Version not found in "pyproject.toml"')


def read_version_from_metadata(metadata_path):
    with open(metadata_path, 'r', encoding='utf-8') as f:
        for line in f:
            if line.lower().startswith('version:'):
                return line.split(':', 1)[1].strip()
    raise ValueError('Version not found in "METADATA"')