"""Helper classes for scripts for cluster support packages."""

from typing import List

from cyberfusion.ClusterSupport._interfaces import (
    APIObjectInterface,
    sort_lists,
)

ENDPOINT_MALWARES = "malwares"
MODEL_MALWARES = "malwares"


class Malware(APIObjectInterface):
    """Represents object."""

    _TABLE_HEADERS = [
        "ID",
        "Name",
        "Path",
        "UNIX User",
        "Last Seen",
        "Cluster",
    ]
    _TABLE_HEADERS_DETAILED: List[str] = []

    _TABLE_FIELDS = [
        "id",
        "name",
        "path",
        "_unix_user_username",
        "last_seen_at",
        "_cluster_label",
    ]
    _TABLE_FIELDS_DETAILED: List[str] = []

    @sort_lists  # type: ignore[misc]
    def _set_attributes_from_model(
        self,
        obj: dict,
    ) -> None:
        """Set class attributes from API output."""
        self.id = obj["id"]
        self.name = obj["name"]
        self.path = obj["path"]
        self.last_seen_at = obj["last_seen_at"]
        self.unix_user_id = obj["unix_user_id"]
        self.cluster_id: int = obj["cluster_id"]
        self.created_at = obj["created_at"]
        self.updated_at = obj["updated_at"]

        self.cluster = self.support.get_clusters(id_=self.cluster_id)[0]
        self.unix_user = self.support.get_unix_users(id_=self.unix_user_id)[0]

        self._cluster_label = self.cluster._label
        self._unix_user_username = self.unix_user.username

    def create(
        self, *, name: str, path: str, last_seen_at: str, unix_user_id: int
    ) -> None:
        """Create object."""
        url = f"/api/v1/{ENDPOINT_MALWARES}"
        data = {
            "name": name,
            "path": path,
            "last_seen_at": last_seen_at,
            "unix_user_id": unix_user_id,
        }

        self.support.request.POST(url, data)
        response = self.support.request.execute()

        self._set_attributes_from_model(response)

        self.support.malwares.append(self)

    def update(self) -> None:
        """Update object."""
        url = f"/api/v1/{ENDPOINT_MALWARES}/{self.id}"
        data = {
            "id": self.id,
            "name": self.name,
            "path": self.path,
            "last_seen_at": self.last_seen_at,
            "unix_user_id": self.unix_user_id,
            "cluster_id": self.cluster_id,
        }

        self.support.request.PUT(url, data)
        response = self.support.request.execute()

        self._set_attributes_from_model(response)
