"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyVertex = exports.DependencyGraph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
/**
 * Represents the dependency graph for a given Node.
 *
 * This graph includes the dependency relationships between all nodes in the
 * node (construct) sub-tree who's root is this Node.
 *
 * Note that this means that lonely nodes (no dependencies and no dependants) are also included in this graph as
 * childless children of the root node of the graph.
 *
 * The graph does not include cross-scope dependencies. That is, if a child on the current scope depends on a node
 * from a different scope, that relationship is not represented in this graph.
 *
 */
class DependencyGraph {
    constructor(node) {
        this._fosterParent = new DependencyVertex();
        const nodes = {};
        function putVertex(construct) {
            nodes[constructs_1.Node.of(construct).path] = new DependencyVertex(construct);
        }
        function getVertex(construct) {
            return nodes[constructs_1.Node.of(construct).path];
        }
        // create all vertices of the graph.
        for (const n of node.findAll()) {
            putVertex(n);
        }
        // create all the edges of the graph.
        for (const dep of node.dependencies) {
            if (!getVertex(dep.target)) {
                // dont cross scope boundaries.
                // since charts only renders its own children, this is ok and
                // has the benefit of simplifying the graph. we should reconsider this behavior when moving
                // to a more general purpose use-case.
                continue;
            }
            const sourceDepNode = getVertex(dep.source);
            const targetDepNode = getVertex(dep.target);
            sourceDepNode.addChild(targetDepNode);
        }
        // create the root.
        for (const n of Object.values(nodes)) {
            if (n.inbound.length === 0) {
                // orphans are dependency roots. lets adopt them!
                this._fosterParent.addChild(n);
            }
        }
    }
    /**
     * Returns the root of the graph.
     *
     * Note that this vertex will always have `null` as its `.value` since it is an artifical root
     * that binds all the connected spaces of the graph.
     */
    get root() {
        return this._fosterParent;
    }
    /**
     * @see Vertex.topology()
     */
    topology() {
        return this._fosterParent.topology();
    }
}
exports.DependencyGraph = DependencyGraph;
_a = JSII_RTTI_SYMBOL_1;
DependencyGraph[_a] = { fqn: "cdk8s.DependencyGraph", version: "1.7.11" };
/**
 * Represents a vertex in the graph.
 *
 * The value of each vertex is an `IConstruct` that is accessible via the `.value` getter.
 */
class DependencyVertex {
    constructor(value = undefined) {
        this._children = new Set();
        this._parents = new Set();
        this._value = value;
    }
    /**
     * Returns the IConstruct this graph vertex represents.
     *
     * `null` in case this is the root of the graph.
     */
    get value() {
        return this._value;
    }
    /**
     * Returns the children of the vertex (i.e dependencies)
     */
    get outbound() {
        return Array.from(this._children);
    }
    /**
     * Returns the parents of the vertex (i.e dependants)
     */
    get inbound() {
        return Array.from(this._parents);
    }
    /**
     * Returns a topologically sorted array of the constructs in the sub-graph.
     */
    topology() {
        const found = new Set();
        const topology = [];
        function visit(n) {
            for (const c of n.outbound) {
                visit(c);
            }
            if (!found.has(n)) {
                topology.push(n);
                found.add(n);
            }
        }
        visit(this);
        return topology.filter(d => d.value).map(d => d.value);
    }
    /**
     * Adds a vertex as a dependency of the current node.
     * Also updates the parents of `dep`, so that it contains this node as a parent.
     *
     * This operation will fail in case it creates a cycle in the graph.
     *
     * @param dep The dependency
     */
    addChild(dep) {
        const cycle = dep.findRoute(this);
        if (cycle.length !== 0) {
            cycle.push(dep);
            throw new Error(`Dependency cycle detected: ${cycle.filter(d => d.value).map(d => constructs_1.Node.of(d.value).path).join(' => ')}`);
        }
        this._children.add(dep);
        dep.addParent(this);
    }
    addParent(dep) {
        this._parents.add(dep);
    }
    findRoute(dst) {
        const route = [];
        visit(this);
        return route;
        function visit(n) {
            route.push(n);
            let found = false;
            for (const c of n.outbound) {
                if (c === dst) {
                    route.push(c);
                    return true;
                }
                found = visit(c);
            }
            if (!found) {
                route.pop();
            }
            return found;
        }
    }
}
exports.DependencyVertex = DependencyVertex;
_b = JSII_RTTI_SYMBOL_1;
DependencyVertex[_b] = { fqn: "cdk8s.DependencyVertex", version: "1.7.11" };
//# sourceMappingURL=data:application/json;base64,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