import { Duration } from './duration';
/**
 * Represents a schedule
 *
 * Note that rates cannot be defined in fractions of minutes.
 */
export declare abstract class Schedule {
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format of 'value unit'
     */
    static expression(expression: string): Schedule;
    /**
     * Construct a schedule from an interval and a time unit
     *
     * Rates may be defined with any unit of time, but when converted into minutes, the duration must be a positive whole number of minutes.
     */
    static rate(duration: Duration): Schedule;
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options: CronOptions): Schedule;
    /**
     * Retrieve the expression for this schedule
     */
    abstract readonly expressionString: string;
    protected constructor();
    /**
     *
     * @internal
     */
    abstract _bind(): void;
}
/**
 * Options to configure a cron expression
 *
 * All fields are strings so you can use complex expressions. Absence of
 * a field implies '*' or '?', whichever one is appropriate.
 */
export interface CronOptions {
    /**
     * The minute to run this rule at
     *
     * @default - Every minute
     */
    readonly minute?: string;
    /**
     * The hour to run this rule at
     *
     * @default - Every hour
     */
    readonly hour?: string;
    /**
     * The day of the month to run this rule at
     *
     * @default - Every day of the month
     */
    readonly day?: string;
    /**
     * The month to run this rule at
     *
     * @default - Every month
     */
    readonly month?: string;
    /**
     * The year to run this rule at
     *
     * @default - Every year
     */
    readonly year?: string;
    /**
     * The day of the week to run this rule at
     *
     * @default - Any day of the week
     */
    readonly weekDay?: string;
}
