import json
import requests
from pathlib import Path
from datetime import datetime
from requests.auth import HTTPBasicAuth
from typing import List, Dict, Union
from IPython.display import display, Image as IPImage


base_url = "https://ohiodot-it-api.bentley.com"


def get_assetwise_secrets():
    """Loads API secrets from a JSON file."""
    with open(Path.home() / "secrets.json", 'r') as file:
        secrets = json.load(file)
    return secrets['BENTLEY_ASSETWISE_KEY_NAME'], secrets['BENTLEY_ASSETWISE_API']


username, password = get_assetwise_secrets()


# For when you only want to look at a single field and know it's id
def get_field_definition_by_id(base_api_url: str, username: str, password: str, fe_id: int,
                               api_type: str = "api") -> dict:
    """
    Retrieves a single field definition (Field object) from the AssetWise API by its fe_id
    using Basic Auth.

    This function uses the GET /api/Field/{fe_id} endpoint.

    Args:
        base_api_url (str): The base URL of the AssetWise API (e.g., "https://ohiodot-it-api.bentley.com").
        username (str): The key name for Basic Authentication.
        password (str): The API key for Basic Authentication.
        fe_id (int): The ID of the field to retrieve.
        api_type (str, optional): The API type, "api" or "mobile". Defaults to "api".

    Returns:
        dict: A dictionary representing the field object, containing its properties
              (e.g., 'fe_name', 'fe_id', 'fe_description').
              Returns an error message string if the request fails or data cannot be parsed.
    """
    # Use the specific endpoint for getting a field by its ID
    endpoint_path = f"/{api_type}/Field/{fe_id}"  # [3]
    url = f"{base_api_url}{endpoint_path}"

    auth = HTTPBasicAuth(username, password)  #
    headers = {
        "Accept": "application/json"
    }

    try:
        # Use GET method as per documentation for /{apiType}/Field/{fe_id} [3]
        response = requests.get(url, headers=headers, auth=auth)
        response.raise_for_status()  # Raises an HTTPError for 4xx/5xx responses

        data = response.json()

        if data.get("success"):
            # For a single item return, the 'data' key directly contains the object [6]
            return data.get("data", {})
        else:
            return f"API returned an error: {data.get('errorMessage', 'Unknown error')}"

    except requests.exceptions.RequestException as req_err:
        status_code = response.status_code if response is not None else "N/A"
        response_text = response.text if response is not None else "N/A"
        return f"An error occurred: {req_err} (Status Code: {status_code}, Response: {response_text})"
    except ValueError:
        return "Failed to decode JSON response."


def get_bridge_by_sfn(
        sfn: str,
        include_coordinates: bool = True,
        include_parent: bool = False
) -> dict:
    """
    Retrieves a specific bridge asset by its SFN (Structure File Number) using the AssetWise API.

    Args:
        sfn (str): The SFN/asset code of the bridge to retrieve (e.g., '0702870').
        include_coordinates (bool): Whether to include asset coordinates in the response. Defaults to True.
        include_parent (bool): Whether to include the parent as_id in the response. Defaults to False.

    Returns:
        dict: A dictionary representing the bridge asset data.

    Raises:
        requests.exceptions.HTTPError: If the API request encounters an HTTP error.
    """
    base_url = "https://ohiodot-it-api.bentley.com"
    api_url = f"{base_url}/api/Asset/GetAssetByAsCode/{sfn}"

    query_params = {
        "IncludeCoordinates": include_coordinates,
        "IncludeParent": include_parent
    }

    headers = {
        "Accept": "application/json"
    }

    response = requests.get(
        api_url,
        params=query_params,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )

    response.raise_for_status()
    return response.json()['data']


def get_asset_report_files_metadata(
        asset_id: int,
        api_type: str = "api",
        include_asset_file: bool = True
) -> List[Dict[str, Union[int, str, bool, Dict]]]:
    """
    Retrieves metadata for files associated with a specific asset that are
    also mapped to reports.
    """
    username, password = get_assetwise_secrets()
    api_url = f"{base_url}/{api_type}/AssetFilesReportMap/GetAssetFilesForAsset/{asset_id}"
    query_params = {"IncludeAssetFile": include_asset_file}
    headers = {"Accept": "application/json"}

    print(f"Requesting URL: {api_url} with query params: {query_params}")

    response = requests.get(
        api_url,
        params=query_params,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )
    response.raise_for_status()

    print(f"Successfully retrieved file metadata for asset {asset_id}.")
    return response.json().get('data', [])


def download_asset_file_by_id(
    file_id: int,
    api_type: str = "api",
    get_as_thumbnail: bool = False
) -> bytes:
    """
    Downloads the binary content of a specific asset file.
    """
    username, password = get_assetwise_secrets()
    api_url = f"{base_url}/{api_type}/AssetFile/Download/{file_id}"
    query_params = {"get_as_thumbnail": get_as_thumbnail}
    headers = {"Accept": "application/octet-stream"}

    print(f"Requesting download for file ID: {file_id} with thumbnail option: {get_as_thumbnail}")

    response = requests.get(
        api_url,
        params=query_params,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )
    response.raise_for_status()

    print(f"Successfully downloaded file ID: {file_id}.")
    return response.content


def get_asset_cover_image(asset_id: int, api_type: str = "api") -> bytes | None:
    """
    Retrieves the designated cover image for a specific asset by its ID.

    This function directly calls the GET /{apiType}/AssetFile/GetAssetCoverImage/{as_id} endpoint,
    which is the most efficient way to get the cover photo.

    Args:
        asset_id (int): The unique ID of the asset.
        api_type (str): The API type, typically 'api'. Defaults to 'api'.

    Returns:
        bytes: The binary content of the image file, or None if no cover image is found (returns 404).
    """
    username, password = get_assetwise_secrets()
    api_url = f"{base_url}/{api_type}/AssetFile/GetAssetCoverImage/{asset_id}"
    headers = {"Accept": "application/octet-stream"}

    print(f"Requesting cover image for asset ID: {asset_id} at URL: {api_url}")

    response = requests.get(
        api_url,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )

    # A 404 status code from this endpoint specifically means no cover image was found.
    if response.status_code == 404:
        print("API returned 404: No cover image is designated for this asset.")
        return None

    # For other errors, raise an exception.
    response.raise_for_status()

    print(f"Successfully retrieved cover image for asset {asset_id}.")
    return response.content


class AssetWiseBridge:
    """
    Represents a single bridge asset from the Bentley AssetWise API, fetched by its SFN.
    It provides access to the bridge's core data, as well as its related inspections and element data.

    Example:
        bridge = AssetWiseBridge('0702870')
        print(bridge)  # Shows a summary including latest inspection

        # Access lazy-loaded properties
        latest_insp_reports = bridge.inspections
        element_data = bridge.elements
        all_snbi = bridge.snbi_data
    """

    def __init__(self, sfn: str):
        """
        Initializes the AssetWiseBridge by fetching its core data from the API.

        Args:
            sfn (str): The SFN (Structure File Number) of the bridge asset.

        Raises:
            ValueError: If no asset is found for the specified SFN.
            requests.exceptions.HTTPError: If the API request fails.
        """
        self.sfn = sfn
        raw_data = get_bridge_by_sfn(sfn, include_coordinates=True)

        if not raw_data:
            raise ValueError(f"No AssetWise asset found for SFN '{sfn}'")

        # Dynamically assign attributes from the API response
        for key, value in raw_data.items():
            # A simple conversion from camelCase to snake_case for Pythonic access
            snake_case_key = ''.join(['_' + i.lower() if i.isupper() else i for i in key]).lstrip('_')
            setattr(self, snake_case_key, value)

        # Initialize placeholders for lazy-loaded data
        self._elements = None
        self._inspections = None
        self._snbi_data = None

    @property
    def elements(self) -> List[Dict]:
        """Lazy-loads and returns all structural elements for the asset."""
        if self._elements is None:
            print("Fetching element data...")
            self._elements = get_elements_for_asset(base_url, username, password, self.as_id)
        return self._elements

    @property
    def inspections(self) -> List[Dict]:
        """Lazy-loads and returns all approved inspection reports for the asset."""
        if self._inspections is None:
            print("Fetching approved inspection reports...")
            self._inspections = get_all_approved_inspections(self.as_id)
        return self._inspections

    @property
    def snbi_data(self) -> Dict:
        """Lazy-loads, formats, and returns all SNBI data for the asset."""
        if self._snbi_data is None:
            print("Fetching and formatting all SNBI data...")
            raw_data = get_all_odot_snbi_data(self.as_id)
            self._snbi_data = format_assetwise_output(raw_data)
        return self._snbi_data

    def get_all_inspection_data(self, reports_to_return: int = 5) -> Dict:
        """
        Retrieves and processes full field values for the most recent inspection reports.
        """
        print(f"Fetching full data for the {reports_to_return} most recent inspection reports...")
        # Get the most recent reports (not necessarily approved)
        recent_reports = get_inspection_reports_for_asset(self.as_id, reports_to_return=reports_to_return)

        # Re-use the logic from your notebook's get_all_inspection_data function
        # We pass an empty dict for field_ids to let it build its cache.
        # We pass an empty dict for field_ids to let it build its cache.
        return get_all_inspection_data(recent_reports, {})

    def get_cover_photo(self, display_photo: bool = True):
        """
        Finds and optionally displays the full-resolution designated cover photo for the asset.

        Args:
            display_photo (bool): If True, the photo will be displayed in the output.
                                  Defaults to True.

        Returns:
            bytes: The binary content of the image file, or None if not found.
        """
        print("Searching for the designated cover photo file...")
        try:
            # Directly call the efficient endpoint to get the cover image.
            image_bytes = get_asset_cover_image(self.as_id)

            if image_bytes and display_photo:
                print("Displaying cover photo:")
                display(IPImage(data=image_bytes))

            # The get_asset_cover_image function returns None on 404,
            # and logs the appropriate message.
            return image_bytes
        except Exception as e:
            print(f"Failed to download or display photo: {e}")
            return None

    def __repr__(self) -> str:
        """
        Provides a nicely formatted, developer-friendly summary of the AssetWise bridge.
        """
        # Helper to safely get attributes
        def get(attr, default='N/A'):
            return getattr(self, attr, default)

        # Create a clickable Google Maps link from coordinates
        map_url = "N/A"
        if hasattr(self, 'coordinates') and self.coordinates:
            lat = self.coordinates.get('latitude')
            lon = self.coordinates.get('longitude')
            if lat and lon:
                map_url = f"https://www.google.com/maps?q={lat},{lon}"

        asset_type_name = get('at_name', 'N/A')
        if hasattr(self, 'asset_type') and isinstance(self.asset_type, dict):
            asset_type_name = self.asset_type.get('at_name', 'N/A')

        # Add latest inspection info
        latest_inspection_info = "No approved inspections found."
        if self.inspections:
            latest_report = self.inspections[0]
            try:
                dt = datetime.fromisoformat(latest_report['ast_date'])
                date_str = dt.strftime('%m/%d/%Y')

                # Safely access nested inspection type name from the correct field
                insp_type = "N/A"
                type_maps = latest_report.get('inspectionReportInspTypeMaps')
                if type_maps and len(type_maps) > 0:
                    insp_type_obj = type_maps[0].get('inspectionType')
                    if insp_type_obj:
                        insp_type = insp_type_obj.get('it_name', 'Unknown Type')

                latest_inspection_info = f"{date_str} ({insp_type})"
            except (KeyError, IndexError, TypeError):
                latest_inspection_info = "Data format error."

        element_count = len(self.elements) if self._elements is not None else "Not yet fetched"

        repr_str = (
            f"<AssetWiseBridge SFN: '{self.sfn}'>\n"
            f"  Asset Name:    {get('as_name', 'No Name')}\n"
            f"  Asset ID:      {get('as_id')}\n"
            f"  Asset Type:    {asset_type_name}\n"
            f"  Description:   {get('as_description', 'No Description')}\n"
            f"  Location Map:  {map_url}\n"
            f"  Elements:      {element_count} defined\n"
            f"  Latest Insp:   {latest_inspection_info}\n"
            f"\nFor full data, access properties like .elements, .inspections, or .snbi_data"
        )
        return repr_str


def get_elements_for_asset(base_api_url: str, username: str, password: str, as_id: int, api_type: str = "api") -> List[
    Dict]:
    """
    Retrieves elements and their condition states for a specific asset.
    Uses the GET /{apiType}/StructureElement/GetElements/{objectType}/{objectId}/{asId}/{segmentId}/{elementId} endpoint.
    objectType for Asset is 0. [4, 6, 25-51]
    segmentId and elementId are set to 0 to indicate retrieval of all segments and elements under the given asset,
    based on common API conventions for integer path parameters.
    """
    object_type_asset = 0  # As per ObjectType enum, 0 = Asset [128, etc.]

    # Construct the endpoint path to get all elements for a specific asset.
    # We use objectType, objectId (as_id), asId, segmentId, and elementId.
    # The segmentId parameter is noted as 'allowEmptyValue' in the documentation [52],
    # and 0 is a common convention for "all" or "not specified" for integer IDs in API paths.
    # We apply the same convention for elementId, which is also a required integer path parameters
    endpoint_path = f"/{api_type}/StructureElement/GetElements/{object_type_asset}/{as_id}/{as_id}/0/0"
    url = f"{base_api_url}{endpoint_path}"

    auth = HTTPBasicAuth(username, password)  # [13-15]
    headers = {
        "Accept": "application/json"  # [13-15, 53]
    }

    response = None  # Define response here so it's available in except block
    try:
        response = requests.get(url, headers=headers, auth=auth)  # Use GET method [4-6]
        response.raise_for_status()  # Raise an exception for HTTP errors [13-15, 20]

        data = response.json()

        if data.get("success"):  # Check if the API response was successful [21]
            # The 'data' field contains the list of Elements objects [5, 6, 54]
            # Each 'Elements' object includes an 'elementState' with condition details [1-3]
            return data.get("data", [])
        else:
            print(
                f"API returned an error while fetching elements for asset ID {as_id}: {data.get('errorMessage', 'Unknown error')}")
            return []

    except requests.exceptions.RequestException as req_err:
        status_code = response.status_code if response is not None else "N/A"
        response_text = response.text if response is not None else "N/A"
        print(
            f"An HTTP request error occurred while fetching elements for asset ID {as_id}: {req_err} (Status Code: {status_code}, Response: {response_text})")
        return []
    except ValueError:
        print(f"Failed to decode JSON response for elements for asset ID {as_id}.")
        return []


def get_inspection_reports_for_asset(
        asset_id: int,
        reports_to_return: int = 99,  # Defaults to 5 of the most recent reports
        api_type: str = "api"
) -> List[Dict[str, Union[int, str, bool, Dict]]]:
    """
    Retrieves recent inspection reports for a specific asset, regardless of report status.

    Args:
        asset_id (int): The unique ID of the asset.
        reports_to_return (int): The number of most recent reports to return. Defaults to 5.
        api_type (str): The API type, typically 'api'. Defaults to 'api' [20].

    Returns:
        List[Dict]: A list of dictionaries, each representing an InspectionReportHelper object.
                    Returns an empty list if no data is found or an error occurs.
    Raises:
        requests.exceptions.HTTPError: If the API request encounters an HTTP error.
    """
    username, password = get_assetwise_secrets()
    api_url = f"{base_url}/{api_type}/InspectionReport/GetMostRecentReportsForAsset/{asset_id}"
    query_params = {"reportsToReturn": reports_to_return}
    headers = {"Accept": "application/json"}

    response = requests.get(
        api_url,
        params=query_params,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )
    response.raise_for_status()  # Raise an exception for HTTP errors

    # The 'data' field contains the list of InspectionReportHelper objects
    return response.json().get('data', [])


def get_full_inspection_report(
        ast_id: int,
        api_type: str = "api"
) -> Dict[str, Union[int, str, bool, List, Dict]]:
    """
    Retrieves the full details of a specific inspection report by its Asset Task ID (ast_id).
    This includes all fields, such as recorded condition states within the 'values' array,
    and various inspection dates.

    Args:
        ast_id (int): The unique Asset Task ID (e.g., 332776) of the inspection report.
                      This ID is provided by the InspectionReportHelper.
        api_type (str): The API type, typically 'api'. Defaults to 'api'.

    Returns:
        Dict: A dictionary representing the complete InspectionReport object.
              This object contains fields like 'ast_inspection_date', 'ast_begin_date',
              'ast_end_date', and a 'values' array, where detailed condition states
              and their specific recording dates are expected.

    Raises:
        requests.exceptions.HTTPError: If the API request encounters an HTTP error (e.g., 4xx or 5xx response).
    """
    username, password = get_assetwise_secrets()  # Authenticates the request
    base_url = "https://ohiodot-it-api.bentley.com"  # Base URL for the AssetWise API

    # Constructs the API endpoint for retrieving a single InspectionReport by ast_id
    api_url = f"{base_url}/{api_type}/Value/GetValuesForReport/{ast_id}"

    headers = {
        "Accept": "application/json"  # Specifies that the client expects a JSON response
    }

    print(f"Requesting full inspection report for AST ID: {ast_id} at URL: {api_url}")

    response = requests.get(
        api_url,
        headers=headers,
        auth=HTTPBasicAuth(username, password)
    )

    # Raises an HTTPError for bad responses (4xx or 5xx)
    response.raise_for_status()

    print(f"Successfully retrieved full inspection report for AST ID: {ast_id}.")

    # Extracts the 'data' field from the JSON response, which contains the InspectionReport object
    return response.json().get('data', {})


def get_all_inspection_data(inspection_reports, field_ids):
    i = 0
    all_inspection_values = {}

    for inspection in inspection_reports:
        dt = datetime.fromisoformat(inspection['ast_date'])
        inspection_name = f"{dt.strftime('%m/%d/%Y')} - {inspection['reportType']['rt_name']} - {inspection['inspectionTypes'][0]['it_name']}"
        all_inspection_values[inspection_name] = {}
        print(f"{inspection_name}\n")

        inspection_values = get_full_inspection_report(inspection['ast_id'])

        for field in inspection_values:  # Very Slow, probably because of get_field_definition_by_id, should get built out ahead of time instead
            if field['fe_id'] in field_ids:
                field_values = {}
                field_values['fe_name'] = field_ids[field['fe_id']]
            else:
                field_values = get_field_definition_by_id(base_url, username, password, field['fe_id'])
                field_ids[field['fe_id']] = field_values
                print("**** NEW VALUE FOUND, THE DATABASE SHOULD BE UPDATED ****")

            key = field['fe_id']
            all_inspection_values[inspection_name][key] = (field_values['fe_name'], field['va_value'])

            i += 1

    return all_inspection_values


def get_all_odot_snbi_data(asset_id: int):
    username, password = get_assetwise_secrets()
    base_url = "https://ohiodot-it-api.bentley.com"  # Base URL for the AssetWise API
    response_data = []

    for i in range(0, 6, 1):
        api_url = f"{base_url}/api/FormElement/GetRfgTemplateElements/{asset_id}/0/100000{i}"
        print(api_url)

        headers = {
            "Accept": "application/json"  # Specifies that the client expects a JSON response
        }

        print(
            f"Requesting all current values for Asset ID: {asset_id} at URL: {api_url} "
        )

        response = requests.get(
            api_url,
            headers=headers,
            auth=HTTPBasicAuth(username, password)  # Authenticates the request
        )

        response.raise_for_status()  # Raise an exception for HTTP errors

        print(f"Successfully retrieved all current values for Asset ID: {asset_id}.")

        response_data.append(response.json()['data'])

    return response_data


def format_assetwise_output(response):
    element_dict = {}

    # Step 1: Group elements by el_id
    for page in response:
        for instance in page:
            for element in instance['elements']:
                el_id = element['el_id']
                # Ensure field is not None before trying to access it
                field = element.get('field')
                value = (element.get('value'), field)
                element_dict.setdefault(el_id, []).append(value)

    # Step 2: Organize values into a clean structure
    organized_dict = {}

    for el_id, entries in element_dict.items():
        organized_dict[el_id] = []

        for value, field in entries:
            # Define a safe label, checking if field exists
            label = field['fe_name'] if field else f"Unnamed Field (el_id: {el_id})"

            # If the value is a list (but not a string), unpack it
            if isinstance(value, list) and not isinstance(value, str):
                for item in value:
                    organized_dict[el_id].append({
                        'label': label,
                        'value': item.get('value') if isinstance(item, dict) else item
                    })
            else:
                organized_dict[el_id].append({
                    'label': label,
                    'value': value
                })

    return organized_dict


def get_all_approved_inspections(as_id: int, api_type: str = "api") -> List[Dict]:
    """
    Retrieves ALL approved inspection reports for a specific asset.
    Uses the GET /{apiType}/InspectionReport/GetAllApproved/{as_id} endpoint.

    Args:
        as_id (int): The unique ID of the asset.
        api_type (str): The API type, typically 'api'. Defaults to 'api'.

    Returns:
        List[Dict]: A list of dictionaries, each representing an InspectionReportHelper object.
                    Returns an empty list if no data is found or an error occurs.
    """
    # 1. Get auth secrets, consistent with other functions
    username, password = get_assetwise_secrets()

    # 2. Construct URL from global base_url
    api_url = f"{base_url}/{api_type}/InspectionReport/GetAllApproved/{as_id}"

    # 3. Set headers and auth
    headers = {"Accept": "application/json"}
    auth = HTTPBasicAuth(username, password)

    # 4. Add logging
    print(f"Requesting all approved inspection reports for asset ID: {as_id} at URL: {api_url}")

    # 5. Use a robust try/except block for the request
    response = None  # Define response here so it's available in except block
    try:
        response = requests.get(api_url, headers=headers, auth=auth)
        response.raise_for_status()  # Raise an exception for HTTP errors (4xx, 5xx)

        data = response.json()

        if data.get("success"):
            print(f"Successfully retrieved all approved inspection reports for asset {as_id}.")
            # The 'data' field contains the list of InspectionReportHelper objects
            return data.get("data", [])
        else:
            # Handle API-level error (success=false)
            print(f"API returned an error for asset {as_id}: {data.get('errorMessage', 'Unknown error')}")
            return []

    except requests.exceptions.RequestException as req_err:
        status_code = response.status_code if response is not None else "N/A"
        response_text = response.text if response is not None else "N/CSS"
        print(
            f"An HTTP request error occurred for asset {as_id}: {req_err} (Status Code: {status_code}, Response: {response_text})")
        return []
    except ValueError:  # Handle JSON decode error
        print(f"Failed to decode JSON response for asset ID {as_id}.")
        return []


def get_all_bridges_paged(
        starting_row: int = 0,
        items_per_page: int = 100,
        api_type: str = "api",
        include_coordinates: bool = False,
        include_parent: bool = False
):
    """
    Retrieves a paged list of assets from the AssetWise API using a POST request.

    Args:
        starting_row (int): The starting row for the page (0-indexed). Defaults to 0.
        items_per_page (int): The number of items to return per page. Defaults to 100.
        api_type (str): The API type, either "api" or "mobile". Defaults to "api".
        include_coordinates (bool): Whether to include asset coordinates in the response. Defaults to False.
        include_parent (bool): Whether to include the parent as_id in the response. Defaults to False.

    Returns:
        dict: A dictionary containing the API response data for assets, including pagination info.

    Raises:
        requests.exceptions.HTTPError: If the API request returns a non-200 status.
    """
    username, password = get_assetwise_secrets()

    api_url = f"https://ohiodot-it-api.bentley.com/{api_type}/Asset/GetAssets"

    # Query parameters (IncludeCoordinates and IncludeParent are query params for both GET and POST)
    query_params = {
        "IncludeCoordinates": include_coordinates,
        "IncludeParent": include_parent
    }

    headers = {
        "Accept": "application/json",
        "Content-Type": "application/json"  # Essential for sending JSON in the request body
    }

    # Request body for pagination, matching the RequestPaging schema
    request_body = {
        "starting": starting_row,
        "count": items_per_page
    }

    print(f"Requesting URL: {api_url} with query params: {query_params}, body: {request_body}")

    response = requests.post(
        api_url,
        params=query_params,
        headers=headers,
        auth=HTTPBasicAuth(username, password),
        json=request_body  # Send pagination data as JSON in the body
    )

    response.raise_for_status()  # Raise an exception for HTTP errors

    print("Successfully retrieved paged assets.")
    return response.json()
