"""Base module"""

import enum
from abc import ABC
from typing import Any


@enum.unique
class HypervisorPlatform(enum.Enum):
    """Type of virtualization platform"""

    KVM = "kvm"
    HYPERV = "hyperv"


class Base(ABC):
    """Base class"""

    host: str
    platform: HypervisorPlatform
    auth: tuple[str, str]

    __driver_map__: dict[HypervisorPlatform, Any]

    def __init__(self, driver) -> None:
        self.driver = driver

    @classmethod
    async def connect(cls, platform: HypervisorPlatform, **driver_data):
        """Connect to driver"""
        if not hasattr(cls, "__driver_map__") or not cls.__driver_map__:
            raise NotImplementedError("Subclasses must define 'DRIVER_MAP' or override __post_init__")

        driver_cls = cls.__driver_map__.get(platform)
        if not driver_cls:
            raise ValueError("Unknown type of hypervisor")

        driver = await driver_cls.connect(**driver_data)

        return cls(driver=driver)
