"Hyper-V hypervisor driver"

import os

from .base import HyperVirtualDriver
from .vm import HyperVirtualMachineDriver


class HyperVirtualHostDriver(HyperVirtualDriver):
    """Driver for managing Hyper-V server"""

    async def get_vms(self) -> list[HyperVirtualMachineDriver]:
        """Get list of virtual machines on the hypervisor"""
        async with self.get_ps() as ps:
            ps.add_command("Get-VM")
            ps.add_command("Select")
            ps.add_parameter("Property", "Id")
            result = await self.exec_ps(ps)

        return [HyperVirtualMachineDriver(conn=self.conn, uuid=str(r.Id)) for r in result]

    async def import_vm(self, source: str, storage: str, name: str) -> str:
        """Import a virtual machine from the source"""
        virtual_machine_path = snapshot_file_path = os.path.join(storage + ":", "hyperv", name)
        vhd_destination_path = os.path.join(virtual_machine_path, "Virtual Hard Disks")
        source_path = os.path.join(source, "Virtual Machines")

        async with self.get_ps() as ps:
            ps.add_command("Get-ChildItem")
            ps.add_parameters(Path=source_path, Filter="*.vmcx")
            ps.add_parameter("Recurse", True)
            result = await self.exec_ps(ps)
        vmcx_path = result[0].FullName

        async with self.get_ps() as ps:
            ps.add_command("Import-VM")
            ps.add_parameters(
                Path=vmcx_path,
                GenerateNewId=True,
                Copy=True,
                VirtualMachinePath=virtual_machine_path,
                VhdDestinationPath=vhd_destination_path,
                SnapshotFilePath=snapshot_file_path,
            )
            ps.add_command("Rename-VM")
            ps.add_parameters(NewName=name, Passthru=True)
            ps.add_command("Select")
            ps.add_parameter("Property", "Id")
            result = await self.exec_ps(ps)
        return str(result[0].Id)

    async def get_storages(self) -> list:
        """Get information about the host storage systems"""
        async with self.get_ps() as ps:
            ps.add_command("Get-WmiObject")
            ps.add_argument("win32_logicaldisk")
            result = await self.exec_ps(ps)
        storages = []
        for s in result:
            if s.DriveType != 3:
                continue
            storages.append(
                {
                    "name": s.DeviceID[0],
                    "size": s.Size,
                    "free": s.FreeSpace,
                }
            )
        return storages
