"KVM drivers"

import asyncio
from typing import Any, Callable

import libvirt


def request_cred(user: str, password: str):
    """Callback function for authentication"""

    def inner(credentials: list, user_data):  # pylint: disable=unused-argument
        for credential in credentials:
            if credential[0] == libvirt.VIR_CRED_AUTHNAME:
                credential[4] = user
            elif credential[0] == libvirt.VIR_CRED_PASSPHRASE:
                credential[4] = password
        return 0

    return inner


async def to_async(func: Callable, *args, **kwargs) -> Any:
    return await asyncio.to_thread(func, *args, **kwargs)


class KernelVirtualDriver:
    """Common class for connecting to KVM server"""

    conn: libvirt.virConnect
    ssh_key: str | None = None

    def __init__(self, conn: libvirt.virConnect) -> None:
        self.conn = conn

    @classmethod
    async def connect(cls, host: str, auth: tuple[str, str], *args, **kwargs):
        """Connect to driver"""
        conn = await cls._connect(host=host, auth=auth)
        return cls(conn=conn, *args, **kwargs)

    @classmethod
    async def _connect(cls, host: str, auth: tuple[str, str]) -> libvirt.virConnect:
        user, password = auth
        if cls.ssh_key:
            uri = f"qemu+libssh2://{user}@{host}/system?sshauth=privkey&keyfile={cls.ssh_key}&known_hosts_verify=auto"
            conn = await asyncio.to_thread(libvirt.open, uri)
        elif user and password:
            authd = [
                [libvirt.VIR_CRED_AUTHNAME, libvirt.VIR_CRED_PASSPHRASE],
                request_cred(user, password),
                None,
            ]
            conn = await asyncio.to_thread(
                libvirt.openAuth,
                f"qemu+libssh2://{user}@{host}/system?sshauth=password&known_hosts_verify=auto",
                authd,
                0,
            )
        else:
            raise ValueError("At least one SSH key or password is required")

        if conn is None:
            raise ConnectionError(f"Failed to connect to KVM host {host}")
        return conn
