"Handle libvirt exceptions"

from functools import wraps

import libvirt

from ...exceptions import DiskNotFoundError, VirtualMachineError, VirtualMachineNotFoundError, VirtWrapperError

KVM_EXCEPTIONS_MAP = {
    libvirt.VIR_ERR_NO_DOMAIN: VirtualMachineNotFoundError,
    libvirt.VIR_ERR_NO_STORAGE_VOL: DiskNotFoundError,
    libvirt.VIR_ERR_INTERNAL_ERROR: VirtualMachineError,
}


def wrap_libvirt(default_exc=VirtWrapperError):
    """Wrapper for handling libvirt exceptions"""

    def wrap_method(fn):
        @wraps(fn)
        async def wrapper(*args, **kwargs):
            try:
                return await fn(*args, **kwargs)
            except libvirt.libvirtError as e:
                code = e.get_error_code()
                if not code:
                    raise default_exc from e
                exc_type = KVM_EXCEPTIONS_MAP.get(code, default_exc)
                raise exc_type(str(e)) from e

        return wrapper

    def decorator(cls):
        for name, attr in cls.__dict__.items():
            if name.startswith("_"):
                continue
            if callable(attr):
                setattr(cls, name, wrap_method(attr))
            elif isinstance(attr, classmethod):
                setattr(cls, name, classmethod(wrap_method(attr.__func__)))
            elif isinstance(attr, staticmethod):
                setattr(cls, name, classmethod(wrap_method(attr.__func__)))
        return cls

    return decorator
