"KVM hypervisor driver"

import os
from datetime import datetime
from random import randint
from xml.etree import ElementTree as ET

import libvirt

from .base import KernelVirtualDriver, to_async
from .vm import KernelVirtualMachineDriver


class KernelVirtualHostDriver(KernelVirtualDriver):
    """Driver for managing KVM server"""

    async def get_vms(self) -> list[KernelVirtualMachineDriver]:
        """Get list of virtual machines on the hypervisor"""
        domains = await to_async(self.conn.listAllDomains)
        return [KernelVirtualMachineDriver(conn=self.conn, domain=domain) for domain in domains]

    async def import_vm(self, source: str, storage: str, name: str) -> str:
        """Import a virtual machine from the source"""
        xml_pool = f"""<pool type='dir'>
  <name>import_{name}</name>
  <target>
    <path>{source}</path>
  </target>
</pool>"""
        root = ET.parse(os.path.join(source, "config.xml")).getroot()

        uuid = root.find("uuid")
        if uuid is not None:
            root.remove(uuid)

        xml_name = root.find("name")
        if xml_name is None:
            xml_name = ET.SubElement(root, "name")
            root.insert(3, xml_name)
        xml_name.text = name

        xml_title = root.find("title")
        if xml_title is None:
            xml_title = ET.SubElement(root, "title")
            root.insert(3, xml_title)
        xml_title.text = name

        def random_suffix():
            timestamp = int((datetime.now() - datetime(1970, 1, 1)).total_seconds())
            rand_hex = "".join(f"{randint(0x00, 0xFF):02x}" for _ in range(10))
            return f"_{timestamp}_{rand_hex}"

        target_pool = await to_async(self.conn.storagePoolLookupByName, storage)
        import_pool = await to_async(
            self.conn.storagePoolCreateXML, xml_pool, libvirt.VIR_STORAGE_POOL_CREATE_WITH_BUILD
        )
        imported_volumes = []
        try:
            for src in root.iterfind("devices/disk/source"):
                source_file = src.get("file") or src.get("volume")
                if source_file is None:
                    raise ValueError("Unable to find virtual disk")
                orig_filename = os.path.basename(source_file)
                import_vol = await to_async(import_pool.storageVolLookupByName, orig_filename)

                disk_suffix = random_suffix()
                filename = orig_filename
                while filename in await to_async(target_pool.listVolumes):
                    filename = name + disk_suffix + os.path.splitext(orig_filename)[-1]
                    disk_suffix = random_suffix()

                xml_vol = f"""<volume>
  <name>{filename}</name>
  <target>
    <format type='qcow2'/>
    <permissions>
      <mode>0600</mode>
      <label>virt_image_t</label>
    </permissions>
  </target>
</volume>"""
                target_vol = target_pool.createXMLFrom(xml_vol, import_vol, 0)
                if src.get("file"):
                    src.set("file", target_vol.path())
                elif src.get("volume"):
                    src.set("pool", target_pool.name())
                    src.set("volume", target_vol.name())

                imported_volumes.append(target_vol)
        except Exception as e:
            for volume in imported_volumes:
                await to_async(volume.destroy)
            raise e
        finally:
            await to_async(import_pool.destroy)

        for interface in root.findall("devices/interface"):
            for remove_target in ("mac", "address"):  # Remove some options, they will be chosen automatically
                interface_to_remove = interface.find(remove_target)
                if interface_to_remove is None:
                    continue
                interface.remove(interface_to_remove)

        result_domain = await to_async(self.conn.defineXML, ET.tostring(root, encoding="unicode"))
        return result_domain.UUIDString()

    async def get_storages(self) -> list[dict]:
        """Get information about the host storage systems"""
        result = []
        for pool in await to_async(self.conn.listAllStoragePools):
            info = pool.info()
            result.append(
                {
                    "name": pool.name(),
                    "size": info[1],
                    "free": info[3],
                }
            )
        return result
