"""Hypervisors management module"""

from .base import Base, HypervisorPlatform
from .drivers.hyperv import HyperVirtualHostDriver
from .drivers.kvm import KernelVirtualHostDriver
from .storage import Storage
from .vm import VirtualMachine


class Hypervisor(Base):
    """Essential class for managing the hypervisor"""

    __driver_map__ = {
        HypervisorPlatform.KVM: KernelVirtualHostDriver,
        HypervisorPlatform.HYPERV: HyperVirtualHostDriver,
    }

    driver: KernelVirtualHostDriver | HyperVirtualHostDriver

    async def virtual_machines(self) -> list[VirtualMachine]:
        """Get list of virtual machines on the hypervisor"""
        vms = await self.driver.get_vms()
        return [VirtualMachine(driver=driver) for driver in vms]

    async def import_vm(self, source: str, storage: str, name: str) -> VirtualMachine:
        """Import a virtual machine from a source path"""
        return VirtualMachine(
            driver=await self.driver.import_vm(source=source, storage=storage, name=name),
        )

    async def storages(self) -> list[Storage]:
        """Get information about the host storage systems"""
        return [Storage(**storage) for storage in await self.driver.get_storages()]
