"""Virtual machine management module"""

import enum

from .base import Base, HypervisorPlatform
from .disk import VirtualDisk
from .drivers.hyperv import HyperVirtualMachineDriver
from .drivers.kvm import KernelVirtualMachineDriver
from .exceptions import SnapshotNotFoundError
from .memory import MemoryStat
from .network import VirtualNetworInteface
from .snapshot import Snapshot


@enum.unique
class VirtualMachineState(enum.Enum):
    """Possible virtual machine states"""

    # Common states
    RUNNING = "Running"
    PAUSED = "Paused"
    # KVM states

    BLOCKED = "Blocked"
    SHUTDOWN = "Shutdown"
    SHUTOFF = "Shutoff"
    CRASHED = "Crashed"
    NOSTATE = "No state"

    # Hyper-V states
    OTHER = "Other"
    OFF = "Off"
    STOPPING = "Stopping"
    SAVED = "Saved"
    STARTING = "Starting"
    RESET = "Reset"
    SAVING = "Saving"
    PAUSING = "Pausing"
    RESUMING = "Resuming"
    FAST_SAVED = "FastSaved"
    FAST_SAVING = "FastSaving"
    FORCE_SHUTDOWN = "ForceShutdown"
    FORCE_REBOOT = "ForceReboot"
    HIBERNATED = "Hibernated"
    RUNNING_CRITICAL = "RunningCritical"
    OFF_CRITICAL = "OffCritical"
    STOPPING_CRITICAL = "StoppingCritical"
    SAVED_CRITICAL = "SavedCritical"
    PAUSED_CRITICAL = "PausedCritical"
    STARTING_CRITICAL = "StartingCritical"
    RESET_CRITICAL = "ResetCritical"
    SAVING_CRITICAL = "SavingCritical"
    PAUSING_CRITICAL = "PausingCritical"
    RESUMING_CRITICAL = "ResumingCritical"
    FAST_SAVED_CRITICAL = "FastSavedCritical"
    FAST_SAVING_CRITICAL = "FastSavingCritical"


class VirtualMachine(Base):
    """Essential class for managing the virtual machine"""

    __driver_map__ = {
        HypervisorPlatform.KVM: KernelVirtualMachineDriver,
        HypervisorPlatform.HYPERV: HyperVirtualMachineDriver,
    }
    driver: KernelVirtualMachineDriver | HyperVirtualMachineDriver

    @property
    def id(self) -> str:
        return self.driver.id

    async def name(self) -> str:
        """Get the virtual machine name"""
        return await self.driver.get_name()

    async def set_name(self, name: str) -> None:
        """Set new the virtual machine name"""
        await self.driver.set_name(name=name)

    async def state(self) -> VirtualMachineState:
        """Get the virtual machine state"""
        return VirtualMachineState(await self.driver.get_state())

    async def description(self) -> str | None:
        """Get the virtual machine description"""
        return await self.driver.get_description()

    async def guest_os(self) -> str | None:
        """Get the name of the virtual machine guest operating system"""
        return await self.driver.get_guest_os()

    async def memory_stat(self) -> MemoryStat:
        """Get the memory statistic of the virtual machine"""
        memory_stat = await self.driver.get_memory_stat()
        return MemoryStat(**memory_stat)

    async def cpus(self) -> int:
        """Get cores number"""
        return await self.driver.get_cpus()

    async def set_cpus(self, cpus: int) -> None:
        """Change number of cores"""
        await self.driver.set_cpus(cpus=cpus)

    async def snapshots(self) -> list[Snapshot]:
        """Get the list of the virtual machine snapshots"""
        return [Snapshot(**snapshot) for snapshot in await self.driver.get_snapshots()]

    async def disks(self) -> list[VirtualDisk]:
        """Get the list of the virtual machine connected disks"""
        return [VirtualDisk(**disk) for disk in await self.driver.get_disks()]

    async def networks(self) -> list[VirtualNetworInteface]:
        """Get the list of the virtual machine network adapters"""
        return [VirtualNetworInteface(**net) for net in await self.driver.get_networks()]

    async def run(self) -> None:
        """Power on the virtual machine"""
        await self.driver.run()

    async def shutdown(self) -> None:
        """Shutdown the virtual machine"""
        await self.driver.shutdown()

    async def poweroff(self) -> None:
        """Force off the virtual machine"""
        await self.driver.poweroff()

    async def save(self) -> None:
        """Pause the virtual machine and temporarily saving its memory state to a file"""
        await self.driver.save()

    async def suspend(self) -> None:
        """Pause the virtual machine and temporarily saving its memory state"""
        await self.driver.suspend()

    async def resume(self) -> None:
        """Unpause the suspended virtual machine"""
        await self.driver.resume()

    async def snap_create(self, name: str, description: str) -> Snapshot:
        """Create a new snapshot of virtual machine"""
        await self.driver.snapshot_create(name=name, description=description)
        for s in await self.driver.get_snapshots():
            if s["name"] == name:
                return Snapshot(**s)
        raise SnapshotNotFoundError("Unable to find snapshot after creating it")

    async def export(self, storage: str) -> str:
        """Export the virtual machine to a storage destination"""
        return await self.driver.export(storage=storage)
