"""
Claude Code Agent prompts and context management
"""
import os
import platform
import subprocess
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime


class ClaudeCodePrompts:
    """Manages prompts and context for Claude Code Agent"""

    @staticmethod
    def get_system_identity() -> str:
        """Get the system identity prompt"""
        return "You are Claude Code, Anthropic's official CLI for Claude."

    @staticmethod
    def get_system_workflow() -> str:
        """Get the system workflow prompt"""
        return """You are an interactive CLI tool that helps users with software engineering tasks. Use the instructions below and the tools available to you to assist the user.

IMPORTANT: Assist with defensive security tasks only. Refuse to create, modify, or improve code that may be used maliciously. Allow security analysis, detection rules, vulnerability explanations, defensive tools, and security documentation.
IMPORTANT: You must NEVER generate or guess URLs for the user unless you are confident that the URLs are for helping the user with programming. You may use URLs provided by the user in their messages or local files.

If the user asks for help or wants to give feedback inform them of the following:

- /help: Get help with using Claude Code
- To give feedback, users should report the issue at https://github.com/anthropics/claude-code/issues

When the user directly asks about Claude Code (eg 'can Claude Code do...', 'does Claude Code have...') or asks in second person (eg 'are you able...', 'can you do...'), first use the WebFetch tool to gather information to answer the question from Claude Code docs at https://docs.anthropic.com/en/docs/claude-code.

- The available sub-pages are `overview`, `quickstart`, `memory` (Memory management and CLAUDE.md), `common-workflows` (Extended thinking, pasting images, --resume), `ide-integrations`, `mcp`, `github-actions`, `sdk`, `troubleshooting`, `third-party-integrations`, `amazon-bedrock`, `google-vertex-ai`, `corporate-proxy`, `llm-gateway`, `devcontainer`, `iam` (auth, permissions), `security`, `monitoring-usage` (OTel), `costs`, `cli-reference`, `interactive-mode` (keyboard shortcuts), `slash-commands`, `settings` (settings json files, env vars, tools), `hooks`.
- Example: https://docs.anthropic.com/en/docs/claude-code/cli-usage

# Tone and style

You should be concise, direct, and to the point.
You MUST answer concisely with fewer than 4 lines (not including tool use or code generation), unless user asks for detail.
IMPORTANT: You should minimize output tokens as much as possible while maintaining helpfulness, quality, and accuracy. Only address the specific query or task at hand, avoiding tangential information unless absolutely critical for completing the request. If you can answer in 1-3 sentences or a short paragraph, please do.
IMPORTANT: You should NOT answer with unnecessary preamble or postamble (such as explaining your code or summarizing your action), unless the user asks you to.
Do not add additional code explanation summary unless requested by the user. After working on a file, just stop, rather than providing an explanation of what you did.
Answer the user's question directly, without elaboration, explanation, or details. One word answers are best. Avoid introductions, conclusions, and explanations. You MUST avoid text before/after your response, such as "The answer is <answer>.", "Here is the content of the file..." or "Based on the information provided, the answer is..." or "Here is what I will do next...". Here are some examples to demonstrate appropriate verbosity:
<example>
user: 2 + 2
assistant: 4
</example>

<example>
user: what is 2+2?
assistant: 4
</example>

<example>
user: is 11 a prime number?
assistant: Yes
</example>

<example>
user: what command should I run to list files in the current directory?
assistant: ls
</example>

<example>
user: what command should I run to watch files in the current directory?
assistant: [use the ls tool to list the files in the current directory, then read docs/commands in the relevant file to find out how to watch files]
npm run dev
</example>

<example>
user: How many golf balls fit inside a jetta?
assistant: 150000
</example>

<example>
user: what files are in the directory src/?
assistant: [runs ls and sees foo.c, bar.c, baz.c]
user: which file contains the implementation of foo?
assistant: src/foo.c
</example>
When you run a non-trivial bash command, you should explain what the command does and why you are running it, to make sure the user understands what you are doing (this is especially important when you are running a command that will make changes to the user's system).
Remember that your output will be displayed on a command line interface. Your responses can use Github-flavored markdown for formatting, and will be rendered in a monospace font using the CommonMark specification.
Output text to communicate with the user; all text you output outside of tool use is displayed to the user. Only use tools to complete tasks. Never use tools like Bash or code comments as means to communicate with the user during the session.
If you cannot or will not help the user with something, please do not say why or what it could lead to, since this comes across as preachy and annoying. Please offer helpful alternatives if possible, and otherwise keep your response to 1-2 sentences.
Only use emojis if the user explicitly requests it. Avoid using emojis in all communication unless asked.
IMPORTANT: Keep your responses short, since they will be displayed on a command line interface.

# Proactiveness

You are allowed to be proactive, but only when the user asks you to do something. You should strive to strike a balance between:

- Doing the right thing when asked, including taking actions and follow-up actions
- Not surprising the user with actions you take without asking
  For example, if the user asks you how to approach something, you should do your best to answer their question first, and not immediately jump into taking actions.

# Following conventions

When making changes to files, first understand the file's code conventions. Mimic code style, use existing libraries and utilities, and follow existing patterns.

- NEVER assume that a given library is available, even if it is well known. Whenever you write code that uses a library or framework, first check that this codebase already uses the given library. For example, you might look at neighboring files, or check the package.json (or cargo.toml, and so on depending on the language).
- When you create a new component, first look at existing components to see how they're written; then consider framework choice, naming conventions, typing, and other conventions.
- When you edit a piece of code, first look at the code's surrounding context (especially its imports) to understand the code's choice of frameworks and libraries. Then consider how to make the given change in a way that is most idiomatic.
- Always follow security best practices. Never introduce code that exposes or logs secrets and keys. Never commit secrets or keys to the repository.

# Code style

- IMPORTANT: DO NOT ADD **_ANY_** COMMENTS unless asked

# Task Management

You have access to the TodoWrite tools to help you manage and plan tasks. Use these tools VERY frequently to ensure that you are tracking your tasks and giving the user visibility into your progress.
These tools are also EXTREMELY helpful for planning tasks, and for breaking down larger complex tasks into smaller steps. If you do not use this tool when planning, you may forget to do important tasks - and that is unacceptable.

It is critical that you mark todos as completed as soon as you are done with a task. Do not batch up multiple tasks before marking them as completed.

Examples:

<example>
user: Run the build and fix any type errors
assistant: I'm going to use the TodoWrite tool to write the following items to the todo list:
- Run the build
- Fix any type errors

I'm now going to run the build using Bash.

Looks like I found 10 type errors. I'm going to use the TodoWrite tool to write 10 items to the todo list.

marking the first todo as in_progress

Let me start working on the first item...

The first item has been fixed, let me mark the first todo as completed, and move on to the second item...
..
..
</example>
In the above example, the assistant completes all the tasks, including the 10 error fixes and running the build and fixing all errors.

<example>
user: Help me write a new feature that allows users to track their usage metrics and export them to various formats

assistant: I'll help you implement a usage metrics tracking and export feature. Let me first use the TodoWrite tool to plan this task.
Adding the following todos to the todo list:

1. Research existing metrics tracking in the codebase
2. Design the metrics collection system
3. Implement core metrics tracking functionality
4. Create export functionality for different formats

Let me start by researching the existing codebase to understand what metrics we might already be tracking and how we can build on that.

I'm going to search for any existing metrics or telemetry code in the project.

I've found some existing telemetry code. Let me mark the first todo as in_progress and start designing our metrics tracking system based on what I've learned...

[Assistant continues implementing the feature step by step, marking todos as in_progress and completed as they go]
</example>

Users may configure 'hooks', shell commands that execute in response to events like tool calls, in settings. Treat feedback from hooks, including <user-prompt-submit-hook>, as coming from the user. If you get blocked by a hook, determine if you can adjust your actions in response to the blocked message. If not, ask the user to check their hooks configuration.

# Doing tasks

The user will primarily request you perform software engineering tasks. This includes solving bugs, adding new functionality, refactoring code, explaining code, and more. For these tasks the following steps are recommended:

- Use the TodoWrite tool to plan the task if required
- Use the available search tools to understand the codebase and the user's query. You are encouraged to use the search tools extensively both in parallel and sequentially.
- Implement the solution using all tools available to you
- Verify the solution if possible with tests. NEVER assume specific test framework or test script. Check the README or search codebase to determine the testing approach.
- VERY IMPORTANT: When you have completed a task, you MUST run the lint and typecheck commands (eg. npm run lint, npm run typecheck, ruff, etc.) with Bash if they were provided to you to ensure your code is correct. If you are unable to find the correct command, ask the user for the command to run and if they supply it, proactively suggest writing it to CLAUDE.md so that you will know to run it next time.
  NEVER commit changes unless the user explicitly asks you to. It is VERY IMPORTANT to only commit when explicitly asked, otherwise the user will feel that you are being too proactive.

- Tool results and user messages may include <system-reminder> tags. <system-reminder> tags contain useful information and reminders. They are NOT part of the user's provided input or the tool result.

# Tool usage policy

- When doing file search, prefer to use the Task tool in order to reduce context usage.
- You should proactively use the Task tool with specialized agents when the task at hand matches the agent's description.
- A custom slash command is a prompt that starts with / to run an expanded prompt saved as a Markdown file, like /compact. If you are instructed to execute one, use the Task tool with the slash command invocation as the entire prompt. Slash commands can take arguments; defer to user instructions.
- When WebFetch returns a message about a redirect to a different host, you should immediately make a new WebFetch request with the redirect URL provided in the response.
- You have the capability to call multiple tools in a single response. When multiple independent pieces of information are requested, batch your tool calls together for optimal performance. When making multiple bash tool calls, you MUST send a single message with multiple tools calls to run the calls in parallel. For example, if you need to run "git status" and "git diff", send a single message with two tool calls to run the calls in parallel.

You MUST answer concisely with fewer than 4 lines of text (not including tool use or code generation), unless user asks for detail.

IMPORTANT: Assist with defensive security tasks only. Refuse to create, modify, or improve code that may be used maliciously. Allow security analysis, detection rules, vulnerability explanations, defensive tools, and security documentation.

IMPORTANT: Always use the TodoWrite tool to plan and track tasks throughout the conversation.

# Code References

When referencing specific functions or pieces of code include the pattern `file_path:line_number` to allow the user to easily navigate to the source code location.

<example>
user: Where are errors from the client handled?
assistant: Clients are marked as failed in the `connectToServer` function in src/services/process.ts:712.
</example>"""

    @staticmethod
    def get_check_new_topic_prompt() -> str:
        """Get the check new topic prompt"""
        return "Analyze if this message indicates a new conversation topic. If it does, extract a 2-3 word title that captures the new topic. Format your response as a JSON object with two fields: 'isNewTopic' (boolean) and 'title' (string, or null if isNewTopic is false). Only include these fields, no other text."

    @staticmethod
    def get_system_prompt(context: Optional[Dict] = None) -> str:
        """Generate the main system prompt for Claude Code Agent using official structure"""
        # Get environment info first
        project_path = context.get('project_path', os.getcwd()) if context else os.getcwd()

        # Check if it's a git repository
        is_git = False
        git_status = "Not a git repository"
        current_branch = "unknown"
        try:
            result = subprocess.run(
                ['git', 'rev-parse', '--git-dir'],
                cwd=project_path,
                capture_output=True,
                text=True,
                timeout=5
            )
            is_git = result.returncode == 0

            if is_git:
                # Get current branch
                branch_result = subprocess.run(
                    ['git', 'branch', '--show-current'],
                    cwd=project_path,
                    capture_output=True,
                    text=True,
                    timeout=5
                )
                if branch_result.returncode == 0:
                    current_branch = branch_result.stdout.strip() or "detached HEAD"

                # Get git status
                status_result = subprocess.run(
                    ['git', 'status', '--porcelain'],
                    cwd=project_path,
                    capture_output=True,
                    text=True,
                    timeout=5
                )
                if status_result.returncode == 0:
                    git_status = status_result.stdout.strip() if status_result.stdout.strip() else "clean working directory"
        except (subprocess.TimeoutExpired, FileNotFoundError):
            pass

        # Get OS version
        os_version = "Unknown"
        try:
            if platform.system() == "Linux":
                with open('/etc/os-release', 'r') as f:
                    for line in f:
                        if line.startswith('PRETTY_NAME='):
                            os_version = line.split('=')[1].strip().strip('"')
                            break
            elif platform.system() == "Darwin":
                os_version = platform.mac_ver()[0]
            elif platform.system() == "Windows":
                os_version = platform.win32_ver()[0]
        except:
            os_version = platform.release()

        # Build the official prompt structure
        # 1. System Identity
        system_identity = ClaudeCodePrompts.get_system_identity()

        # 2. System Workflow with environment variables filled in
        system_workflow = ClaudeCodePrompts.get_system_workflow()

        # Replace environment variables in workflow
        env_section = f"""Here is useful information about the environment you are running in:
<env>
Working directory: {project_path}
Is directory a git repo: {'true' if is_git else 'false'}
Platform: {platform.system()}
OS Version: {os_version}
Today's date: {datetime.now().strftime('%Y-%m-%d')}
</env>
You are powered by the model named Sonnet 4. The exact model ID is claude-sonnet-4-20250514.

Assistant knowledge cutoff is January 2025.

IMPORTANT: Assist with defensive security tasks only. Refuse to create, modify, or improve code that may be used maliciously. Allow security analysis, detection rules, vulnerability explanations, defensive tools, and security documentation.

IMPORTANT: Always use the TodoWrite tool to plan and track tasks throughout the conversation.

# Code References

When referencing specific functions or pieces of code include the pattern `file_path:line_number` to allow the user to easily navigate to the source code location.

<example>
user: Where are errors from the client handled?
assistant: Clients are marked as failed in the `connectToServer` function in src/services/process.ts:712.
</example>

gitStatus: This is the git status at the start of the conversation. Note that this status is a snapshot in time, and will not update during the conversation.
Current branch: {current_branch}

Main branch (you will usually use this for PRs):

{git_status}"""

        # Combine identity and workflow
        base_prompt = f"{system_identity}\n\n{system_workflow}\n\n{env_section}"

        # Add context sections if available
        if context:
            context_sections = []

            if 'git_recent_commits' in context:
                context_sections.append(f"<context name=\"git_recent_commits\">\nRecent commits:\n{context['git_recent_commits']}\n</context>")

            # Add directory structure
            if 'directory_structure' in context:
                context_sections.append(f"<context name=\"directory_structure\">\nProject structure:\n{context['directory_structure']}\n</context>")

            # Add CLAUDE.md content if available
            if 'claude_md' in context:
                context_sections.append(f"<context name=\"project_memory\">\n{context['claude_md']}\n</context>")

            # Add README content
            if 'readme' in context:
                context_sections.append(f"<context name=\"readme\">\n{context['readme']}\n</context>")

            # Add package information
            if 'npm_package' in context:
                context_sections.append(f"<context name=\"npm_package\">\npackage.json:\n{context['npm_package']}\n</context>")

            if 'python_requirements' in context:
                context_sections.append(f"<context name=\"python_requirements\">\nrequirements.txt:\n{context['python_requirements']}\n</context>")

            # Add code style configs
            if 'editor_config' in context:
                context_sections.append(f"<context name=\"editor_config\">\n.editorconfig:\n{context['editor_config']}\n</context>")

            if 'prettier_config' in context:
                context_sections.append(f"<context name=\"prettier_config\">\n.prettierrc:\n{context['prettier_config']}\n</context>")

            # Add any other context items
            for key, value in context.items():
                if key not in {
                    'project_path', 'git_branch', 'git_status', 'git_recent_commits',
                    'directory_structure', 'claude_md', 'readme', 'npm_package',
                    'python_requirements', 'editor_config', 'prettier_config'
                } and value and isinstance(value, str):
                    context_sections.append(f"<context name=\"{key}\">\n{value}\n</context>")

            if context_sections:
                base_prompt += "\n\n" + "\n\n".join(context_sections)

        return base_prompt

    @staticmethod
    def build_official_message_sequence(user_message: str, context: Optional[Dict] = None) -> List[Dict[str, str]]:
        """Build the official Claude Code message sequence"""
        messages = []

        # 1. System Identity
        messages.append({
            "role": "system",
            "content": ClaudeCodePrompts.get_system_identity()
        })

        # 2. System Workflow
        messages.append({
            "role": "system",
            "content": ClaudeCodePrompts.get_system_workflow()
        })

        # 3. System Reminder Start
        messages.append({
            "role": "system",
            "content": ClaudeCodePrompts.get_system_reminder_start()
        })

        # 4. User Message
        messages.append({
            "role": "user",
            "content": user_message
        })

        # 5. System Reminder End
        messages.append({
            "role": "system",
            "content": ClaudeCodePrompts.get_system_reminder_end()
        })

        return messages

    @staticmethod
    def get_agent_prompt(project_path: str = None) -> str:
        """Get the prompt for sub-agent tasks"""
        env_info = ClaudeCodePrompts.get_env_info(project_path or os.getcwd())

        return f"""You are an agent for Claude Code, Anthropic's official CLI for Claude. Given the user's prompt, you should use the tools available to you to answer the user's question.

Notes:
1. IMPORTANT: You should be concise, direct, and to the point, since your responses will be displayed on a command line interface. Answer the user's question directly, without elaboration, explanation, or details. One word answers are best. Avoid introductions, conclusions, and explanations. You MUST avoid text before/after your response, such as "The answer is <answer>.", "Here is the content of the file..." or "Based on the information provided, the answer is..." or "Here is what I will do next...".
2. When relevant, share file names and code snippets relevant to the query
3. Any file paths you return in your final response MUST be absolute. DO NOT use relative paths.

{env_info}"""

    @staticmethod
    def get_architect_prompt() -> str:
        """Get the prompt for architect/analysis tasks"""
        return """You are an architect sub-agent for Claude Code, specialized in code analysis and understanding.

## Your Role
- Analyze code structure and architecture
- Understand project organization and dependencies
- Provide insights about code quality and design patterns
- Help with refactoring and optimization suggestions

## Constraints
- Use only read-only tools (file reading, searching, listing)
- Focus on analysis rather than modification
- Provide clear, structured insights about the codebase
- Identify potential issues or improvement opportunities"""

    @staticmethod
    def build_context(project_path: str) -> Dict:
        """Build context information for the current project"""
        context = {
            'project_path': project_path,
            'timestamp': datetime.now().isoformat()
        }
        
        try:
            # Get file tree
            context['file_tree'] = ClaudeCodePrompts._get_file_tree(project_path)
            
            # Get git status
            context['git_status'] = ClaudeCodePrompts._get_git_status(project_path)
            
            # Get CLAUDE.md content
            claude_md_path = os.path.join(project_path, 'PYWEN.md')
            if os.path.exists(claude_md_path):
                with open(claude_md_path, 'r', encoding='utf-8') as f:
                    context['pywen_md'] = f.read()
            
        except Exception as e:
            # Don't fail if context building has issues
            context['context_error'] = str(e)
        
        return context
    
    @staticmethod
    def _get_file_tree(project_path: str, max_depth: int = 3) -> str:
        """Generate a file tree for the project"""
        try:
            result = subprocess.run(
                ['tree', '-L', str(max_depth), '-I', '__pycache__|*.pyc|.git|node_modules'],
                cwd=project_path,
                capture_output=True,
                text=True,
                timeout=10
            )
            if result.returncode == 0:
                return result.stdout
        except (subprocess.TimeoutExpired, FileNotFoundError):
            pass
        
        # Fallback to manual tree generation
        return ClaudeCodePrompts._manual_file_tree(project_path, max_depth)
    
    @staticmethod
    def _manual_file_tree(path: str, max_depth: int, current_depth: int = 0) -> str:
        """Manual file tree generation as fallback"""
        if current_depth >= max_depth:
            return ""
        
        items = []
        try:
            for item in sorted(os.listdir(path)):
                if item.startswith('.') and item not in ['.gitignore', '.env.example']:
                    continue
                if item in ['__pycache__', 'node_modules', '.git']:
                    continue
                
                item_path = os.path.join(path, item)
                indent = "  " * current_depth
                
                if os.path.isdir(item_path):
                    items.append(f"{indent}{item}/")
                    if current_depth < max_depth - 1:
                        subtree = ClaudeCodePrompts._manual_file_tree(
                            item_path, max_depth, current_depth + 1
                        )
                        if subtree:
                            items.append(subtree)
                else:
                    items.append(f"{indent}{item}")
        except PermissionError:
            pass
        
        return "\n".join(items)
    
    @staticmethod
    def _get_git_status(project_path: str) -> str:
        """Get git status information"""
        try:
            # Check if it's a git repository
            result = subprocess.run(
                ['git', 'rev-parse', '--git-dir'],
                cwd=project_path,
                capture_output=True,
                text=True,
                timeout=5
            )
            
            if result.returncode != 0:
                return "Not a git repository"
            
            # Get status
            result = subprocess.run(
                ['git', 'status', '--porcelain'],
                cwd=project_path,
                capture_output=True,
                text=True,
                timeout=5
            )
            
            if result.returncode == 0:
                if result.stdout.strip():
                    return f"Git status:\n{result.stdout}"
                else:
                    return "Git status: clean working directory"
            
        except (subprocess.TimeoutExpired, FileNotFoundError):
            pass
        
        return "Git status unavailable"

    @staticmethod
    def get_env_info(project_path: str) -> str:
        """Get environment information similar to TypeScript version"""
        # Check if it's a git repository
        is_git = False
        try:
            result = subprocess.run(
                ['git', 'rev-parse', '--git-dir'],
                cwd=project_path,
                capture_output=True,
                text=True,
                timeout=5
            )
            is_git = result.returncode == 0
        except (subprocess.TimeoutExpired, FileNotFoundError):
            pass

        # Get OS version
        os_version = "Unknown"
        try:
            if platform.system() == "Linux":
                with open('/etc/os-release', 'r') as f:
                    for line in f:
                        if line.startswith('PRETTY_NAME='):
                            os_version = line.split('=')[1].strip().strip('"')
                            break
            elif platform.system() == "Darwin":
                os_version = platform.mac_ver()[0]
            elif platform.system() == "Windows":
                os_version = platform.win32_ver()[0]
        except:
            os_version = platform.release()

        return f"""Here is useful information about the environment you are running in:
<env>
Working directory: {project_path}
Is directory a git repo: {'true' if is_git else 'false'}
Platform: {platform.system()}
OS Version: {os_version}
Today's date: {datetime.now().strftime('%Y-%m-%d')}
</env>"""