# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

from __future__ import annotations

from dataclasses import dataclass
from typing import Union

from .._errors import ExpressionError, TokenError
from .._tokenstream import Token, TokenStream, TokenType

__all__ = ["Node", "ProductNode", "AssociationNode", "IdentifierNode", "Parser"]

# ===================================
# -- Expression Tree Nodes
# ===================================


class Node:
    """Base class for all expression tree nodes generated by the
    parser for task parameter combination expressions.
    """

    def collect_identifiers(self, destination: list[str]) -> None:
        if isinstance(self, (ProductNode, AssociationNode)):
            for node in self.children:
                node.collect_identifiers(destination)
        elif isinstance(self, IdentifierNode):
            destination.append(self.parameter)


@dataclass
class ProductNode(Node):
    """Outer product operator on a list of non-product nodes.

    e.g. A * B * C
    """

    children: tuple[Node, ...]

    def __str__(self) -> str:
        return " * ".join(str(child) for child in self.children)

    def __repr__(self) -> str:
        return f"Product({', '.join(repr(child) for child in self.children)})"


@dataclass
class AssociationNode(Node):
    """Association operator on a list of nodes.

    e.g. (A, B, C)
    """

    children: tuple[Node, ...]

    def __str__(self) -> str:
        return f"({', '.join(str(child) for child in self.children)})"

    def __repr__(self) -> str:
        return f"Assoc({', '.join(repr(child) for child in self.children)})"


@dataclass
class IdentifierNode(Node):
    """Leaf node representing a task parameter identifier."""

    parameter: str

    def __str__(self) -> str:
        return self.parameter

    def __repr__(self) -> str:
        return self.parameter


# =========================================
# -- Token types for the lexical analysis
# =========================================


class IdentToken(Token):
    """An identifier matching the regex: ^[^\\d\\W][\\w]*$
    [^\\d\\W] = _ or unicode non-digit letter
    [\\w] = _ or unicode letter including digits
    """

    pass


class StarToken(Token):
    """The '*' character."""

    pass


class CommaToken(Token):
    """The ',' character."""

    pass


class LeftParenToken(Token):
    """The '(' character"""

    pass


class RightParenToken(Token):
    """The ')' character"""

    pass


# Map of TokenTypes to their corresponding Token class.
# Required by the TokenStream used by the parser to map
# lexical tokens to the correct token class.
_tokenmap = {
    TokenType.NAME: IdentToken,
    TokenType.STAR: StarToken,
    TokenType.LPAREN: LeftParenToken,
    TokenType.RPAREN: RightParenToken,
    TokenType.COMMA: CommaToken,
}


class Parser:
    """Task parameter combination expression parser.

    Full Grammar:
        <expr> ::= <element> ( '*' <element> )*
        <element> ::= <identifier> | <association_expr>
        <association_expr> ::= '(' <expr> ( ',' <expr> )* ')'
    """

    def parse(self, expr: str) -> Node:
        """Generate an expression tree for the given string interpolation expression.

        Raises:
            TokenError: If an unexpected token is encountered.
            ExpressionError: If the expresssion is malformed.
        """

        # Raises: TokenError
        self._tokens = TokenStream(expr, supported_tokens=_tokenmap)

        if self._tokens.at_end():
            raise ExpressionError("Empty expression")

        result = self._expression()
        if not self._tokens.at_end():
            token = self._tokens.next()
            raise TokenError(self._tokens.expr, token.value, token.start)

        return result

    def _expression(self) -> Node:
        """Matches an expression statement.

        Grammar:
            <expr> ::= <element> ( '*' <element> )*

        Raises:
            TokenError: If an unexpected token is encountered
            ExpressionError: If the expresssion is malformed.

        Returns:
            Node: Expression tree node for the recursively matched expression.
        """

        children: list[Node] = []
        # Parse: <element>
        # Raises: TokenError, ExpressionError
        children.append(self._element())
        try:
            # Parse: ( '*' <element> )*
            while isinstance(self._tokens.lookahead(0), StarToken):
                # consume the '*'
                self._tokens.next()
                # Parse: <element>
                # Raises: TokenError, ExpressionError
                children.append(self._element())
        except IndexError:
            # Catches the lookahead on the while condition. Not having anything after a name is okay
            pass

        if len(children) == 1:
            return children[0]
        return ProductNode(tuple(children))

    def _association_expression(self) -> AssociationNode:
        """Matches an association statement in the expression.

        Grammar:
            <association_expr> ::= '(' <expr> ( ',' <expr> )* ')'

        Raises:
            TokenError: If an unexpected token is encountered
            ExpressionError: If the expresssion is malformed.

        Returns:
            AssociationNode: Expression tree node for the recursively matched
                association expression.
        """

        children: list[Node] = []
        # We only get into this function by doing a token lookahead and
        # seeing a left-paren. Consume the left paren.
        self._tokens.next()

        # Parse: <expr>
        expr = self._expression()
        children.append(expr)
        # Parse: ( ',' <expr> )+
        try:
            while isinstance(self._tokens.lookahead(0), CommaToken):
                # Consume the ','
                self._tokens.next()
                # Raises: TokenError, ExpressionError
                expr = self._expression()
                children.append(expr)
        except IndexError:
            # Catches the lookahead on the while condition. Not having a dot after the name is okay.
            pass
        # Parse: ')'
        try:
            token = self._tokens.next()
            if not isinstance(token, RightParenToken):
                raise TokenError(self._tokens.expr, token.value, token.start)
        except IndexError:
            raise ExpressionError(f"Unexpected end of expression: {self._tokens.expr}")

        if len(children) == 1:
            raise ExpressionError(
                f"Association expression '( {str(children[0])} )' must have more than one term."
            )

        return AssociationNode(tuple(children))

    def _element(self) -> Union[IdentifierNode, AssociationNode]:
        """Matches an element statement from the grammar.

        Raises:
            TokenError: If an unexpected token is encountered
            ExpressionError: If the expresssion is malformed.

        Grammar:
            <element> ::= <identifier> | <association_expr>

        Returns:
            Union[IdentifierNode, AssociationNode]: Expression tree node for the
                recursively matched element.
        """

        try:
            # Raises: IndexError
            lookahead = self._tokens.lookahead(0)
        except IndexError:
            raise ExpressionError(f"Unexpected end of expression: {self._tokens.expr}")
        if isinstance(lookahead, IdentToken):
            token = self._tokens.next()
            return IdentifierNode(token.value)
        elif isinstance(lookahead, LeftParenToken):
            # Raises: TokenError, ExpressionError
            return self._association_expression()

        raise TokenError(self._tokens.expr, lookahead.value, lookahead.start)
