# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

from __future__ import annotations

from abc import ABC, abstractmethod
from collections.abc import Iterable, Iterator, Sized
from dataclasses import dataclass, field
from decimal import Decimal
from functools import reduce
from operator import mul
import re
from typing import AbstractSet, Any, Optional, Union

from ._internal import (
    CombinationExpressionAssociationNode,
    CombinationExpressionIdentifierNode,
    CombinationExpressionNode,
    CombinationExpressionParser,
    CombinationExpressionProductNode,
)
from ._range_expr import IntRangeExpr
from ._types import ParameterValue, ParameterValueType, StepParameterSpace, TaskParameterSet
from .v2023_09 import (
    RangeExpressionTaskParameterDefinition as RangeExpressionTaskParameterDefinition_2023_09,
    RangeListTaskParameterDefinition as RangeListTaskParameterDefinition_2023_09,
    TaskParameterType as TaskParameterType_2023_09,
    TaskChunksRangeConstraint as TaskChunksRangeConstraint_2023_09,
)

__all__ = ["StepParameterSpaceIterator"]


RangeTaskParameter = Union[
    RangeListTaskParameterDefinition_2023_09, RangeExpressionTaskParameterDefinition_2023_09
]
TaskParameter = RangeTaskParameter


class StepParameterSpaceIterator(Iterable[TaskParameterSet], Sized):
    """An iterator for the multidimensional space of all possible task parameter values,
    returning an individual task or a chunk of tasks in sequence.

    Note: The ordering of inputs generated by this iterator is **NOT**
    guaranteed to be invariant across versions of this library.

    In most cases, the iteration order is dictated by the order of each parameter's values, and
    the order of the expressions within the combination expression of the StepParameterSpace.
    Ordering is row-major (right-most moves fastest) for products ('*') in the expression.

    An exception to this rule occurs with adaptive chunked iteration. The iteration order is different
    when the parameter space includes a CHUNK[INT] parameter, and its `chunks.targetRuntimeSeconds`
    value is non-zero. No matter where the CHUNK[INT] parameter is place in the combination expression,
    it becomes the innermost iteration dimension in order to keep the implementation simple and
    memory-efficient.

    In the adaptive chunked iteration case, the iterator's `chunks_default_task_count` value is mutable
    so that the code using it can collect task runtime statistics and adapt the size of the chunk to more
    closely match the target runtime.

    For example, given:
        A = [1,2,3]
        B = [1,2]
        C = [10,11]
    Then:
        combination = "A * (B,C)"
        results in the order:
        (A=1,B=1,C=10), (A=1,B=2,C=11), (A=2,B=1,C=10), (A=2,B=2,C=11), (A=3,B=1,C=10), (A=3,B=2,C=11)
    Or
        combination = "(B,C) * A"
        results in the order:
        (A=1,B=1,C=10), (A=2,B=1,C=10), (A=3,B=1,C=10), (A=1,B=2,C=11), (A=2,B=2,C=11), (A=3,B=2,C=11)

    For comparison:
        A = [3,2,1]
        B = [1,2]
        C = [10,11]
        combination = "A * (B,C)"
    would be the order:
        (A=3,B=1,C=10), (A=3,B=2,C=11), (A=2,B=1,C=10), (A=2,B=2,C=11), (A=1,B=1,C=10), (A=1,B=2,C=11)

    Args:
        space (Optional[StepParameterSpace]): The parameter space to iterate over.
        chunks_task_count_override (Optional[int]): If provided, turns off adaptive chunking
            and overrides the default task count from the parameter space with the provided value.
            For example, to iterate over all the individual tasks, set this value to 1.

    Attributes:
        names (AbstractSet[str]): The set of all task parameter names.
        chunks_adaptive (bool): If True, the iterator is configured for adaptive chunking. This means
            the caller can modify the chunks_default_task_count property when it determines a different
            chunk size is appropriate to better match the targetRuntimeSeconds of the CHUNK parameter.
        chunks_default_task_count (Optional[int]): If chunking is disabled, is None. If chunking is
            enabled, is the default chunk size to use for grouping tasks into chunks. If adaptive
            chunking is enabled, the caller can modify this value and subsequent iteration will reflect
            the new chunk size.
    """

    _parameters: dict[str, TaskParameter]
    _expr_tree: Node
    _iter: NodeIterator
    _parsedtree: CombinationExpressionNode

    _chunks_adaptive: bool
    _chunks_parameter_name: Optional[str]
    _chunks_default_task_count: Optional[int]

    def __init__(
        self,
        *,
        space: Optional[StepParameterSpace],
        chunks_task_count_override: Optional[int] = None,
    ):
        self._chunks_adaptive = False
        self._chunks_parameter_name = None
        self._chunks_default_task_count = None

        # Special case the zero-dimensional space with one element
        if space is None:
            self._parameters = {}
            self._parsetree = None
            self._expr_tree = ZeroDimSpaceNode()
            self._iter = self._expr_tree.iter()
        else:
            if space.combination is None:
                # space.taskParameterDefinitions is a dict[str,TaskParameter]
                combination = "*".join(name for name in space.taskParameterDefinitions)
            else:
                combination = space.combination
            self._parameters = dict(space.taskParameterDefinitions)

            # Determine whether and what kind of chunking to do
            for name, param in self._parameters.items():
                if param.type == TaskParameterType_2023_09.CHUNK_INT:
                    if param.chunks is None:
                        raise ValueError(
                            f"CHUNK[INT] parameter '{name}' must have a chunks definition."
                        )
                    self._chunks_adaptive = (
                        param.chunks.targetRuntimeSeconds is not None
                        and int(param.chunks.targetRuntimeSeconds) > 0
                    )
                    self._chunks_parameter_name = name
                    self._chunks_default_task_count = int(param.chunks.defaultTaskCount)
                    break

            # If the chunks task count is overridden, and the parameter space uses chunking
            if (
                chunks_task_count_override is not None
                and self._chunks_default_task_count is not None
            ):
                self._chunks_adaptive = False
                self._chunks_default_task_count = chunks_task_count_override

            # Raises: TokenError, ExpressionError
            self._parsetree = CombinationExpressionParser().parse(combination)
            self._expr_tree = self._create_expr_tree(
                self._parsetree,
                chunks_adaptive=self._chunks_adaptive,
                chunks_default_task_count=self._chunks_default_task_count,
            )
            self._iter = self._expr_tree.iter()

    @property
    def names(self) -> AbstractSet[str]:
        """Get the names of all parameters in the parameter space."""
        return self._parameters.keys()

    @property
    def chunks_adaptive(self) -> bool:
        """True if the parameter space includes a CHUNK[INT] parameter with a non-zero target runtime."""
        return self._chunks_adaptive

    @property
    def chunks_parameter_name(self) -> Optional[str]:
        """The name of the CHUNK[INT] parameter, if any."""
        return self._chunks_parameter_name

    @property
    def chunks_default_task_count(self) -> Optional[int]:
        """The default task count for the CHUNK[INT] parameter, if any.

        Returns:
            Optional[int]: The default task count, or None if the parameter space does not include a CHUNK[INT] parameter.
        """
        return self._chunks_default_task_count

    @chunks_default_task_count.setter
    def chunks_default_task_count(self, value: int):
        """Set the default task count for the CHUNK[INT] parameter.

        Args:
            value (int): The new default task count.

        Raises:
            ValueError: If the parameter space does not use adaptive chunking, i.e. a CHUNK[INT] parameter with a target runtime.
        """
        if not self._chunks_adaptive:
            raise ValueError(
                "The parameter space does not use adaptive chunking, so cannot modify chunks_default_task_count."
            )
        if not (isinstance(value, int) and value > 0):
            raise ValueError("chunks_default_task_count must be a positive integer.")

        # The expr tree will raise if the parameter space does not use adaptive chunking
        self._expr_tree.set_chunks_default_task_count(value)
        self._chunks_default_task_count = value

    def reset_iter(self) -> None:
        self._iter.reset_iter()

    def __iter__(self) -> StepParameterSpaceIterator:
        return self

    def __next__(self) -> TaskParameterSet:
        result: TaskParameterSet = TaskParameterSet()
        self._iter.next(result)
        return result

    def __len__(self) -> int:
        """The number of task parameter sets that are defined by this parameter space"""
        return len(self._expr_tree)

    def __getitem__(self, index: int) -> TaskParameterSet:
        """Get a specific task parameter set given an index.
        This cannot be used when adaptive chunking is active.

        Note: The ordering of inputs is **NOT** guaranteed to be
        invariant across versions of this library.

        Args:
            index (int): Index for a task parameter set to fetch.

        Returns:
            dict[str, Union[int, float, str]]: Values of every task parameter. Dictionary key
                is the parameter  name.

        Raises:
            RuntimeError: When getitem is called but adaptive chunking is active.
        """
        return self._expr_tree[index]

    def __contains__(self, params: TaskParameterSet) -> bool:
        """Check if a specific task parameter set is included in this parameter space.

        In the case of chunking, the chunked dimension is treated via a subset operator
        instead of whether it equals one of the values along the chunked dimension."""
        try:
            self.validate_containment(params)
            return True
        except ValueError:
            return False

    def validate_containment(self, params: TaskParameterSet):
        """Check if a specific task parameter set is included in this parameter space,
        raising a ValueError if it is not.

        In the case of chunking, the chunked dimension is treated via a subset operator
        instead of whether it equals one of the values along the chunked dimension."""
        # The task parameter names must match
        params_keys = sorted(params.keys())
        space_keys = sorted(self._parameters.keys())
        if params_keys != space_keys:
            raise ValueError(
                f"Task parameter names {params_keys} do not match the parameter space names {space_keys}."
            )
        # Check containment against the expr tree nodes
        self._expr_tree.validate_containment(params)

    def _create_expr_tree(
        self,
        root: CombinationExpressionNode,
        chunks_adaptive: bool,
        chunks_default_task_count: Optional[int],
    ) -> Node:
        """Recursively make a copy of the given Parser-generated expression tree using
        the Node types defined in this file.
        """
        if isinstance(root, CombinationExpressionIdentifierNode):
            name = root.parameter
            parameter = self._parameters[name]
            if parameter.type == TaskParameterType_2023_09.CHUNK_INT:
                if parameter.chunks is None:
                    raise ValueError(
                        f"CHUNK[INT] parameter '{name}' must have a chunks definition."
                    )
                if chunks_default_task_count is None:
                    raise ValueError(
                        f"CHUNK[INT] parameter '{name}' must have a default task count."
                    )
                # Expand the range to a list
                if isinstance(parameter.range, list):
                    parameter_range: list[int] = [int(v) for v in parameter.range]
                else:
                    parameter_range = list[int](parameter.range)

                if chunks_adaptive:
                    if (
                        parameter.chunks.rangeConstraint
                        == TaskChunksRangeConstraint_2023_09.CONTIGUOUS
                    ):
                        return AdaptiveContiguousChunkIdentifierNode(
                            name=name,
                            type=ParameterValueType(parameter.type),
                            range=parameter_range,
                            chunks_default_task_count=chunks_default_task_count,
                        )
                    else:
                        return AdaptiveNoncontiguousChunkIdentifierNode(
                            name=name,
                            type=ParameterValueType(parameter.type),
                            range=parameter_range,
                            chunks_default_task_count=chunks_default_task_count,
                        )
                else:
                    # Divide the range into contiguous or noncontiguous chunks
                    if (
                        parameter.chunks.rangeConstraint
                        == TaskChunksRangeConstraint_2023_09.CONTIGUOUS
                    ):
                        chunk_list = divide_int_list_into_contiguous_chunks(
                            chunks_default_task_count, parameter_range
                        )
                    else:
                        chunk_list = divide_int_list_into_noncontiguous_chunks(
                            chunks_default_task_count, parameter_range
                        )

                    # With the chunks determined, we can use the range list node for iteration
                    return RangeListIdentifierNode(
                        name=name,
                        type=ParameterValueType(parameter.type),
                        range=chunk_list,  # type: ignore
                        range_set=set(parameter_range),
                        range_constraint=parameter.chunks.rangeConstraint,
                    )
            elif isinstance(parameter.range, list):
                return RangeListIdentifierNode(
                    name=name,
                    type=ParameterValueType(parameter.type),
                    range=parameter.range,
                    range_set=set(parameter.range),
                )
            else:
                return RangeExpressionIdentifierNode(
                    name=name,
                    type=ParameterValueType(parameter.type),
                    range=parameter.range,
                )
        elif isinstance(root, CombinationExpressionAssociationNode):
            return AssociationNode(
                children=tuple(
                    self._create_expr_tree(child, chunks_adaptive, chunks_default_task_count)
                    for child in root.children
                ),
            )
        else:
            assert isinstance(root, CombinationExpressionProductNode)
            # For adaptive chunking, shift the chunked parameter to the innermost dimension
            if chunks_adaptive:
                children: Any = []
                chunked_child = None
                for child in root.children:
                    if (
                        isinstance(child, CombinationExpressionIdentifierNode)
                        and self._parameters[child.parameter].type
                        == TaskParameterType_2023_09.CHUNK_INT
                    ):
                        chunked_child = child
                    else:
                        children.append(child)
                if chunked_child is not None:
                    children.append(chunked_child)
            else:
                children = root.children

            return ProductNode(
                children=tuple(
                    self._create_expr_tree(child, chunks_adaptive, chunks_default_task_count)
                    for child in children
                ),
            )

    def __eq__(self, other: object) -> bool:
        # For assisting unit testing
        if not isinstance(other, StepParameterSpaceIterator):  # pragma: no cover
            return NotImplemented
        return self._parameters == other._parameters and repr(self._parsetree) == repr(
            other._parsetree
        )


def divide_chunk_sizes(chunks_default_task_count: int, task_count: int) -> list[int]:
    """Determines evenly distributed chunk lengths for a specified task count divided
    into the requested number of chunks."""
    if chunks_default_task_count <= 0:
        chunk_count = 1
    else:
        chunk_count = (task_count + chunks_default_task_count - 1) // chunks_default_task_count

    if chunk_count >= task_count:
        return [1] * task_count
    elif chunk_count <= 1:
        return [task_count]

    # Each chunk will be one of two sizes, small_chunk_size or small_chunk_size + 1
    small_chunk_size, leftovers = divmod(task_count, chunk_count)
    # Create a list of chunk sizes starting from smaller_chunk_size
    chunk_sizes = [small_chunk_size] * chunk_count
    # Distribute the leftovers evenly across the chunks
    for i in range(leftovers):
        chunk_sizes[(i * chunk_count) // leftovers] += 1
    return chunk_sizes


def divide_int_interval_into_chunks(
    chunks_default_task_count: int, first_value: int, last_value: int
) -> list[str]:
    """Divide the provided value interval into CONTIGUOUS chunks, given the default chunk size to use. Every
    chunk is in the form of "<first_value>-<last_value>" even if the first and last values are equal.
    """
    task_count = last_value - first_value + 1
    chunk_sizes = divide_chunk_sizes(chunks_default_task_count, task_count)
    chunks = []
    for chunk_size in chunk_sizes:
        chunks.append(f"{first_value}-{first_value + chunk_size - 1}")
        first_value += chunk_size
    return chunks


def divide_int_list_into_contiguous_chunks(
    chunks_default_task_count: int, values: list[int]
) -> list[str]:
    """Divide the provided list of values into CONTIGUOUS chunks, given the default chunk size to use. Every
    chunk is in the form of "<first_value>-<last_value>" even if the first and last values are equal.
    """
    if len(values) == 0:
        return []

    interval_starts = [0]
    interval_starts.extend(
        i + 1 for i, (v0, v1) in enumerate(zip(values, values[1:])) if v1 != v0 + 1
    )
    interval_starts.append(len(values))
    chunks = []
    for interval_index in range(len(interval_starts) - 1):
        index_start, index_end = (
            interval_starts[interval_index],
            interval_starts[interval_index + 1] - 1,
        )
        value_start, value_end = (values[index_start], values[index_end])
        chunks.extend(
            divide_int_interval_into_chunks(chunks_default_task_count, value_start, value_end)
        )
    return chunks


def divide_int_list_into_noncontiguous_chunks(
    chunks_default_task_count: int, values: list[int]
) -> list[str]:
    """Divide the provided list of values into NONCONTIGUOUS chunks, given the default chunk size to use."""
    chunk_sizes = divide_chunk_sizes(chunks_default_task_count, len(values))
    chunks = []
    index_start = 0
    for chunk_size in chunk_sizes:
        chunks.append(str(IntRangeExpr.from_list(values[index_start : index_start + chunk_size])))
        index_start += chunk_size
    return chunks


# -------
# Mirror classes for the parsed combination expression tree.
# We create these to separate out the functionality required by this class
# around iterating, length calculations, getitem, etc from the needs of
# an expression validator. To validate the expression it is sufficient to
# parse the expression and then collect all referenced identifiers into a list.


class NodeIterator(ABC):
    """Our own special iterator base class. We don't use the standard
    Python iterator interface. For efficiency we need to pass a
    dict into the next() operator and have the method add/modify the
    appropriate key/values in it. With the standard iterator we would
    have __next__ returning a dict and end up with a tonne of intermediary
    dicts created as we ran __next__ on the entire tree.

    When iterating with adaptive chunking, the `chunks_default_task_count`
    is not None, and can be modified. It's the responsibility of the caller
    to perform any statistics and heuristics to determine updates to the chunk
    size.
    """

    @abstractmethod
    def next(self, result: TaskParameterSet) -> None:
        raise NotImplementedError("Base class")  # pragma: no cover

    @abstractmethod
    def reset_iter(self) -> None:
        raise NotImplementedError("Base class")  # pragma: no cover


class Node(ABC, Sized):
    @abstractmethod
    def __getitem__(self, index: int) -> TaskParameterSet:
        raise NotImplementedError("Base class")  # pragma: no cover

    @abstractmethod
    def iter(self) -> NodeIterator:
        raise NotImplementedError("Base class")  # pragma: no cover

    def set_chunks_default_task_count(self, value: int) -> None:
        raise ValueError(
            "The parameter space does not use adaptive chunking, so cannot modify chunks_default_task_count."
        )

    def validate_containment(self, params: TaskParameterSet):
        """Validates that the parameter space contains the provided parameters. Raises ValueError if not."""
        raise NotImplementedError("Base class")  # pragma: no cover


class ZeroDimSpaceIter(NodeIterator):
    """Iterator for a zero-dimensional space

    Attributes:
        _first_value: True if and only if we have not yet evaluated the first value of the iterator.
    """

    _first_value: bool

    def __init__(self):
        self._first_value = True

    def reset_iter(self) -> None:
        self._first_value = True

    def next(self, result: TaskParameterSet) -> None:
        if self._first_value:
            self._first_value = False
            result.clear()
        else:
            raise StopIteration


class ZeroDimSpaceNode(Node):
    """A zero-dimensional space"""

    def __init__(self):
        pass

    def __len__(self) -> int:
        return 1

    def __getitem__(self, index: int) -> TaskParameterSet:
        if -1 <= index <= 0:
            return TaskParameterSet()
        else:
            raise IndexError(f"index {index} is out of range")

    def iter(self) -> NodeIterator:
        return ZeroDimSpaceIter()

    def validate_containment(self, params: TaskParameterSet):
        pass


class ProductNodeIter(NodeIterator):
    """Iterator for a ProductNode

    Attributes:
        _children: Iterators for the child nodes of this node in the expression tree.
        _prev_result: Cached result from the previous iteration. See next() for explanation.
        _first_value: True if and only if we have not yet evaluated the first value of the iterator.
    """

    _children: tuple[NodeIterator, ...]
    _prev_result: TaskParameterSet
    _first_value: bool

    def __init__(self, children: tuple[Node, ...]):
        self._children = tuple(child.iter() for child in children)
        self._prev_result = TaskParameterSet()
        self._first_value = True

    def reset_iter(self) -> None:
        self._first_value = True
        for child in self._children:
            child.reset_iter()

    def next(self, result: TaskParameterSet) -> None:
        # We have to start from the previous result because we don't 'next' every
        # iterator when evaluting a ProductNode since not every value changes on each
        # iteration. So we start from the previous values and overwrite the values that
        # change
        if self._first_value:
            self._first_value = False
            for child in self._children:
                # Raises: StopIteration
                child.next(self._prev_result)
        else:
            # The way to think of the next() operation on a product node is
            # a row-major order traversal of a multidimensional space.
            # For example, if we have:
            #   A = [1,2]; B = [3,4]; C = [5,6]; and
            #  expr="A * B * C"
            # then the parameters on the right change more rapidly
            # than the parameters on the left. In this case our traversal
            # order is:
            #    A  |  B  |  C
            #   ---------------
            #    1  |  3  |  5
            #    1  |  3  |  6
            #    1  |  4  |  5
            #    1  |  4  |  6
            #    2  |  3  |  5
            #    2  |  3  |  6
            #    2  |  4  |  5
            #    2  |  4  |  6
            #
            # To achieve, algorithmically, think about grade-school addition by
            # 1 with a carry.
            # We advance the right-most parameter by one. If that overflows (i.e.
            # hits the end of the iterator) then we reset its iterator and advance
            # the next parameter to its left. Repeating as neccessary until we have
            # hit the end of the left-most iterator -- which indicates the end of
            # the product iteration.

            pos = len(self._children) - 1
            while True:
                try:
                    # Raises: StopIteration
                    self._children[pos].next(self._prev_result)
                    break
                except StopIteration as e:
                    if pos > 0:
                        self._children[pos].reset_iter()
                        self._children[pos].next(self._prev_result)
                        pos -= 1
                    else:
                        raise e from None
        result.update(self._prev_result)


@dataclass
class ProductNode(Node):
    children: tuple[Node, ...]
    _len: Optional[int] = field(default=None, init=False, repr=False, compare=False)

    def __len__(self) -> int:
        if self._len is None:
            self._len = reduce(mul, (len(child) for child in self.children), 1)
        return self._len

    def __getitem__(self, index: int) -> TaskParameterSet:
        if index < 0:
            index = len(self) + index
        if not (0 <= index < len(self)):
            raise IndexError()
        result = dict()
        # See ProductNodeIter.next() for a long comment on how we define the ordering
        # of a product node's values.
        # To go from a specific index to the value we use the fact that we're doing
        # a row-major-ordering. Say we have the product "A * B * C" and we're looking
        # for the index of a product-value where the components of that product-value
        # are at indicies (a, b, c) --  i.e. A at index 'a', B at index 'b', and C at
        # index 'c'.
        # Then, the product-value index is simply:
        #   index = (a * len(B) + b)*len(C) + c
        # This generalizes... it's a big nested multiply-add.
        # Our algorithm here just finds (a,b,c) given index -- which we can do
        # because we know index, len(A), len(B), and len(C) -- and then recursively
        # finds the values of A @ index a, B @ index b, and C & index c to compose the
        # value for this product node.
        pos = len(self.children)
        while pos > 0:
            pos -= 1
            if pos > 0:
                child_length = len(self.children[pos])
                child_index = index % child_length
                index = index // child_length
            else:
                child_index = index
            result.update(self.children[pos][child_index])
        return result

    def iter(self) -> ProductNodeIter:
        return ProductNodeIter(self.children)

    def set_chunks_default_task_count(self, value: int) -> None:
        # If chunks_default_task_count is settable, the last child is
        # an AdaptiveChunkIdentifierNode and will accept the value,
        # otherwise it will raise an exception.
        self.children[-1].set_chunks_default_task_count(value)

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        for child in self.children:
            child.validate_containment(params)


class AssociationNodeIter(NodeIterator):
    """Iterator for an AssociationNode

    Attributes:
        _children: Iterators for the child nodes of this node in the expression tree.
    """

    _children: tuple[NodeIterator, ...]

    def __init__(self, children: tuple[Node, ...]):
        self._children = tuple(child.iter() for child in children)

    def reset_iter(self) -> None:
        for child in self._children:
            child.reset_iter()

    def next(self, result: TaskParameterSet) -> None:
        for child in self._children:
            # Raises: StopIteration
            child.next(result)


@dataclass
class AssociationNode(Node):
    children: tuple[Node, ...]
    _len: Optional[int] = field(default=None, init=False, repr=False, compare=False)

    def __len__(self) -> int:
        if self._len is None:
            self._len = len(self.children[0])
        return self._len

    def __getitem__(self, index: int) -> TaskParameterSet:
        result = TaskParameterSet()
        for child in self.children:
            result.update(child[index])
        return result

    def iter(self) -> AssociationNodeIter:
        return AssociationNodeIter(self.children)

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        # To keep this code simple, we perform a linear search over the full subspace.
        # TODO: Make a more efficient implementation.
        it = self.iter()
        # Use the first iteration value to take a subset of the keys from params
        first_value: TaskParameterSet = TaskParameterSet()
        it.next(first_value)

        # Restrict the params to the keys in this associative expression so that
        # we can use equality checking in the linear search.
        params = {key: value for key, value in params.items() if key in first_value}
        if first_value == params:
            return
        # Perform the linear search
        try:
            while True:
                value: TaskParameterSet = TaskParameterSet()
                it.next(value)
                if value == params:
                    return
        except StopIteration:
            pass
        raise ValueError(
            f"The values { {name: param.value for name, param in params.items()} }, of an association expression in the combination expression, do not appear in the parameter space."
        )


class RangeListIdentifierNodeIterator(NodeIterator):
    """Iterator for a RangeListIdentifierNode

    Attributes:
        _it: Iterator for the corresponding task parameter
        _node: The RangeListIdentifierNode this is iterating over.
    """

    _it: Iterator[Union[str, int, float, Decimal]]
    _node: RangeListIdentifierNode

    def __init__(self, node: RangeListIdentifierNode):
        self._node = node
        self.reset_iter()

    def reset_iter(self) -> None:
        self._it = iter(self._node.range)

    def next(self, result: TaskParameterSet) -> None:
        # Raises: StopIteration
        v = next(self._it)
        result[self._node.name] = ParameterValue(type=self._node.type, value=str(v))


INTERVAL_RE = re.compile(r"\s*(-?[0-9]+)\s*-\s*(-?[0-9]+)\s*")


@dataclass
class RangeListIdentifierNode(Node):
    name: str
    type: ParameterValueType
    range: list[Union[str, int, float, Decimal]]
    range_set: set
    range_constraint: Optional[TaskChunksRangeConstraint_2023_09] = None
    _len: int = field(init=False, repr=False, compare=False)

    def __post_init__(self):
        self._len = len(self.range)

    def __len__(self) -> int:
        return self._len

    def __getitem__(self, index: int) -> TaskParameterSet:
        return {self.name: ParameterValue(type=self.type, value=str(self.range[index]))}

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        param = params[self.name]
        if param.type != self.type:
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} does not match the parameter space type {self.type.name}."
            )
        if self.type == ParameterValueType.CHUNK_INT:
            # The chunk must be an interval if the range constraint is CONTIGUOUS
            if (
                self.range_constraint == TaskChunksRangeConstraint_2023_09.CONTIGUOUS
                and not INTERVAL_RE.match(param.value)
            ):
                raise ValueError(
                    f"Parameter {self.name} of type {param.type.value} value {param.value} is not a contiguous interval like '1-5' as required by the chunk range constraint."
                )
            # The chunk must be a subset of the range
            try:
                range_expr = IntRangeExpr.from_str(param.value)
            except ValueError:
                # If the value is not a range expression
                raise ValueError(
                    f"Parameter {self.name} of type {param.type.value} value {param.value} is not a valid range expression like '1-5,10' as required by the chunk range constraint."
                )
            for v in range_expr:
                if v not in self.range_set:
                    raise ValueError(
                        f"Parameter {self.name} of type {param.type.value} value {param.value} is not a subset of the range in the parameter space."
                    )
        else:
            # The value must be an item in the range
            if param.value not in self.range_set:
                raise ValueError(
                    f"Parameter {self.name} of type {param.type.value} value {param.value} is not in the parameter space range."
                )

    def iter(self) -> RangeListIdentifierNodeIterator:
        return RangeListIdentifierNodeIterator(self)


class RangeExpressionIdentifierNodeIterator(NodeIterator):
    """Iterator for a RangeExpressionIdentifierNode

    Attributes:
        _it: Iterator for the corresponding task parameter
        _node: The RangeExpressionIdentifierNode this is iterating over.
    """

    _it: Iterator[int]
    _node: RangeExpressionIdentifierNode

    def __init__(self, node: RangeExpressionIdentifierNode):
        self._node = node
        self.reset_iter()

    def reset_iter(self) -> None:
        self._it = iter(self._node.range)

    def next(self, result: TaskParameterSet) -> None:
        # Raises: StopIteration
        v = next(self._it)
        result[self._node.name] = ParameterValue(type=self._node.type, value=str(v))


@dataclass
class RangeExpressionIdentifierNode(Node):
    name: str
    type: ParameterValueType
    range: IntRangeExpr
    range_constraint: Optional[TaskChunksRangeConstraint_2023_09] = None
    _len: int = field(init=False, repr=False, compare=False)

    def __post_init__(self):
        self._len = len(self.range)

    def __len__(self) -> int:
        return self._len

    def __getitem__(self, index: int) -> TaskParameterSet:
        return {self.name: ParameterValue(type=self.type, value=str(self.range[index]))}

    def iter(self) -> RangeExpressionIdentifierNodeIterator:
        return RangeExpressionIdentifierNodeIterator(self)

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        param = params[self.name]
        if param.type != self.type:
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} does not match the parameter space type {self.type.name}."
            )
        # The value must be an item in the range
        try:
            if int(param.value) not in self.range:
                raise ValueError(
                    f"Parameter {self.name} of type {param.type.value} value {param.value} is not in the parameter space range."
                )
        except ValueError:
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} value {param.value} is not in the parameter space range."
            )


class AdaptiveContiguousChunkIdentifierNodeIterator(NodeIterator):
    """Iterator for an AdaptiveChunkIdentifierNode

    Attributes:
        _index: The index into _node.range for the start of the next chunk.
        _node: The AdaptiveChunkIdentifierNode this is iterating over.
    """

    _index: int
    _node: AdaptiveContiguousChunkIdentifierNode

    def __init__(self, node: AdaptiveContiguousChunkIdentifierNode):
        self._node = node
        self.reset_iter()

    def reset_iter(self) -> None:
        self._index = 0

    def next(self, result: TaskParameterSet) -> None:
        if self._index >= len(self._node.range):
            raise StopIteration()

        chunks_default_task_count = self._node.chunks_default_task_count
        parameter_range = self._node.range
        range_len = len(parameter_range)
        first_index = self._index
        first_v = last_v = parameter_range[first_index]

        # Step through the range values until reaching the chunk size or a non-contiguous value
        index = first_index + 1
        while (
            (index - first_index) < chunks_default_task_count
            and index < range_len
            and parameter_range[index] == last_v + 1
        ):
            last_v += 1
            index += 1

        self._index = index
        result[self._node.name] = ParameterValue(type=self._node.type, value=f"{first_v}-{last_v}")


@dataclass
class AdaptiveContiguousChunkIdentifierNode(Node):
    name: str
    type: ParameterValueType
    range: list[int]
    _range_set: set[int] = field(init=False, repr=False, compare=False)
    chunks_default_task_count: int
    """This value can be modified by the caller to adapt the chunk size."""

    def __post_init__(self):
        self._range_set = set(self.range)

    def __len__(self) -> int:
        raise ValueError(
            "Length is not available because the parameter space uses adaptive chunking."
        )

    def __getitem__(self, index: int) -> TaskParameterSet:
        raise LookupError(
            "Items cannot be retrieved by index because the parameter space uses adaptive chunking."
        )

    def iter(self) -> AdaptiveContiguousChunkIdentifierNodeIterator:
        return AdaptiveContiguousChunkIdentifierNodeIterator(self)

    def set_chunks_default_task_count(self, value: int) -> None:
        self.chunks_default_task_count = value

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        param = params[self.name]
        if param.type != self.type:
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} does not match the parameter space type {self.type.name}."
            )
        # The chunk must be an interval if the range constraint is CONTIGUOUS
        if not INTERVAL_RE.match(param.value):
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} value {param.value} is not a contiguous interval like '1-5' as required by the chunk range constraint."
            )
        # The chunk must be a subset of the range
        for v in IntRangeExpr.from_str(param.value):
            if v not in self._range_set:
                raise ValueError(
                    f"Parameter {self.name} of type {param.type.value} value {param.value} is not in the parameter space range."
                )


class AdaptiveNoncontiguousChunkIdentifierNodeIterator(NodeIterator):
    """Iterator for an AdaptiveNoncontiguousChunkIdentifierNode

    Attributes:
        _index: The index into _node.range for the start of the next chunk.
        _node: The AdaptiveNoncontiguousChunkIdentifierNode this is iterating over.
    """

    _index: int
    _node: AdaptiveNoncontiguousChunkIdentifierNode

    def __init__(self, node: AdaptiveNoncontiguousChunkIdentifierNode):
        self._node = node
        self.reset_iter()

    def reset_iter(self) -> None:
        self._index = 0

    def next(self, result: TaskParameterSet) -> None:
        if self._index >= len(self._node.range):
            raise StopIteration()

        # Form a range expression of the default chunk size or until the end of the range
        first_index = self._index
        next_first_index = min(
            first_index + self._node.chunks_default_task_count, len(self._node.range)
        )
        v = str(IntRangeExpr.from_list(self._node.range[first_index:next_first_index]))

        self._index = next_first_index
        result[self._node.name] = ParameterValue(type=self._node.type, value=v)


@dataclass
class AdaptiveNoncontiguousChunkIdentifierNode(Node):
    name: str
    type: ParameterValueType
    range: list[int]
    _range_set: set[int] = field(init=False, repr=False, compare=False)
    chunks_default_task_count: int
    """This value can be modified by the caller to adapt the chunk size."""

    def __post_init__(self):
        self._range_set = set(self.range)
        self._len = len(self.range)

    def __len__(self) -> int:
        raise ValueError(
            "Length is not available because the parameter space uses adaptive chunking."
        )

    def __getitem__(self, index: int) -> TaskParameterSet:
        raise LookupError(
            "Items cannot be retrieved by index because the parameter space uses adaptive chunking."
        )

    def iter(self) -> AdaptiveNoncontiguousChunkIdentifierNodeIterator:
        return AdaptiveNoncontiguousChunkIdentifierNodeIterator(self)

    def set_chunks_default_task_count(self, value: int) -> None:
        self.chunks_default_task_count = value

    def validate_containment(self, params: TaskParameterSet):
        """Checks if the params restricted to this node are part of the node's range."""
        param = params[self.name]
        if param.type != self.type:
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} does not match the parameter space type {self.type.name}."
            )
        # The chunk must be a subset of the range
        try:
            for v in IntRangeExpr.from_str(param.value):
                if v not in self._range_set:
                    raise ValueError(
                        f"Parameter {self.name} of type {param.type.value} value {param.value} is not in the parameter space range."
                    )
        except ValueError:
            # If the value is not a range expression
            raise ValueError(
                f"Parameter {self.name} of type {param.type.value} value {param.value} is not a valid range expression like '1-5,10' as required by the chunk range constraint."
            )
