"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user;
        this.group = props.group;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-24.ContainerSecurityContext", version: "2.8.86" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-24.EnvValue", version: "2.8.86" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol?.toString(),
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-24.Container", version: "2.8.86" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-24.Cpu", version: "2.8.86" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-24.EnvFrom", version: "2.8.86" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-24.Env", version: "2.8.86" };
//# sourceMappingURL=data:application/json;base64,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