"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.hasAutoscaler = false;
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
        if (this.isolate) {
            this.connections.isolate();
        }
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const myPorts = container.extractContainerPorts(this);
        const myPortNumbers = myPorts.map(p => p.number);
        const ports = options.ports ?? myPorts.map(p => ({ port: p.number, targetPort: p.number, protocol: p.protocol }));
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        // validate the ports are owned by our containers
        for (const port of ports) {
            const targetPort = port.targetPort ?? port.port;
            if (!myPortNumbers.includes(targetPort)) {
                throw new Error(`Unable to expose deployment ${this.name} via a service: Port ${targetPort} is not exposed by any container`);
            }
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.hasAutoscaler ? undefined : (this.replicas ?? 2),
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
    /**
     * @see IScalable.markHasAutoscaler()
     */
    markHasAutoscaler() {
        this.hasAutoscaler = true;
    }
    /**
     * @see IScalable.toScalingTarget()
     */
    toScalingTarget() {
        return {
            kind: this.apiObject.kind,
            apiVersion: this.apiObject.apiVersion,
            name: this.name,
            containers: this.containers,
            replicas: this.replicas,
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-24.Deployment", version: "2.8.86" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-24.PercentOrAbsolute", version: "2.8.86" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-24.DeploymentStrategy", version: "2.8.86" };
//# sourceMappingURL=data:application/json;base64,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