# WCG 使用示例 / Usage Examples

本文档提供 WCG 的详细使用示例。

## 基本使用

### 1. 安装

```bash
pip install wcg
```

### 2. 初始化配置

首次运行会自动创建默认配置文件：

```bash
wcg run-once
# 会在 config/config.yaml 创建默认配置
```

或者手动创建配置文件：

```bash
mkdir -p config
cat > config/config.yaml << 'EOF'
llm:
  api_url: "https://api.openai.com/v1"
  api_key: "sk-xxxxxxxxxxxxx"
  model: "gpt-3.5-turbo"

github_token: "ghp_xxxxxxxxxxxxx"

repositories:
  - name: "facebook/react"
    branch: "main"
    webhook_url: "https://hooks.slack.com/services/YOUR/WEBHOOK/URL"

schedule_time: "10:00"
timezone: "Asia/Shanghai"  # UTC+8 中国标准时间
summary_dir: "summaries"
EOF
```

**时区说明**: 默认使用 `Asia/Shanghai`（UTC+8）时区。如果您的服务器时间不准确，可以在配置中指定时区，程序会自动按照指定的时区执行定时任务。

### 3. 使用 Web 界面配置（推荐）

```bash
wcg-web
```

访问 http://localhost:5000 进行可视化配置。

## 高级使用

### 使用 Python API

```python
from wcg import ConfigManager, GitHubClient, CommitSummarizer, WebhookNotifier, SummaryStorage

# 加载配置
config_manager = ConfigManager("config/config.yaml")
config = config_manager.load()

# 初始化客户端
github_client = GitHubClient(config.github_token)
summarizer = CommitSummarizer(
    config.llm.api_url,
    config.llm.api_key,
    config.llm.model
)

# 获取提交记录
commits = github_client.get_commits_last_24h("facebook/react", "main")

# 生成汇总
summary = summarizer.summarize_commits(commits, "facebook/react", "main")

# 保存到本地
storage = SummaryStorage(config.summary_dir)
storage.save_summary(summary, "facebook/react", "main")

# 发送到 Webhook
notifier = WebhookNotifier(config.repositories[0].webhook_url)
notifier.send_summary(summary, "facebook/react", "main")
```

### 自定义调度时间

```python
from wcg import CommitWatcher

# 创建 watcher
watcher = CommitWatcher("config/config.yaml")

# 立即运行一次
watcher.run_once()

# 或者启动定时任务
watcher.start()

# 保持运行
import time
try:
    while True:
        time.sleep(60)
except KeyboardInterrupt:
    watcher.stop()
```

### 多仓库配置

```yaml
repositories:
  - name: "facebook/react"
    branch: "main"
    webhook_url: "https://hooks.slack.com/services/YOUR/WEBHOOK/URL1"
  
  - name: "vuejs/vue"
    branch: "main"
    webhook_url: "https://hooks.slack.com/services/YOUR/WEBHOOK/URL2"
  
  - name: "your-org/your-repo"
    branch: "develop"
    webhook_url: "https://discord.com/api/webhooks/YOUR/WEBHOOK"
```

### 使用不同的 LLM 服务

#### OpenAI

```yaml
llm:
  api_url: "https://api.openai.com/v1"
  api_key: "sk-xxxxxxxxxxxxx"
  model: "gpt-3.5-turbo"
```

#### Azure OpenAI

```yaml
llm:
  api_url: "https://your-resource.openai.azure.com/openai/deployments/your-deployment"
  api_key: "your-azure-key"
  model: "gpt-35-turbo"
```

#### 本地 LLM (如 Ollama)

```yaml
llm:
  api_url: "http://localhost:11434/v1"
  api_key: "ollama"
  model: "llama2"
```

#### 其他兼容服务

任何提供 OpenAI 兼容 API 的服务都可以使用：

```yaml
llm:
  api_url: "https://api.your-provider.com/v1"
  api_key: "your-api-key"
  model: "your-model-name"
```

### Webhook 配置示例

#### Slack

```yaml
repositories:
  - name: "your-org/repo"
    branch: "main"
    webhook_url: "https://hooks.slack.com/services/T00000000/B00000000/XXXXXXXXXXXX"
```

#### Discord

```yaml
repositories:
  - name: "your-org/repo"
    branch: "main"
    webhook_url: "https://discord.com/api/webhooks/123456789/xxxxxxxxxxxxx"
```

#### 飞书（Lark）

```yaml
repositories:
  - name: "your-org/repo"
    branch: "main"
    webhook_url: "https://open.feishu.cn/open-apis/bot/v2/hook/xxxxxxxxxxxxx"
```

#### 自定义 Webhook

```yaml
repositories:
  - name: "your-org/repo"
    branch: "main"
    webhook_url: "https://your-server.com/api/notifications"
```

发送的 JSON 格式：

```json
{
  "repository": "your-org/repo",
  "branch": "main",
  "summary": "# Markdown 格式的汇总内容...",
  "type": "commit_summary"
}
```

## 输出示例

生成的汇总示例：

```markdown
# facebook/react (main) - 每日提交汇总

**生成时间**: 2024-01-20 10:00:00

## 新增功能

### packages/react-dom

- **packages/react-dom/src/client/ReactDOMRoot.js** (+45) - [PR #26789](https://github.com/facebook/react/pull/26789/files)
  - Add new createRoot API implementation

### packages/react

- **packages/react/src/React.js** (+23) - [PR #26789](https://github.com/facebook/react/pull/26789/files)
  - Export new Root API

## 代码修改

### packages/react-reconciler

- **packages/react-reconciler/src/ReactFiberWorkLoop.js** (+67/-34) - [PR #26790](https://github.com/facebook/react/pull/26790/files)
  - Optimize fiber work loop performance
  
- **packages/react-reconciler/src/ReactFiberCommitWork.js** (+12/-8) - [PR #26790](https://github.com/facebook/react/pull/26790/files)
  - Update commit phase logic

## 删除内容

### packages/react-dom

- **packages/react-dom/src/client/ReactDOMLegacy.js** (-156) - [PR #26791](https://github.com/facebook/react/pull/26791/files)
  - Remove deprecated legacy API
```

## 故障排查

### 问题：无法连接 GitHub API

```bash
# 检查 GitHub token 是否有效
curl -H "Authorization: token YOUR_TOKEN" https://api.github.com/user

# 检查 token 权限
# 需要 repo 权限才能访问私有仓库
```

### 问题：LLM API 调用失败

```bash
# 测试 API 连接
curl -X POST "YOUR_LLM_API_URL/chat/completions" \
  -H "Authorization: Bearer YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "model": "gpt-3.5-turbo",
    "messages": [{"role": "user", "content": "Hello"}]
  }'
```

### 问题：定时任务没有执行

```bash
# 检查日志
wcg start --log-level DEBUG

# 确认时区设置
date
timedatectl

# 如果服务器时间不准确，在配置文件中设置时区
# config/config.yaml
timezone: "Asia/Shanghai"  # 或其他目标时区
```

**时区配置说明**：
- 默认使用 `Asia/Shanghai`（UTC+8，中国标准时间）
- 如果服务器时间不准确，可在配置中指定 `timezone` 字段
- 使用标准 IANA 时区名称（如 `Asia/Shanghai`、`America/New_York`）
- 程序会按照配置的时区在 `schedule_time` 执行任务
- 即使服务器时区不同，也能准确按目标时区执行
```

### 问题：Webhook 发送失败

```bash
# 测试 webhook
curl -X POST "YOUR_WEBHOOK_URL" \
  -H "Content-Type: application/json" \
  -d '{
    "repository": "test/repo",
    "branch": "main",
    "summary": "Test message",
    "type": "commit_summary"
  }'
```

## 最佳实践

### 1. 安全性

- 使用环境变量存储敏感信息：

```bash
export WCG_GITHUB_TOKEN="ghp_xxxxx"
export WCG_LLM_API_KEY="sk-xxxxx"
```

- 限制配置文件权限：

```bash
chmod 600 config/config.yaml
```

### 2. 性能优化

- 为大型仓库调整时间窗口
- 使用更快的 LLM 模型（如 gpt-3.5-turbo）
- 定期清理旧的汇总文件

### 3. 监控

```bash
# 使用 systemd 查看日志
journalctl -u wcg -f

# 或使用日志文件
wcg start --log-level INFO > wcg.log 2>&1 &
tail -f wcg.log
```

### 4. 备份

定期备份配置和汇总：

```bash
# 备份脚本
#!/bin/bash
DATE=$(date +%Y%m%d)
tar -czf wcg-backup-$DATE.tar.gz config/ summaries/
```

## 集成示例

### 与 CI/CD 集成

```yaml
# .github/workflows/wcg-summary.yml
name: Daily Commit Summary

on:
  schedule:
    - cron: '0 10 * * *'  # 每天 10:00 UTC
  workflow_dispatch:

jobs:
  summarize:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      
      - name: Set up Python
        uses: actions/setup-python@v4
        with:
          python-version: '3.12'
      
      - name: Install WCG
        run: pip install wcg
      
      - name: Run Summary
        env:
          WCG_GITHUB_TOKEN: ${{ secrets.GITHUB_TOKEN }}
          WCG_LLM_API_KEY: ${{ secrets.OPENAI_API_KEY }}
        run: wcg run-once --config .github/wcg-config.yaml
```

### 与 Slack Bot 集成

创建一个 Slack 应用接收汇总并进行格式化展示。

### 与团队协作工具集成

将汇总发送到 Notion、Confluence 等团队文档平台。

## 更多资源

- [GitHub Repository](https://github.com/wZuck/WCG)
- [Issue Tracker](https://github.com/wZuck/WCG/issues)
- [API Documentation](https://github.com/wZuck/WCG#readme)
