#!/bin/bash

# WCG Quick Start Script
# This script helps you get started with WCG quickly

set -e

echo "🔍 WCG - Watch Commits for Git"
echo "================================"
echo ""

# Check if Python is installed
if ! command -v python3 &> /dev/null; then
    echo "❌ Python 3 is not installed. Please install Python 3.12 or later."
    exit 1
fi

# Check Python version
PYTHON_VERSION=$(python3 -c 'import sys; print(".".join(map(str, sys.version_info[:2])))')
echo "✓ Found Python $PYTHON_VERSION"

# Install WCG
echo ""
echo "📦 Installing WCG..."
pip install wcg || {
    echo "❌ Failed to install WCG"
    exit 1
}

echo "✓ WCG installed successfully"

# Create config directory
echo ""
echo "📁 Setting up configuration..."
mkdir -p config
mkdir -p summaries

# Check if config exists
if [ -f "config/config.yaml" ]; then
    echo "⚠️  Configuration file already exists at config/config.yaml"
    read -p "Do you want to overwrite it? (y/N) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "Keeping existing configuration."
    else
        cp config/config.yaml config/config.yaml.backup
        echo "✓ Backed up existing config to config/config.yaml.backup"
    fi
fi

# Create configuration file with prompts
if [ ! -f "config/config.yaml" ] || [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    echo "Let's configure WCG. You can change these settings later."
    echo ""
    
    # LLM Configuration
    read -p "LLM API URL [https://api.openai.com/v1]: " LLM_URL
    LLM_URL=${LLM_URL:-https://api.openai.com/v1}
    
    read -p "LLM API Key: " LLM_KEY
    if [ -z "$LLM_KEY" ]; then
        LLM_KEY="your-api-key-here"
        echo "⚠️  Warning: You need to set a valid API key in config/config.yaml"
    fi
    
    read -p "LLM Model [gpt-3.5-turbo]: " LLM_MODEL
    LLM_MODEL=${LLM_MODEL:-gpt-3.5-turbo}
    
    # GitHub Configuration
    read -p "GitHub Token: " GITHUB_TOKEN
    if [ -z "$GITHUB_TOKEN" ]; then
        GITHUB_TOKEN="your-github-token-here"
        echo "⚠️  Warning: You need to set a valid GitHub token in config/config.yaml"
    fi
    
    # Repository Configuration
    read -p "Repository name (owner/repo) [facebook/react]: " REPO_NAME
    REPO_NAME=${REPO_NAME:-facebook/react}
    
    read -p "Branch name [main]: " BRANCH_NAME
    BRANCH_NAME=${BRANCH_NAME:-main}
    
    read -p "Webhook URL: " WEBHOOK_URL
    if [ -z "$WEBHOOK_URL" ]; then
        WEBHOOK_URL="https://example.com/webhook"
        echo "⚠️  Warning: You need to set a valid webhook URL in config/config.yaml"
    fi
    
    read -p "Schedule time (HH:MM) [10:00]: " SCHEDULE_TIME
    SCHEDULE_TIME=${SCHEDULE_TIME:-10:00}
    
    # Create config file
    cat > config/config.yaml << EOF
llm:
  api_url: "$LLM_URL"
  api_key: "$LLM_KEY"
  model: "$LLM_MODEL"

github_token: "$GITHUB_TOKEN"

repositories:
  - name: "$REPO_NAME"
    branch: "$BRANCH_NAME"
    webhook_url: "$WEBHOOK_URL"

schedule_time: "$SCHEDULE_TIME"
summary_dir: "summaries"
EOF
    
    echo "✓ Configuration saved to config/config.yaml"
fi

echo ""
echo "🎉 Setup complete!"
echo ""
echo "Next steps:"
echo ""
echo "1. Edit config/config.yaml to verify your settings"
echo "2. Test the configuration:"
echo "   wcg run-once"
echo ""
echo "3. Start the scheduler:"
echo "   wcg start"
echo ""
echo "4. Or use the web interface:"
echo "   wcg-web"
echo "   Then visit http://localhost:5000"
echo ""
echo "📚 Documentation:"
echo "   - README: https://github.com/wZuck/WCG#readme"
echo "   - Usage: USAGE.md"
echo "   - Deploy: DEPLOY.md"
echo ""
