"""Command-line interface for WCG."""

import sys
import argparse
import logging
from .scheduler import CommitWatcher, setup_logging


def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(
        description='WCG - Watch Commits for Git',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  wcg start                    # Start the scheduler
  wcg run-once                 # Run once immediately
  wcg --config custom.yaml     # Use custom config file
  wcg --log-level DEBUG        # Set log level
        """
    )
    
    parser.add_argument(
        'command',
        choices=['start', 'run-once'],
        help='Command to execute'
    )
    
    parser.add_argument(
        '--config',
        default=None,
        help='Path to configuration file (default: config/config.yaml)'
    )
    
    parser.add_argument(
        '--log-level',
        default='INFO',
        choices=['DEBUG', 'INFO', 'WARNING', 'ERROR'],
        help='Logging level (default: INFO)'
    )
    
    args = parser.parse_args()
    
    # Setup logging
    setup_logging(args.log_level)
    logger = logging.getLogger(__name__)
    
    # Initialize watcher
    watcher = CommitWatcher(config_path=args.config)
    
    if args.command == 'start':
        logger.info("Starting WCG scheduler...")
        watcher.start()
        
        # Keep running
        try:
            import time
            while True:
                time.sleep(60)
        except KeyboardInterrupt:
            logger.info("Shutting down...")
            watcher.stop()
            sys.exit(0)
    
    elif args.command == 'run-once':
        logger.info("Running WCG once...")
        watcher.run_once()
        logger.info("Done!")


if __name__ == '__main__':
    main()
