"""Tests for storage module."""

import os
import tempfile
from datetime import datetime

from wcg.storage import SummaryStorage


class TestSummaryStorage:
    """Test summary storage."""
    
    def test_save_summary(self):
        """Test saving a summary."""
        with tempfile.TemporaryDirectory() as tmpdir:
            storage = SummaryStorage(tmpdir)
            
            summary = "# Test Summary\n\nThis is a test."
            repo_name = "owner/repo"
            branch = "main"
            
            file_path = storage.save_summary(summary, repo_name, branch)
            
            assert os.path.exists(file_path)
            
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
                assert "Test Summary" in content
                assert "owner/repo" in content
                assert "main" in content
    
    def test_get_summary(self):
        """Test retrieving a summary."""
        with tempfile.TemporaryDirectory() as tmpdir:
            storage = SummaryStorage(tmpdir)
            
            summary = "# Test Summary\n\nContent here."
            repo_name = "test/repo"
            branch = "develop"
            date = datetime(2024, 1, 15)
            
            # Save
            storage.save_summary(summary, repo_name, branch, date)
            
            # Retrieve
            retrieved = storage.get_summary(repo_name, branch, date)
            
            assert retrieved is not None
            assert "Test Summary" in retrieved
    
    def test_get_nonexistent_summary(self):
        """Test retrieving a non-existent summary."""
        with tempfile.TemporaryDirectory() as tmpdir:
            storage = SummaryStorage(tmpdir)
            
            result = storage.get_summary("nonexistent/repo", "main")
            
            assert result is None
    
    def test_list_summaries(self):
        """Test listing summaries."""
        with tempfile.TemporaryDirectory() as tmpdir:
            storage = SummaryStorage(tmpdir)
            
            # Save multiple summaries
            storage.save_summary("Summary 1", "repo1/test", "main")
            storage.save_summary("Summary 2", "repo2/test", "main")
            storage.save_summary("Summary 3", "repo1/test", "develop")
            
            # List all
            all_summaries = storage.list_summaries()
            assert len(all_summaries) == 3
            
            # Filter by repo
            repo1_summaries = storage.list_summaries(repo_name="repo1/test")
            assert len(repo1_summaries) == 2
            
            # Filter by branch
            main_summaries = storage.list_summaries(branch="main")
            assert len(main_summaries) == 2
