"""Tests for commit summarizer with custom prompts."""

from unittest.mock import MagicMock, patch
from wcg.summarizer import CommitSummarizer
from wcg.github_client import CommitInfo


class TestCommitSummarizer:
    """Test commit summarizer functionality."""
    
    def test_custom_prompts_initialization(self):
        """Test that custom prompts are properly initialized."""
        custom_system = "Custom system prompt"
        custom_user = "Custom user template with {repo_name}"
        
        summarizer = CommitSummarizer(
            api_url="https://test.com/v1",
            api_key="test-key",
            model="gpt-4",
            system_prompt=custom_system,
            user_prompt_template=custom_user
        )
        
        assert summarizer.system_prompt == custom_system
        assert summarizer.user_prompt_template == custom_user
    
    def test_default_prompts_when_none(self):
        """Test that default prompts are used when None is provided."""
        summarizer = CommitSummarizer(
            api_url="https://test.com/v1",
            api_key="test-key",
            model="gpt-4"
        )
        
        # Verify default prompts are set
        assert summarizer.system_prompt is not None
        assert len(summarizer.system_prompt) > 0
        assert summarizer.user_prompt_template is not None
        assert "{repo_name}" in summarizer.user_prompt_template
        assert "{branch}" in summarizer.user_prompt_template
        assert "{organized}" in summarizer.user_prompt_template
    
    def test_prompt_template_formatting(self):
        """Test that the prompt template is properly formatted."""
        custom_template = "Repo: {repo_name}, Branch: {branch}, Data: {organized}"
        
        summarizer = CommitSummarizer(
            api_url="https://test.com/v1",
            api_key="test-key",
            model="gpt-4",
            user_prompt_template=custom_template
        )
        
        organized_data = {"added": {}, "modified": {}, "deleted": {}}
        prompt = summarizer._create_prompt(
            organized=organized_data,
            repo_name="test/repo",
            branch="main"
        )
        
        assert "Repo: test/repo" in prompt
        assert "Branch: main" in prompt
        assert "Data:" in prompt
    
    @patch('wcg.summarizer.OpenAI')
    def test_custom_system_prompt_sent_to_api(self, mock_openai):
        """Test that custom system prompt is sent to the API."""
        custom_system = "You are a test assistant"
        
        # Mock the OpenAI client
        mock_client = MagicMock()
        mock_response = MagicMock()
        mock_response.choices = [MagicMock(message=MagicMock(content="Test summary"))]
        mock_client.chat.completions.create.return_value = mock_response
        mock_openai.return_value = mock_client
        
        summarizer = CommitSummarizer(
            api_url="https://test.com/v1",
            api_key="test-key",
            model="gpt-4",
            system_prompt=custom_system
        )
        
        # Create a commit
        from datetime import datetime
        commit = CommitInfo(
            sha="abc123",
            message="Test commit",
            author="test-user",
            date=datetime(2024, 1, 1, 0, 0, 0),
            files_changed=[
                {
                    "filename": "test.py",
                    "status": "modified",
                    "additions": 5,
                    "deletions": 2
                }
            ],
            pr_number=None,
            pr_url=None
        )
        
        # Generate summary
        summarizer.summarize_commits([commit], "test/repo", "main")
        
        # Verify the system prompt was used
        call_args = mock_client.chat.completions.create.call_args
        messages = call_args[1]['messages']
        
        assert len(messages) == 2
        assert messages[0]['role'] == 'system'
        assert messages[0]['content'] == custom_system
