# Circuit-Synth Project

Professional PCB design with Python and KiCad integration.

## = Quick Start

```bash
# Generate KiCad project from Python
uv run python main.py

# Open in KiCad
open output/circuit.kicad_pro
```

## = Project Structure

```
project/
   main.py            # Your circuit design
   output/            # Generated KiCad files (.kicad_pro, .kicad_sch, .kicad_pcb)
   memory-bank/       # Automatic documentation (optional)
   .claude/           # AI assistant configuration (optional)
   README.md          # This file
```

## < Writing Your Circuit

Edit `main.py` to define your circuit:

```python
from circuit_synth import Component, Net, circuit

@circuit(name="MyCircuit")
def my_circuit():
    """Your circuit description"""

    # Define components
    r1 = Component(
        symbol="Device:R",
        ref="R",
        value="10k",
        footprint="Resistor_SMD:R_0603_1608Metric"
    )

    c1 = Component(
        symbol="Device:C",
        ref="C",
        value="0.1uF",
        footprint="Capacitor_SMD:C_0603_1608Metric"
    )

    # Define nets
    vcc = Net('VCC')
    gnd = Net('GND')
    signal = Net('SIGNAL')

    # Make connections
    r1[1] += vcc
    r1[2] += signal
    c1[1] += signal
    c1[2] += gnd

# Generate KiCad files
if __name__ == "__main__":
    circuit = my_circuit()
    circuit.generate_kicad_project(
        project_name="output",
        placement_algorithm="hierarchical",
        generate_pcb=True
    )
    print(" Generated KiCad project in output/")
```

## =' Workflow

1. **Design** your circuit in `main.py`
2. **Generate** KiCad files: `uv run python main.py`
3. **Open** in KiCad and verify schematic
4. **Edit** PCB layout and prepare for manufacturing
5. **Iterate** by modifying Python and regenerating

## = Component Examples

### Basic Components
```python
# Resistor
r = Component(symbol="Device:R", ref="R", value="1k",
              footprint="Resistor_SMD:R_0603_1608Metric")

# Capacitor
c = Component(symbol="Device:C", ref="C", value="10uF",
              footprint="Capacitor_SMD:C_0805_2012Metric")

# LED
led = Component(symbol="Device:LED", ref="D", value="RED",
                footprint="LED_SMD:LED_0603_1608Metric")
```

### Microcontrollers
```python
# ESP32-C6
esp32 = Component(
    symbol="RF_Module:ESP32-C6-MINI-1",
    ref="U",
    footprint="RF_Module:ESP32-C6-MINI-1"
)

# STM32
mcu = Component(
    symbol="MCU_ST_STM32F4:STM32F411CEUx",
    ref="U",
    footprint="Package_QFP:LQFP-48_7x7mm_P0.5mm"
)
```

### Power Components
```python
# 3.3V Linear Regulator
vreg = Component(
    symbol="Regulator_Linear:AMS1117-3.3",
    ref="U",
    footprint="Package_TO_SOT_SMD:SOT-223-3_TabPin2"
)
```

### Connectors
```python
# USB-C
usb = Component(
    symbol="Connector:USB_C_Receptacle_USB2.0_16P",
    ref="J",
    footprint="Connector_USB:USB_C_Receptacle_HRO_TYPE-C-31-M-12"
)

# Pin header
header = Component(
    symbol="Connector_Generic:Conn_01x10_Pin",
    ref="J",
    footprint="Connector_PinHeader_2.54mm:PinHeader_1x10_P2.54mm_Vertical"
)
```

## > AI-Powered Design

Using Claude Code for assistance:

```
"Find me a KiCad symbol for voltage regulator"
"What footprint should I use for 0603 resistor?"
"Help me design a USB-C power supply circuit"
"Check JLCPCB availability for STM32F411"
```

## = Resources

- **Circuit-Synth**: https://github.com/circuit-synth/circuit-synth
- **Documentation**: https://circuit-synth.readthedocs.io
- **KiCad**: https://docs.kicad.org

## = Next Steps

1. Edit `main.py` with your circuit design
2. Run `uv run python main.py` to generate KiCad files
3. Open `output/circuit.kicad_pro` in KiCad
4. Finalize PCB layout and generate manufacturing files

**Happy designing!** <
