from datetime import datetime
from typing import TYPE_CHECKING

from .base import FantraxBaseObject
from .player import Player
from .team import Team

if TYPE_CHECKING:
    from .league import League


class Transaction(FantraxBaseObject):
    """Represents a single Transaction.

    Attributes:
        league (League): The League instance this object belongs to.
        id (str): Transaction ID.
        team (Team): Team who made the Transaction.
        date (datetime): Transaction Date.
        players (list[Player]): Players in the Transaction.

    """

    def __init__(self, league: "League", data: list[dict]) -> None:
        super().__init__(league, data)
        self.id: str = self._data[0]["txSetId"]
        self.team: Team = self.league.team(self._data[0]["cells"][0]["teamId"])
        self.date: datetime = datetime.strptime(self._data[0]["cells"][1]["content"], "%a %b %d, %Y, %I:%M%p")
        tc = "transactionCode"
        self.players: list[Player] = [Player(self.league, p["scorer"], transaction_type=p["claimType"] if p[tc] == "CLAIM" else p[tc]) for p in self._data]

    def __str__(self) -> str:
        return str(self.players)
