# Copyright 2025 Oliver Ratzesberger
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for directory scanner."""

import tempfile
from pathlib import Path

import pytest

from ascii_guard.config import Config
from ascii_guard.scanner import is_text_file, scan_directory, scan_paths


class TestIsTextFile:
    """Test text file detection."""

    def test_text_file_detected(self) -> None:
        """Test that text files are correctly identified."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            text_file = tmppath / "test.txt"
            text_file.write_text("This is a text file\nWith multiple lines\n")

            assert is_text_file(text_file) is True

    def test_empty_file_is_text(self) -> None:
        """Test that empty files are considered text."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            empty_file = tmppath / "empty.txt"
            empty_file.touch()

            assert is_text_file(empty_file) is True

    def test_binary_file_detected(self) -> None:
        """Test that binary files are correctly identified."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            binary_file = tmppath / "test.bin"
            binary_file.write_bytes(b"\x00\x01\x02\x03\x04\x05")

            assert is_text_file(binary_file) is False

    def test_large_file_rejected(self) -> None:
        """Test that files exceeding max_size are rejected."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            large_file = tmppath / "large.txt"
            # Create a 2MB file
            large_file.write_text("x" * (2 * 1024 * 1024))

            # With max_size_mb=1, should be rejected
            assert is_text_file(large_file, max_size_mb=1) is False

            # With max_size_mb=0 (unlimited), should be accepted
            assert is_text_file(large_file, max_size_mb=0) is True

    def test_nonexistent_file(self) -> None:
        """Test that non-existent files return False."""
        assert is_text_file(Path("/nonexistent/file.txt")) is False

    def test_utf8_file(self) -> None:
        """Test that UTF-8 encoded files are detected."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            utf8_file = tmppath / "utf8.txt"
            utf8_file.write_text("Hello 世界 🌍", encoding="utf-8")

            assert is_text_file(utf8_file) is True


class TestScanDirectory:
    """Test directory scanning."""

    def test_scan_empty_directory(self) -> None:
        """Test scanning an empty directory."""
        with tempfile.TemporaryDirectory() as tmpdir:
            config = Config()
            files = scan_directory(tmpdir, config)
            assert files == []

    def test_scan_directory_with_text_files(self) -> None:
        """Test scanning directory with text files."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "file1.txt").write_text("content")
            (tmppath / "file2.md").write_text("content")

            config = Config(exclude=[])  # No excludes
            files = scan_directory(tmppath, config)

            assert len(files) == 2
            assert any(f.name == "file1.txt" for f in files)
            assert any(f.name == "file2.md" for f in files)

    def test_scan_respects_excludes(self) -> None:
        """Test that exclude patterns are respected."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "keep.md").write_text("content")
            (tmppath / "remove.txt").write_text("content")

            config = Config(exclude=["*.txt"])
            files = scan_directory(tmppath, config)

            assert len(files) == 1
            assert files[0].name == "keep.md"

    def test_scan_respects_includes(self) -> None:
        """Test that include patterns override excludes."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "temp1.txt").write_text("content")
            (tmppath / "important.txt").write_text("content")

            config = Config(exclude=["*.txt"], include=["!important.txt"])
            files = scan_directory(tmppath, config)

            assert len(files) == 1
            assert files[0].name == "important.txt"

    def test_scan_respects_extensions(self) -> None:
        """Test that only specified extensions are scanned."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "file.md").write_text("content")
            (tmppath / "file.txt").write_text("content")
            (tmppath / "file.py").write_text("content")

            config = Config(extensions=[".md", ".txt"], exclude=[])
            files = scan_directory(tmppath, config)

            assert len(files) == 2
            assert any(f.name == "file.md" for f in files)
            assert any(f.name == "file.txt" for f in files)
            assert not any(f.name == "file.py" for f in files)

    def test_scan_excludes_directories(self) -> None:
        """Test that excluded directories are not descended into."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create directory structure
            (tmppath / "keep").mkdir()
            (tmppath / "keep" / "file.txt").write_text("content")

            (tmppath / "node_modules").mkdir()
            (tmppath / "node_modules" / "file.txt").write_text("content")

            config = Config(exclude=["node_modules/"])
            files = scan_directory(tmppath, config)

            # Should only find file in keep/
            assert len(files) == 1
            assert "keep" in str(files[0])
            assert "node_modules" not in str(files[0])

    def test_scan_recursive(self) -> None:
        """Test that scanning is recursive."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create nested structure
            (tmppath / "level1").mkdir()
            (tmppath / "level1" / "file1.txt").write_text("content")
            (tmppath / "level1" / "level2").mkdir()
            (tmppath / "level1" / "level2" / "file2.txt").write_text("content")

            config = Config(exclude=[])
            files = scan_directory(tmppath, config)

            assert len(files) == 2

    def test_scan_follows_symlinks_when_configured(self) -> None:
        """Test that symlinks are followed when configured."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create target directory with file
            target_dir = tmppath / "target"
            target_dir.mkdir()
            (target_dir / "file.txt").write_text("content")

            # Create symlink
            link_dir = tmppath / "link"
            try:
                link_dir.symlink_to(target_dir)
            except (OSError, NotImplementedError):
                # Symlinks might not be supported on this system
                pytest.skip("Symlinks not supported")

            # Scan without following symlinks
            config = Config(exclude=[], follow_symlinks=False)
            files = scan_directory(tmppath, config)
            # Should only find target/file.txt
            assert len(files) == 1

            # Scan with following symlinks
            config = Config(exclude=[], follow_symlinks=True)
            files = scan_directory(tmppath, config)
            # Should find file twice (once in target, once via link)
            # Or at least once if symlink resolution works differently
            assert len(files) >= 1

    def test_scan_directory_not_found(self) -> None:
        """Test that FileNotFoundError is raised for non-existent directory."""
        with pytest.raises(FileNotFoundError):
            scan_directory("/nonexistent/directory", Config())

    def test_scan_not_a_directory(self) -> None:
        """Test that NotADirectoryError is raised for file path."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            file_path = tmppath / "file.txt"
            file_path.write_text("content")

            with pytest.raises(NotADirectoryError):
                scan_directory(file_path, Config())

    def test_scan_respects_max_file_size(self) -> None:
        """Test that files exceeding max_file_size are excluded."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create small file
            small = tmppath / "small.txt"
            small.write_text("small content")

            # Create large file (2MB)
            large = tmppath / "large.txt"
            large.write_text("x" * (2 * 1024 * 1024))

            # Scan with max_file_size=1
            config = Config(exclude=[], max_file_size=1)
            files = scan_directory(tmppath, config)

            # Should only find small file
            assert len(files) == 1
            assert files[0].name == "small.txt"

    def test_scan_excludes_binary_files(self) -> None:
        """Test that binary files are automatically excluded."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            (tmppath / "text.txt").write_text("text content")
            (tmppath / "binary.bin").write_bytes(b"\x00\x01\x02\x03")

            config = Config(exclude=[])
            files = scan_directory(tmppath, config)

            # Should only find text file
            assert len(files) == 1
            assert files[0].name == "text.txt"


class TestScanPaths:
    """Test scan_paths utility."""

    def test_scan_single_file(self) -> None:
        """Test scanning a single file path."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            file = tmppath / "test.txt"
            file.write_text("content")

            files = scan_paths([file])
            assert len(files) == 1
            assert files[0] == file.resolve()

    def test_scan_multiple_files(self) -> None:
        """Test scanning multiple file paths."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            file1 = tmppath / "test1.txt"
            file2 = tmppath / "test2.txt"
            file1.write_text("content")
            file2.write_text("content")

            files = scan_paths([file1, file2])
            assert len(files) == 2

    def test_scan_directory_path(self) -> None:
        """Test scanning a directory path."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "file1.txt").write_text("content")
            (tmppath / "file2.txt").write_text("content")

            config = Config(exclude=[])
            files = scan_paths([tmppath], config)
            assert len(files) == 2

    def test_scan_mixed_paths(self) -> None:
        """Test scanning mixed file and directory paths."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create file
            file = tmppath / "file.txt"
            file.write_text("content")

            # Create directory with file
            subdir = tmppath / "subdir"
            subdir.mkdir()
            (subdir / "nested.txt").write_text("content")

            config = Config(exclude=[])
            files = scan_paths([file, subdir], config)
            assert len(files) == 2

    def test_scan_paths_skips_nonexistent(self) -> None:
        """Test that non-existent paths are skipped."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            existing = tmppath / "exists.txt"
            existing.write_text("content")
            nonexistent = tmppath / "nonexistent.txt"

            files = scan_paths([existing, nonexistent])
            assert len(files) == 1
            assert files[0] == existing.resolve()

    def test_scan_paths_files_bypass_filters(self) -> None:
        """Test that explicit file paths bypass config filters."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            file = tmppath / "excluded.txt"
            file.write_text("content")

            # Config that would exclude this file
            config = Config(exclude=["*.txt"])

            # But explicit file path should still be included
            files = scan_paths([file], config)
            assert len(files) == 1
            assert files[0] == file.resolve()

    def test_scan_paths_with_default_config(self) -> None:
        """Test that scan_paths works without explicit config."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)
            (tmppath / "file.txt").write_text("content")

            # Should use default config
            files = scan_paths([tmppath])
            # Default config excludes common directories
            assert len(files) >= 0  # May or may not find files depending on structure
