from typing import Optional

from chalk._lsp.error_builder import get_resolver_error_builder
from chalk.features import DataFrame
from chalk.features.feature_field import Feature
from chalk.features.feature_set import Features
from chalk.features.resolver import RESOLVER_REGISTRY, OnlineResolver, ResourceHint
from chalk.features.underscore import Underscore
from chalk.ml.model_reference import MODEL_REFERENCE_REGISTRY
from chalk.ml.model_version import ModelVersion
from chalk.utils.collections import ensure_tuple


def generate_inference_resolver(
    inputs: list[Underscore], model_version: ModelVersion, resource_hint: Optional[ResourceHint] = None
) -> Feature:
    output_feature = Feature()
    previous_hook = output_feature.hook

    def hook(features: type[Features]) -> None:
        if previous_hook:
            previous_hook(features)

        pkey = features.__chalk_primary__
        pkey_string = str(features.__chalk_primary__)

        def resolver_factory():
            def fn(inp: DataFrame):
                result = model_version.predict(inp[[c for c in inp.columns if c != pkey_string]].to_pyarrow())
                return inp[pkey_string].with_columns({output_feature: result})

            cleaned_inputs = []
            for i in inputs:
                try:
                    cleaned_inputs.append(Feature.from_root_fqn(output_feature.namespace + str(i)[1:]))
                except Exception as e:
                    raise ValueError(f"Could not find feature for input {i}: {e}")

            identifier = model_version.identifier or ""
            model_reference = MODEL_REFERENCE_REGISTRY.get((model_version.name, identifier), None)
            if model_reference is not None:
                model_reference.relations.append(([i.fqn for i in cleaned_inputs], output_feature.fqn))

            return OnlineResolver(
                function_definition="",
                filename="",
                fqn=f"{model_version.name}__{output_feature.namespace}.{output_feature.name}",
                doc=None,
                inputs=[DataFrame[[pkey, *ensure_tuple(cleaned_inputs)]]],
                state=None,
                output=Features[DataFrame[output_feature, pkey]],
                fn=fn,
                environment=None,
                machine_type=None,
                default_args=[None],
                timeout=None,
                cron=None,
                when=None,
                tags=None,
                owner=None,
                resource_hint=resource_hint,
                data_sources=None,
                is_sql_file_resolver=False,
                source_line=None,
                lsp_builder=get_resolver_error_builder(fn),
                parse=None,
                static=False,
                total=False,
                autogenerated=False,
                unique_on=None,
                partitioned_by=None,
                data_lineage=None,
                sql_settings=None,
            )

        RESOLVER_REGISTRY.add_to_deferred_registry(resolver_factory, override=False)

    output_feature.hook = hook

    return output_feature
