"""Module contains method that will be replaced by the plugin.

:author: Pawel Chomicki
"""

import os
import re
from typing import Any, Dict, List, Optional, Tuple

from _pytest.config import Config
from _pytest.main import Session
from _pytest.nodes import Item
from _pytest.reports import TestReport

BOUNDARIES_REGEXP = re.compile(
    r"""
        (?<=[A-Z])(?=[A-Z][a-z]) # Split between acronyms and words, e.g. "PDFFile" -> "PDF File"
        |(?<=[a-z])(?=[A-Z])     # Split between words
        |(?<=[A-Za-z])(?=[0-9])  # Split between letters and numbers
        |(?<=[0-9])(?=[A-Za-z])  # Split between numbers and letters
    """,
    re.VERBOSE,
)


def pytest_runtest_logstart(self, nodeid: str, location: Tuple[str, int, str]) -> None:
    """Signal the start of running a single test item.

    Hook has to be disabled because additional information may break output
    formatting.
    """


def pytest_collection_modifyitems(session: Session, config: Config, items: List[Item]) -> List[Any]:
    def get_module_name(f: Any) -> str:
        return f.listchain()[1].name

    def get_nodeid(f: Any) -> str:
        return "::".join(f.nodeid.split("::")[:-1])

    items.sort(key=get_nodeid)
    items.sort(key=get_module_name)
    return items


def get_report_scopes(report: TestReport) -> List[str]:
    """
    Returns a list of the report's nested scopes, excluding the module.

    >>> report = lambda s: s
    >>> report.nodeid = (
        "specs/user.py::describe_a_user::"
        "describe_email_address::cannot_be_hotmail"
    )
    >>> get_report_scopes(report)
    ['describe_a_user', 'describe_email_address']
    """
    return [i for i in report.nodeid.split("::")[1:-1] if i != "()"]


def pytest_runtest_logreport(self, report: TestReport) -> None:
    """
    Process a test setup/call/teardown report relating to the respective phase
    of executing a test.

    Hook changed to define SPECIFICATION like output format. This hook will
    overwrite also VERBOSE option.
    """
    self.previous_scopes = getattr(self, "previous_scopes", [])
    self.current_scopes = get_report_scopes(report)
    indent = self.config.getini("spec_indent")

    res = self.config.hook.pytest_report_teststatus(report=report, config=self.config)
    cat, letter, word = res
    self.stats.setdefault(cat, []).append(report)
    if _is_ignored(report.nodeid, self.config.getini("spec_ignore").split(",")):
        return
    if not letter and not word:
        return
    if not _is_nodeid_has_test(report.nodeid):
        return
    test_path = _get_test_path(report.nodeid, self.config.getini("spec_header_format"))
    if test_path != self.currentfspath:
        self.currentfspath = test_path
        _print_description(self)

    scope_ind = 0
    for msg in self.current_scopes:
        if msg not in self.previous_scopes:
            msg = [indent * scope_ind + prettify_description(msg)]
            msg = "\n".join(msg)
            if msg:
                _print_description(self, msg)
        scope_ind += 1
    self.previous_scopes = self.current_scopes

    if not isinstance(word, tuple):
        test_name = _get_test_name(report.nodeid)
        parameters = _get_parametrized_parameters(test_name)
        docstring_summary = _get_docstring_summary(report, test_name, parameters)
        markup, test_status = _format_results(report, self.config)
        depth = len(self.current_scopes) or 1
        _print_test_result(self, test_name, docstring_summary, test_status, markup, depth)


def _is_ignored(nodeid: str, ignore_strings: List[str]) -> bool:
    if ignore_strings:
        for ignore_string in ignore_strings:
            if ignore_string and ignore_string in nodeid:
                return True
    return False


def _is_nodeid_has_test(nodeid: str) -> bool:
    if len(nodeid.split("::")) >= 2:
        return True
    return False


def prettify(string: str) -> str:
    return _capitalize_first_letter(_split_boundaries(_replace_underscores(_remove_test_container_prefix(_remove_file_extension(string)))))


def _split_boundaries(string: str) -> str:
    return BOUNDARIES_REGEXP.sub(" ", string)


def prettify_test(string: str) -> str:
    return prettify(_remove_test_prefix(string))


def prettify_description(string: str) -> str:
    return prettify(_append_colon(_remove_test_container_prefix(string)))


def _get_test_path(nodeid: str, header: str) -> str:
    levels = nodeid.split("::")

    module_path = levels[0]
    module_name = os.path.split(levels[0])[1]

    if len(levels) > 2:
        class_name = levels[1]
        test_case = prettify(class_name)
    else:
        class_name = ""
        test_case = prettify(module_name)

    return header.format(
        path=levels[0],
        module_name=module_name,
        module_path=module_path,
        class_name=class_name,
        test_case=test_case,
    )


def _print_description(self, msg: Optional[str] = None) -> None:
    if msg is None:
        msg = self.currentfspath
    if hasattr(self, "_first_triggered"):
        self._tw.line()
    self._tw.line()
    self._tw.write(msg)
    self._first_triggered = True


def _remove_test_container_prefix(nodeid: str) -> str:
    return re.sub("^(Test)|(describe)", "", nodeid)


def _remove_file_extension(nodeid: str) -> str:
    if ".py" not in nodeid:
        return nodeid
    return os.path.splitext(nodeid)[0]


def _remove_module_name(nodeid: str) -> str:
    return nodeid.rsplit("::", 1)[1]


def _remove_test_prefix(nodeid: str) -> str:
    return re.sub("^test_+", "", nodeid)


def _replace_underscores(nodeid: str) -> str:
    return nodeid.replace("__", " ").strip().replace("_", " ").strip()


def _capitalize_first_letter(s: str) -> str:
    return s[:1].capitalize() + s[1:]


def _append_colon(string: str) -> str:
    return "{}:".format(string)


def _get_parametrized_parameters(test_name: str) -> str:
    m = re.search(r"\[.*\]", test_name)
    return m.group(0) if m else ""


def _get_test_name(nodeid: str) -> str:
    test_name = prettify_test(_remove_module_name(nodeid))
    if test_name[:1] == " ":
        test_name_parts = test_name.split("  ")
        if len(test_name_parts) == 1:
            return test_name.strip().capitalize()
        return "The ({}) {}".format(test_name_parts[0][1:].replace(" ", "_"), test_name_parts[1])
    return test_name


def _get_docstring_summary(report: TestReport, test_name: str, parameters: str) -> list[str]:
    docstring_summary: list[str] = getattr(report, "docstring_summary", [])
    if docstring_summary:
        docstring_summary[0] = docstring_summary[0] + parameters
        return docstring_summary
    else:
        return [test_name]


def _format_results(report: TestReport, config: Any) -> Tuple[Dict[str, bool], str]:
    success_indicator = config.getini("spec_success_indicator")
    failure_indicator = config.getini("spec_failure_indicator")
    skipped_indicator = config.getini("spec_skipped_indicator")
    if report.passed:
        return {"green": True}, success_indicator
    elif report.failed:
        return {"red": True}, failure_indicator
    elif report.skipped:
        return {"yellow": True}, skipped_indicator
    return {}, ""


def _print_test_result(
    self,
    test_name: str,
    docstring_summary: list[str],
    test_status: str,
    markup: Dict[str, bool],
    depth: int,
) -> None:
    indent = self.config.getini("spec_indent")

    self._tw.line()
    self._tw.write(
        indent * depth
        + self.config.getini("spec_test_format").format(
            result=test_status,
            name=test_name,
            docstring_summary=docstring_summary[0],
        ),
        **markup,
    )

    for line in docstring_summary[1:]:
        if line.strip() == "":
            break
        self._tw.line()
        self._tw.write(
            indent * (depth + len(test_status)) + line.strip(),
            **markup,
        )
