import pytest
import numpy as np

from optiland import backend as be
from optiland.distribution import (
    CrossDistribution,
    HexagonalDistribution,
    RandomDistribution,
    UniformDistribution,
    RingDistribution,
    LineXDistribution,
    LineYDistribution,
    GaussianQuadrature,
)


# Helper function to count unique points
def count_unique_points(x_coords, y_coords, tol=1e-9):
    """Counts unique (x, y) points with a tolerance."""
    if x_coords is None or y_coords is None or len(x_coords) == 0:
        return 0
    # For backends that might return read-only arrays
    x_coords_writable = np.array(be.to_numpy(x_coords), copy=True)
    y_coords_writable = np.array(be.to_numpy(y_coords), copy=True)

    # Round to a certain number of decimal places to handle floating point inaccuracies
    # The number of decimal places can be derived from tol
    # e.g., tol=1e-9 means 9 decimal places
    decimals = int(-np.log10(tol))

    points = np.stack(
        (np.round(x_coords_writable, decimals), np.round(y_coords_writable, decimals)),
        axis=-1,
    )
    unique_points = np.unique(points, axis=0)
    return unique_points.shape[0]


@pytest.mark.parametrize("backend", ["numpy", "torch"])
class TestDistributionPoints:
    def test_cross_distribution_odd_points(self, backend):
        be.set_backend(backend)
        num_points = 3  # Odd number
        dist = CrossDistribution()
        dist.generate_points(num_points=num_points)
        expected_total_points = 2 * num_points - 1  # Origin counted once
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"CrossDistribution (odd) expected {expected_total_points} unique points, got {unique_count}"
        )

    def test_cross_distribution_even_points(self, backend):
        be.set_backend(backend)
        num_points = 4  # Even number
        dist = CrossDistribution()
        dist.generate_points(num_points=num_points)
        # For even num_points > 0, origin (0,0) is NOT generated by EITHER linspace.
        # So total points = num_points (vertical) + num_points (horizontal) = 2 * num_points.
        expected_total_points = 2 * num_points
        if num_points == 0:  # Edge case
            expected_total_points = 0
            # When num_points is 0, generate_points will result in empty x, y
            # y_line_x/y will be empty. x_line_x/y will be empty.
            # The if num_points % 2 == 1: block won't run.
            # So dist.x/y will be concatenation of empty arrays, which are empty.

        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"CrossDistribution (even) expected {expected_total_points} unique points, got {unique_count}"
        )

    @pytest.mark.parametrize("num_rings", [0, 1, 2, 3])
    def test_hexagonal_distribution(self, backend, num_rings):
        be.set_backend(backend)
        dist = HexagonalDistribution()
        dist.generate_points(num_rings=num_rings)
        if num_rings == 0:
            expected_total_points = 1
        else:
            # Formula: 1 (center) + sum_{i=1 to num_rings} 6*i = 1 + 6 * num_rings * (num_rings+1) / 2
            expected_total_points = 1 + 3 * num_rings * (num_rings + 1)

        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"HexagonalDistribution ({num_rings} rings) expected {expected_total_points} unique points, got {unique_count}"
        )

    def test_random_distribution(self, backend):
        be.set_backend(backend)
        num_points = 100
        dist = RandomDistribution(seed=42)  # Seed for reproducibility
        dist.generate_points(num_points=num_points)
        # For random, exact count might be tricky if true duplicates are astronomically rare
        # but we check if the count matches num_points, assuming no collisions for typical N.
        # The test primarily ensures the generation logic doesn't inherently create duplicates.
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == num_points, (
            f"RandomDistribution expected {num_points} unique points, got {unique_count}"
        )

    def test_uniform_distribution(self, backend):
        be.set_backend(backend)
        num_grid_points = 5  # This means 5x5 grid initially for the underlying meshgrid
        dist = UniformDistribution()
        dist.generate_points(num_points=num_grid_points)
        # The actual number of points is num_grid_points^2 filtered by r2 <= 1
        # We don't know the exact number beforehand without re-implementing the filter.
        # The key is that all points *that are generated* should be unique.
        # So, number of generated points should equal number of unique points.
        num_generated_points = len(dist.x)
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == num_generated_points, (
            f"UniformDistribution expected {num_generated_points} unique points, got {unique_count}"
        )
        assert num_generated_points > 0, "UniformDistribution generated no points"

    def test_ring_distribution(self, backend):
        be.set_backend(backend)
        num_points = 10
        dist = RingDistribution()
        dist.generate_points(num_points=num_points)
        expected_total_points = num_points
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"RingDistribution expected {expected_total_points} unique points, got {unique_count}"
        )

    def test_line_x_distribution(self, backend):
        be.set_backend(backend)
        num_points = 5
        dist = LineXDistribution()  # positive_only=False by default
        dist.generate_points(num_points=num_points)
        expected_total_points = num_points
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"LineXDistribution expected {expected_total_points} unique points, got {unique_count}"
        )

    def test_line_y_distribution(self, backend):
        be.set_backend(backend)
        num_points = 5
        dist = LineYDistribution()  # positive_only=False by default
        dist.generate_points(num_points=num_points)
        expected_total_points = num_points
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"LineYDistribution expected {expected_total_points} unique points, got {unique_count}"
        )

    @pytest.mark.parametrize(
        "num_points", [1, 3, 5]
    )  # Corresponds to radius_dict keys in GaussianQuadrature's _get_radius
    @pytest.mark.parametrize("is_symmetric", [True, False])
    def test_gaussian_quadrature_distribution(
        self, backend, num_points, is_symmetric
    ):  # num_points here is num_rings for GaussianQuad
        be.set_backend(backend)
        dist = GaussianQuadrature(is_symmetric=is_symmetric)
        dist.generate_points(
            num_rings=num_points
        )  # num_points in test param is num_rings for method

        # Expected points calculation based on GaussianQuadrature logic
        # The number of points is determined by the definition of Gaussian Quadrature
        # and not straightforwardly by num_rings like other distributions.
        # For GQ, all generated points should inherently be unique by definition.
        num_generated_points = len(dist.x)

        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == num_generated_points, (
            f"GaussianQuadrature ({num_points} rings, symmetric={is_symmetric}) expected all {num_generated_points} generated points to be unique, but got {unique_count}"
        )
        assert num_generated_points > 0, "GaussianQuadrature generated no points"

    def test_cross_distribution_zero_points(self, backend):
        be.set_backend(backend)
        num_points = 0
        dist = CrossDistribution()
        dist.generate_points(num_points=num_points)
        # Expect 0 points if num_points is 0 for each line, so 0 points total.
        # The current implementation will produce y_line_x=be.zeros(0), y_line_y=linspace(-1,1,0) etc.
        # linspace(-1,1,0) is [], zeros(0) is []. concatenate([],[]) is [].
        # So, dist.x and dist.y will be empty.
        expected_total_points = 0
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"CrossDistribution (zero points) expected {expected_total_points} unique points, got {unique_count}"
        )
        assert len(dist.x) == 0
        assert len(dist.y) == 0

    def test_hexagonal_distribution_negative_rings(self, backend):
        be.set_backend(backend)
        num_rings = -1
        dist = HexagonalDistribution()
        # The generate_points method for HexagonalDistribution has num_rings default to 6.
        # If num_rings < 0, range(num_rings) is empty.
        # It initializes x = be.zeros(1), y = be.zeros(1). Loop doesn't run. So 1 point (origin).
        dist.generate_points(num_rings=num_rings)
        expected_total_points = 1  # Center point only
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"HexagonalDistribution ({num_rings} rings) expected {expected_total_points} unique point (origin), got {unique_count}"
        )

    def test_ring_distribution_zero_points(self, backend):
        be.set_backend(backend)
        num_points = 0
        dist = RingDistribution()
        dist.generate_points(num_points=num_points)
        # linspace(0, 2*pi, 1)[:-1] = [0.0][:-1] = [] for num_points = 0
        # So cos([]) = [], sin([]) = []. dist.x/y will be empty.
        expected_total_points = 0
        unique_count = count_unique_points(dist.x, dist.y)
        assert unique_count == expected_total_points, (
            f"RingDistribution (0 points) expected {expected_total_points} unique points, got {unique_count}"
        )
        assert len(dist.x) == 0
        assert len(dist.y) == 0
