"""
This module contains four napari widgets declared in
different ways:

- a pure Python function flagged with `autogenerate: true`
    in the plugin manifest. Type annotations are used by
    magicgui to generate widgets for each parameter. Best
    suited for simple processing tasks - usually taking
    in and/or returning a layer.
- a `magic_factory` decorated function. The `magic_factory`
    decorator allows us to customize aspects of the resulting
    GUI, including the widgets associated with each parameter.
    Best used when you have a very simple processing task,
    but want some control over the autogenerated widgets. If you
    find yourself needing to define lots of nested functions to achieve
    your functionality, maybe look at the `Container` widget!
- a `magicgui.widgets.Container` subclass. This provides lots
    of flexibility and customization options while still supporting
    `magicgui` widgets and convenience methods for creating widgets
    from type annotations. If you want to customize your widgets and
    connect callbacks, this is the best widget option for you.
- a `QWidget` subclass. This provides maximal flexibility but requires
    full specification of widget layouts, callbacks, events, etc.

References:
- Widget specification: https://napari.org/stable/plugins/building_a_plugin/guides.html#widgets
- magicgui docs: https://pyapp-kit.github.io/magicgui/

Replace code below according to your needs.
"""

import csv
import logging
from pathlib import Path
from typing import TYPE_CHECKING

import numpy as np
from magicgui import magic_factory
from magicgui.widgets import CheckBox, Container, create_widget
from napari.viewer import Viewer
from PIL import Image
from qtpy.QtWidgets import (
    QFileDialog,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QListWidget,
    QMessageBox,
    QPushButton,
    QVBoxLayout,
    QWidget,
)
from skimage.util import img_as_float

logger = logging.getLogger(__name__)
if TYPE_CHECKING:
    import napari


# Uses the `autogenerate: true` flag in the plugin manifest
# to indicate it should be wrapped as a magicgui to autogenerate
# a widget.
def threshold_autogenerate_widget(
    img: "napari.types.ImageData",
    threshold: "float",
) -> "napari.types.LabelsData":
    return img_as_float(img) > threshold


# the magic_factory decorator lets us customize aspects of our widget
# we specify a widget type for the threshold parameter
# and use auto_call=True so the function is called whenever
# the value of a parameter changes
@magic_factory(
    threshold={"widget_type": "FloatSlider", "max": 1}, auto_call=True
)
def threshold_magic_widget(
    img_layer: "napari.layers.Image", threshold: "float"
) -> "napari.types.LabelsData":
    return img_as_float(img_layer.data) > threshold


# if we want even more control over our widget, we can use
# magicgui `Container`
class ImageThreshold(Container):
    def __init__(self, viewer: "napari.viewer.Viewer"):
        super().__init__()
        self._viewer = viewer
        # use create_widget to generate widgets from type annotations
        self._image_layer_combo = create_widget(
            label="Image", annotation="napari.layers.Image"
        )
        self._threshold_slider = create_widget(
            label="Threshold", annotation=float, widget_type="FloatSlider"
        )
        self._threshold_slider.min = 0
        self._threshold_slider.max = 1
        # use magicgui widgets directly
        self._invert_checkbox = CheckBox(text="Keep pixels below threshold")

        # connect your own callbacks
        self._threshold_slider.changed.connect(self._threshold_im)
        self._invert_checkbox.changed.connect(self._threshold_im)

        # append into/extend the container with your widgets
        self.extend(
            [
                self._image_layer_combo,
                self._threshold_slider,
                self._invert_checkbox,
            ]
        )

    def _threshold_im(self):
        image_layer = self._image_layer_combo.value
        if image_layer is None:
            return

        image = img_as_float(image_layer.data)
        name = image_layer.name + "_thresholded"
        threshold = self._threshold_slider.value
        if self._invert_checkbox.value:
            thresholded = image < threshold
        else:
            thresholded = image > threshold
        if name in self._viewer.layers:
            self._viewer.layers[name].data = thresholded
        else:
            self._viewer.add_labels(thresholded, name=name)


class ExampleQWidget(QWidget):
    # your QWidget.__init__ can optionally request the napari viewer instance
    # use a type annotation of 'napari.viewer.Viewer' for any parameter
    def __init__(self, viewer: "napari.viewer.Viewer"):
        super().__init__()
        self.viewer = viewer

        btn = QPushButton("Click me!")
        btn.clicked.connect(self._on_click)

        self.setLayout(QHBoxLayout())
        self.layout().addWidget(btn)

    def _on_click(self):
        print("napari has", len(self.viewer.layers), "layers")


# Project: napari_image_labeler
# Files included below. Create the following structure:
# napari_image_labeler/
# ├── napari_image_labeler
# │   ├── __init__.py
# │   └── _dock_widget.py
# └── setup.cfg

# -------------------------
# File: napari_image_labeler/_dock_widget.py
# -------------------------
"""Dock widget implementation using qtpy (compatible with napari).

Features:
- Select a folder containing images (png/jpg/tif/...)
- List images in the folder
- Show selected image in the napari viewer
- Assign class A/B/C/D via buttons
- Associations saved to a text file (`labels.txt`) in the selected folder
  (format: image_filename,class) — appended and kept in-memory to avoid
  duplicate lines for the same filename (the last label wins).

Usage:
- Install the plugin (see setup.cfg below), run napari, then show the plugin
  via Plugins → Dock widgets → Image labeler (or via the plugin list).

"""


def _is_image_file(p: Path):
    return p.suffix.lower() in {
        ".png",
        ".jpg",
        ".jpeg",
        ".tif",
        ".tiff",
        ".bmp",
        ".gif",
    }


class ImageLabeler(QWidget):
    """A QWidget suitable for use as a napari dock widget.

    The constructor will be called with the napari viewer as first argument by
    napari if the plugin is registered as a dock widget.
    """

    def __init__(self, napari_viewer: Viewer):
        super().__init__()
        self.viewer = napari_viewer
        self.setWindowTitle("Image labeler")

        # State
        self.folder: Path | None = None
        self.images: list[Path] = []
        self.current_image: Path | None = None
        # in-memory mapping filename -> label (str)
        self.labels: dict[str, str] = {}
        self.labels_file_name = "labels.txt"

        # --- Build UI ---
        layout = QVBoxLayout()

        # Folder selector row
        folder_row = QHBoxLayout()
        self.folder_edit = QLineEdit()
        self.folder_edit.setPlaceholderText("Path to folder with images")
        folder_row.addWidget(self.folder_edit)
        btn_browse = QPushButton("Browse")
        btn_browse.clicked.connect(self._on_browse)
        folder_row.addWidget(btn_browse)
        btn_load = QPushButton("Load images")
        btn_load.clicked.connect(self._on_load)
        folder_row.addWidget(btn_load)
        layout.addLayout(folder_row)

        # List of images
        layout.addWidget(QLabel("Images in folder:"))
        self.list_widget = QListWidget()
        self.list_widget.itemSelectionChanged.connect(
            self._on_selection_changed
        )
        layout.addWidget(self.list_widget)

        # Controls row: Show / Prev / Next
        ctrl_row = QHBoxLayout()
        self.btn_show = QPushButton("Show in napari")
        self.btn_show.clicked.connect(self._on_show)
        ctrl_row.addWidget(self.btn_show)
        self.btn_prev = QPushButton("Prev")
        self.btn_prev.clicked.connect(self._on_prev)
        ctrl_row.addWidget(self.btn_prev)
        self.btn_next = QPushButton("Next")
        self.btn_next.clicked.connect(self._on_next)
        ctrl_row.addWidget(self.btn_next)
        layout.addLayout(ctrl_row)

        # Label buttons
        layout.addWidget(QLabel("Assign label:"))
        label_row = QHBoxLayout()
        for cls in ["A", "B", "C", "C3", "D", "E", "E2", "F", "F2", "G", "H","Autres"]:
            b = QPushButton(cls)
            b.clicked.connect(lambda checked, c=cls: self._assign_label(c))
            label_row.addWidget(b)
        layout.addLayout(label_row)

        # Save/export row
        export_row = QHBoxLayout()
        btn_save = QPushButton("Save labels file now")
        btn_save.clicked.connect(self._save_labels_file)
        export_row.addWidget(btn_save)
        btn_clear = QPushButton("Clear labels (memory only)")
        btn_clear.clicked.connect(self._clear_labels)
        export_row.addWidget(btn_clear)
        layout.addLayout(export_row)

        self.setLayout(layout)

    # --- UI callbacks ---
    def _on_browse(self):
        folder = QFileDialog.getExistingDirectory(
            self, "Select folder with images"
        )
        if folder:
            self.folder_edit.setText(folder)

    def _on_load(self):
        text = self.folder_edit.text().strip()
        if not text:
            QMessageBox.warning(
                self, "No folder", "Please provide a folder path."
            )
            return
        p = Path(text)
        if not p.exists() or not p.is_dir():
            QMessageBox.warning(
                self, "Invalid folder", "The path is not a valid directory."
            )
            return
        self.folder = p
        # discover images
        self.images = sorted(
            [f for f in p.iterdir() if f.is_file() and _is_image_file(f)]
        )
        self.list_widget.clear()
        for f in self.images:
            self.list_widget.addItem(f.name)
        # load existing labels file if present
        self._load_labels_file()

    def _on_selection_changed(self):
        items = self.list_widget.selectedItems()
        if not items:
            self.current_image = None
            return
        name = items[0].text()
        if self.folder:
            self.current_image = self.folder / name
        else:
            self.current_image = Path(name)

    def _on_show(self):
        if self.current_image is None:
            QMessageBox.information(
                self, "No image selected", "Select an image first."
            )
            return
        try:
            arr = self._read_image_as_array(self.current_image)
        except Exception as e:
            # QMessageBox.critical(
            #     self, "Read error", f"Could not read image: {e}"
            # )
            # return
            QMessageBox.critical(
                self, "Read error", f"Could not read image: {e}"
            )
            return

        # Add image to napari viewer. Remove existing temporary layer with name 'labeler_preview' first
        layer_name = f"labeler_preview: {self.current_image.name}"
        # remove existing layer with same name
        for layer in list(self.viewer.layers):
            if "labeler_preview" in layer.name:
                self.viewer.layers.remove(layer)
        self.viewer.add_image(arr, name=layer_name, rgb=True)

    def _on_prev(self):
        if not self.images:
            return
        idx = self._current_index() or 0
        idx = max(0, idx - 1)
        self.list_widget.setCurrentRow(idx)
        self._on_show()

    def _on_next(self):
        if not self.images:
            return
        idx = self._current_index() or 0
        idx = min(len(self.images) - 1, idx + 1)
        self.list_widget.setCurrentRow(idx)
        self._on_show()

    # --- Label management ---
    def _assign_label(self, label: str):
        if self.current_image is None:
            QMessageBox.information(
                self, "No image", "Select an image to label."
            )
            return
        fname = self.current_image.name
        self.labels[fname] = label
        self._save_labels_file(
            append=False
        )  # rewrite entire file so last label wins
        # QMessageBox.information(
        #     self, "Saved", f"Assigned label {label} to {fname}"
        # )

    def _save_labels_file(self, append: bool = True):
        """Write labels to the labels file in the folder. By default we rewrite the file
        so the in-memory dictionary controls final content (append=False). If append=True,
        we append new lines (not used in current code but kept for flexibility).
        """
        if self.folder is None:
            QMessageBox.warning(
                self, "No folder", "Load a folder before saving labels."
            )
            return
        out_path = self.folder / self.labels_file_name
        # Rewrite the full file using our in-memory dict so we don't duplicate entries
        try:
            with open(out_path, "w", newline="") as f:
                writer = csv.writer(f)
                writer.writerow(["filename", "label"])  # header
                for fname, lab in sorted(self.labels.items()):
                    writer.writerow([fname, lab])
        except Exception as e:
            # QMessageBox.critical(
            #     self, "Error", f"Could not write labels file: {e}"
            # )
            QMessageBox.critical(
                self, "Read error", f"Could not read image: {e}"
            )
            return

    def _load_labels_file(self):
        if self.folder is None:
            return
        in_path = self.folder / self.labels_file_name
        if not in_path.exists():
            return
        try:
            with open(in_path, newline="") as f:
                reader = csv.DictReader(f)
                for row in reader:
                    if "filename" in row and "label" in row:
                        self.labels[row["filename"]] = row["label"]
        except Exception:
            # ignore parse errors
            # pass
            QMessageBox.critical(self, "Read error", "Could not load file")
            return

    def _clear_labels(self):
        self.labels.clear()
        QMessageBox.information(
            self,
            "Cleared",
            "Labels cleared from memory.\nSave to overwrite file.",
        )

    # def _read_image_as_array(self, p: Path) -> np.ndarray:
    #     from skimage.io import imread

    #     # PIL opens many formats; convert to numpy array (H, W, C) or (H,W) if grayscale
    #     arr = imread(p)
    #     # if palette or single-channel, convert to RGB for safety
    #     print(arr.shape)
    #     if arr.ndim == 2:
    #         return arr
    #     if arr.shape[-1] == 4:
    #         # drop alpha channel
    #         return arr[..., :3]
    #     return arr

    def _read_image_as_array(self, p: Path) -> np.ndarray:
        # PIL opens many formats; convert to numpy array (H, W, C) or (H,W) if grayscale
        im = Image.open(p)
        arr = np.asarray(im)
        # if palette or single-channel, convert to RGB for safety
        print(arr.shape)
        if arr.ndim == 2:
            return arr
        if arr.shape[-1] == 4:
            # drop alpha channel
            return arr[..., :3]
        return arr

    def _current_index(self) -> int | None:
        if self.current_image is None:
            return None
        try:
            return self.images.index(self.current_image)
        except ValueError:
            return None


# -------------------------
# End of files.
# -------------------------

# Notes / suggestions:
# - This implementation uses qtpy (provided by napari) and PIL (Pillow) for reading.
# - Install requirements: pip install "napari[all]" pillow
# - Install plugin locally for development: pip install -e .
# - Launch napari, then open the dock widget (Plugins → Dock widgets → Image labeler)
# - The label file is saved as `labels.txt` (CSV with header) inside the selected folder.
# - If you want automatic keyboard shortcuts for labels or more complex export (COCO, VOC), I
#   can extend the plugin.
