"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,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