"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const projenrc_1 = require("./projenrc");
const util_1 = require("./util");
const constructs_1 = require("./util/constructs");
class FileBase extends component_1.Component {
    /**
     * The projen marker, used to identify files as projen-generated.
     *
     * Value is undefined if the project is being ejected.
     */
    get marker() {
        if (this.project.ejected || !this.shouldAddMarker) {
            return undefined;
        }
        // `marker` is empty if project is being ejected or if explicitly disabled
        const projenrc = projenrc_1.ProjenrcFile.of(this.project)?.filePath ?? common_1.DEFAULT_PROJEN_RC_JS_FILENAME;
        return `${common_1.PROJEN_MARKER}. To modify, edit ${projenrc} and run "${this.project.projenCommand}".`;
    }
    constructor(scope, filePath, options = {}) {
        const project = (0, constructs_1.findClosestProject)(scope);
        const root = project.root;
        const normalizedPath = path.normalize(filePath);
        const projectPath = (0, util_1.normalizePersistedPath)(normalizedPath);
        const absolutePath = path.resolve(project.outdir, projectPath);
        const relativeProjectPath = path.relative(root.outdir, absolutePath);
        const rootProjectPath = (0, util_1.normalizePersistedPath)(relativeProjectPath);
        const autoId = `${new.target.name}@${projectPath}`;
        // Before actually creating the file, ensure the file path is unique within the full project tree
        // This is required because projects can create files inside their subprojects
        if (root.tryFindFile(absolutePath) || scope.node.tryFindChild(autoId)) {
            throw new Error(`There is already a file under ${rootProjectPath}`);
        }
        super(scope, autoId);
        this.node.addMetadata("type", "file");
        this.node.addMetadata("path", rootProjectPath);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = projectPath;
        this.absolutePath = absolutePath;
        this.shouldAddMarker = options.marker ?? true;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? project.commitGenerated ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.annotateGenerated(`/${filePath}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => (0, _resolve_1.resolve)(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            (0, fs_1.rmSync)(filePath, { force: true, recursive: true });
            return;
        }
        // check if the file was changed.
        const prev = (0, util_1.tryReadFileSync)(filePath);
        const prevReadonly = !(0, util_1.isWritable)(filePath);
        const successfulExecutableAssertion = (0, util_1.assertExecutablePermissions)(filePath, this.executable);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            successfulExecutableAssertion) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        (0, util_1.writeFile)(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = (0, util_1.tryReadFileSync)(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.96.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmlsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9maWxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQTRCO0FBQzVCLDZCQUE2QjtBQUU3Qix5Q0FBcUM7QUFDckMscUNBQXdFO0FBQ3hFLDJDQUF3QztBQUN4Qyx5Q0FBMEM7QUFDMUMsaUNBTWdCO0FBQ2hCLGtEQUF1RDtBQXdDdkQsTUFBc0IsUUFBUyxTQUFRLHFCQUFTO0lBd0I5Qzs7OztPQUlHO0lBQ0gsSUFBVyxNQUFNO1FBQ2YsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUNsRCxPQUFPLFNBQVMsQ0FBQztRQUNuQixDQUFDO1FBRUQsMEVBQTBFO1FBQzFFLE1BQU0sUUFBUSxHQUNaLHVCQUFZLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsRUFBRSxRQUFRLElBQUksc0NBQTZCLENBQUM7UUFDM0UsT0FBTyxHQUFHLHNCQUFhLHFCQUFxQixRQUFRLGFBQWEsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLElBQUksQ0FBQztJQUNsRyxDQUFDO0lBRUQsWUFDRSxLQUFpQixFQUNqQixRQUFnQixFQUNoQixVQUEyQixFQUFFO1FBRTdCLE1BQU0sT0FBTyxHQUFHLElBQUEsK0JBQWtCLEVBQUMsS0FBSyxDQUFDLENBQUM7UUFDMUMsTUFBTSxJQUFJLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztRQUMxQixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sV0FBVyxHQUFHLElBQUEsNkJBQXNCLEVBQUMsY0FBYyxDQUFDLENBQUM7UUFDM0QsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQy9ELE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ3JFLE1BQU0sZUFBZSxHQUFHLElBQUEsNkJBQXNCLEVBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUNwRSxNQUFNLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBRW5ELGlHQUFpRztRQUNqRyw4RUFBOEU7UUFDOUUsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQyxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7WUFDdEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsZUFBZSxFQUFFLENBQUMsQ0FBQztRQUN0RSxDQUFDO1FBRUQsS0FBSyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNyQixJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRS9DLElBQUksQ0FBQyxRQUFRLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsQ0FBQztRQUMvRCxJQUFJLENBQUMsVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO1FBQzlDLElBQUksQ0FBQyxJQUFJLEdBQUcsV0FBVyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUM7UUFFOUMsTUFBTSxXQUFXLEdBQUcsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDcEMsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxPQUFPLENBQUMsZUFBZSxJQUFJLElBQUksQ0FBQztRQUN2RSxJQUFJLFNBQVMsSUFBSSxRQUFRLEtBQUssZ0JBQWdCLEVBQUUsQ0FBQztZQUMvQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsSUFBSSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQzVDLENBQUM7UUFDRCxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQztRQUNwRCxJQUFJLGFBQWEsRUFBRSxDQUFDO1lBQ2xCLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEdBQUcsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLElBQUksRUFBRSxDQUFDO2dCQUM5QixNQUFNLElBQUksS0FBSyxDQUNiLDJFQUEyRSxDQUM1RSxDQUFDO1lBQ0osQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBV0Q7O09BRUc7SUFDSSxVQUFVO1FBQ2YsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDbkMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlDLE1BQU0sUUFBUSxHQUFjO1lBQzFCLE9BQU8sRUFBRSxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsRUFBRSxDQUFDLElBQUEsa0JBQU8sRUFBQyxHQUFHLEVBQUUsT0FBTyxDQUFDO1NBQ2pELENBQUM7UUFDRixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakQsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFLENBQUM7WUFDMUIscURBQXFEO1lBQ3JELElBQUEsV0FBTSxFQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7WUFDbkQsT0FBTztRQUNULENBQUM7UUFFRCxpQ0FBaUM7UUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBQSxzQkFBZSxFQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3ZDLE1BQU0sWUFBWSxHQUFHLENBQUMsSUFBQSxpQkFBVSxFQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE1BQU0sNkJBQTZCLEdBQUcsSUFBQSxrQ0FBMkIsRUFDL0QsUUFBUSxFQUNSLElBQUksQ0FBQyxVQUFVLENBQ2hCLENBQUM7UUFDRixJQUNFLElBQUksS0FBSyxTQUFTO1lBQ2xCLE9BQU8sS0FBSyxJQUFJO1lBQ2hCLFlBQVksS0FBSyxJQUFJLENBQUMsUUFBUTtZQUM5Qiw2QkFBNkIsRUFDN0IsQ0FBQztZQUNELElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsUUFBUSxFQUFFLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztZQUN0QixPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUEsZ0JBQVMsRUFBQyxRQUFRLEVBQUUsT0FBTyxFQUFFO1lBQzNCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUN2QixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7U0FDNUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFFNUIsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7SUFDdkIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssb0JBQW9CO1FBQzFCLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNELE1BQU0sUUFBUSxHQUFHLElBQUEsc0JBQWUsRUFBQyxRQUFRLENBQUMsQ0FBQztRQUMzQyxNQUFNLGNBQWMsR0FBRyxRQUFRLEVBQUUsUUFBUSxDQUFDLHNCQUFhLENBQUMsQ0FBQztRQUN6RCxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUNuQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQ3ZCLGtCQUFrQixJQUFJLENBQUMsSUFBSSxvQ0FBb0MsQ0FDaEUsQ0FBQztRQUNKLENBQUM7YUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sSUFBSSxjQUFjLEVBQUUsQ0FBQztZQUMxQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQ3ZCLGtCQUFrQixJQUFJLENBQUMsSUFBSSw4Q0FBOEMsQ0FDMUUsQ0FBQztRQUNKLENBQUM7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQzs7QUF4S0gsNEJBeUtDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgcm1TeW5jIH0gZnJvbSBcImZzXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcbmltcG9ydCB7IHJlc29sdmUgfSBmcm9tIFwiLi9fcmVzb2x2ZVwiO1xuaW1wb3J0IHsgUFJPSkVOX01BUktFUiwgREVGQVVMVF9QUk9KRU5fUkNfSlNfRklMRU5BTUUgfSBmcm9tIFwiLi9jb21tb25cIjtcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gXCIuL2NvbXBvbmVudFwiO1xuaW1wb3J0IHsgUHJvamVucmNGaWxlIH0gZnJvbSBcIi4vcHJvamVucmNcIjtcbmltcG9ydCB7XG4gIGFzc2VydEV4ZWN1dGFibGVQZXJtaXNzaW9ucyxcbiAgaXNXcml0YWJsZSxcbiAgbm9ybWFsaXplUGVyc2lzdGVkUGF0aCxcbiAgdHJ5UmVhZEZpbGVTeW5jLFxuICB3cml0ZUZpbGUsXG59IGZyb20gXCIuL3V0aWxcIjtcbmltcG9ydCB7IGZpbmRDbG9zZXN0UHJvamVjdCB9IGZyb20gXCIuL3V0aWwvY29uc3RydWN0c1wiO1xuXG5leHBvcnQgaW50ZXJmYWNlIEZpbGVCYXNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGlzIGZpbGUgc2hvdWxkIGJlIGNvbW1pdHRlZCB0byBnaXQgb3IgaWdub3JlZC4gQnlcbiAgICogZGVmYXVsdCwgYWxsIGdlbmVyYXRlZCBmaWxlcyBhcmUgY29tbWl0dGVkIGFuZCBhbnRpLXRhbXBlciBpcyB1c2VkIHRvXG4gICAqIHByb3RlY3QgYWdhaW5zdCBtYW51YWwgbW9kaWZpY2F0aW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tbWl0dGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVXBkYXRlIHRoZSBwcm9qZWN0J3MgLmdpdGlnbm9yZSBmaWxlXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVkaXRHaXRpZ25vcmU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBnZW5lcmF0ZWQgZmlsZSBzaG91bGQgYmUgcmVhZG9ubHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHJlYWRvbmx5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgZ2VuZXJhdGVkIGZpbGUgc2hvdWxkIGJlIG1hcmtlZCBhcyBleGVjdXRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhlY3V0YWJsZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEFkZHMgdGhlIHByb2plbiBtYXJrZXIgdG8gdGhlIGZpbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbWFya2VyIHdpbGwgYmUgaW5jbHVkZWQgYXMgbG9uZyBhcyB0aGUgcHJvamVjdCBpcyBub3QgZWplY3RlZFxuICAgKi9cbiAgcmVhZG9ubHkgbWFya2VyPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEZpbGVCYXNlIGV4dGVuZHMgQ29tcG9uZW50IHtcbiAgLyoqXG4gICAqIFRoZSBmaWxlIHBhdGgsIHJlbGF0aXZlIHRvIHRoZSBwcm9qZWN0J3Mgb3V0ZGlyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoZSBmaWxlIHNob3VsZCBiZSByZWFkLW9ubHkgb3IgcmVhZC13cml0ZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seTogYm9vbGVhbjtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoZSBmaWxlIHNob3VsZCBiZSBtYXJrZWQgYXMgZXhlY3V0YWJsZVxuICAgKi9cbiAgcHVibGljIGV4ZWN1dGFibGU6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBhYnNvbHV0ZSBwYXRoIG9mIHRoaXMgZmlsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhYnNvbHV0ZVBhdGg6IHN0cmluZztcblxuICBwcml2YXRlIF9jaGFuZ2VkPzogYm9vbGVhbjtcbiAgcHJpdmF0ZSBzaG91bGRBZGRNYXJrZXI6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBwcm9qZW4gbWFya2VyLCB1c2VkIHRvIGlkZW50aWZ5IGZpbGVzIGFzIHByb2plbi1nZW5lcmF0ZWQuXG4gICAqXG4gICAqIFZhbHVlIGlzIHVuZGVmaW5lZCBpZiB0aGUgcHJvamVjdCBpcyBiZWluZyBlamVjdGVkLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXJrZXIoKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodGhpcy5wcm9qZWN0LmVqZWN0ZWQgfHwgIXRoaXMuc2hvdWxkQWRkTWFya2VyKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIC8vIGBtYXJrZXJgIGlzIGVtcHR5IGlmIHByb2plY3QgaXMgYmVpbmcgZWplY3RlZCBvciBpZiBleHBsaWNpdGx5IGRpc2FibGVkXG4gICAgY29uc3QgcHJvamVucmMgPVxuICAgICAgUHJvamVucmNGaWxlLm9mKHRoaXMucHJvamVjdCk/LmZpbGVQYXRoID8/IERFRkFVTFRfUFJPSkVOX1JDX0pTX0ZJTEVOQU1FO1xuICAgIHJldHVybiBgJHtQUk9KRU5fTUFSS0VSfS4gVG8gbW9kaWZ5LCBlZGl0ICR7cHJvamVucmN9IGFuZCBydW4gXCIke3RoaXMucHJvamVjdC5wcm9qZW5Db21tYW5kfVwiLmA7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihcbiAgICBzY29wZTogSUNvbnN0cnVjdCxcbiAgICBmaWxlUGF0aDogc3RyaW5nLFxuICAgIG9wdGlvbnM6IEZpbGVCYXNlT3B0aW9ucyA9IHt9XG4gICkge1xuICAgIGNvbnN0IHByb2plY3QgPSBmaW5kQ2xvc2VzdFByb2plY3Qoc2NvcGUpO1xuICAgIGNvbnN0IHJvb3QgPSBwcm9qZWN0LnJvb3Q7XG4gICAgY29uc3Qgbm9ybWFsaXplZFBhdGggPSBwYXRoLm5vcm1hbGl6ZShmaWxlUGF0aCk7XG4gICAgY29uc3QgcHJvamVjdFBhdGggPSBub3JtYWxpemVQZXJzaXN0ZWRQYXRoKG5vcm1hbGl6ZWRQYXRoKTtcbiAgICBjb25zdCBhYnNvbHV0ZVBhdGggPSBwYXRoLnJlc29sdmUocHJvamVjdC5vdXRkaXIsIHByb2plY3RQYXRoKTtcbiAgICBjb25zdCByZWxhdGl2ZVByb2plY3RQYXRoID0gcGF0aC5yZWxhdGl2ZShyb290Lm91dGRpciwgYWJzb2x1dGVQYXRoKTtcbiAgICBjb25zdCByb290UHJvamVjdFBhdGggPSBub3JtYWxpemVQZXJzaXN0ZWRQYXRoKHJlbGF0aXZlUHJvamVjdFBhdGgpO1xuICAgIGNvbnN0IGF1dG9JZCA9IGAke25ldy50YXJnZXQubmFtZX1AJHtwcm9qZWN0UGF0aH1gO1xuXG4gICAgLy8gQmVmb3JlIGFjdHVhbGx5IGNyZWF0aW5nIHRoZSBmaWxlLCBlbnN1cmUgdGhlIGZpbGUgcGF0aCBpcyB1bmlxdWUgd2l0aGluIHRoZSBmdWxsIHByb2plY3QgdHJlZVxuICAgIC8vIFRoaXMgaXMgcmVxdWlyZWQgYmVjYXVzZSBwcm9qZWN0cyBjYW4gY3JlYXRlIGZpbGVzIGluc2lkZSB0aGVpciBzdWJwcm9qZWN0c1xuICAgIGlmIChyb290LnRyeUZpbmRGaWxlKGFic29sdXRlUGF0aCkgfHwgc2NvcGUubm9kZS50cnlGaW5kQ2hpbGQoYXV0b0lkKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGVyZSBpcyBhbHJlYWR5IGEgZmlsZSB1bmRlciAke3Jvb3RQcm9qZWN0UGF0aH1gKTtcbiAgICB9XG5cbiAgICBzdXBlcihzY29wZSwgYXV0b0lkKTtcbiAgICB0aGlzLm5vZGUuYWRkTWV0YWRhdGEoXCJ0eXBlXCIsIFwiZmlsZVwiKTtcbiAgICB0aGlzLm5vZGUuYWRkTWV0YWRhdGEoXCJwYXRoXCIsIHJvb3RQcm9qZWN0UGF0aCk7XG5cbiAgICB0aGlzLnJlYWRvbmx5ID0gIXByb2plY3QuZWplY3RlZCAmJiAob3B0aW9ucy5yZWFkb25seSA/PyB0cnVlKTtcbiAgICB0aGlzLmV4ZWN1dGFibGUgPSBvcHRpb25zLmV4ZWN1dGFibGUgPz8gZmFsc2U7XG4gICAgdGhpcy5wYXRoID0gcHJvamVjdFBhdGg7XG4gICAgdGhpcy5hYnNvbHV0ZVBhdGggPSBhYnNvbHV0ZVBhdGg7XG4gICAgdGhpcy5zaG91bGRBZGRNYXJrZXIgPSBvcHRpb25zLm1hcmtlciA/PyB0cnVlO1xuXG4gICAgY29uc3QgZ2xvYlBhdHRlcm4gPSBgLyR7dGhpcy5wYXRofWA7XG4gICAgY29uc3QgY29tbWl0dGVkID0gb3B0aW9ucy5jb21taXR0ZWQgPz8gcHJvamVjdC5jb21taXRHZW5lcmF0ZWQgPz8gdHJ1ZTtcbiAgICBpZiAoY29tbWl0dGVkICYmIGZpbGVQYXRoICE9PSBcIi5naXRhdHRyaWJ1dGVzXCIpIHtcbiAgICAgIHByb2plY3QuYW5ub3RhdGVHZW5lcmF0ZWQoYC8ke2ZpbGVQYXRofWApO1xuICAgIH1cbiAgICBjb25zdCBlZGl0R2l0aWdub3JlID0gb3B0aW9ucy5lZGl0R2l0aWdub3JlID8/IHRydWU7XG4gICAgaWYgKGVkaXRHaXRpZ25vcmUpIHtcbiAgICAgIHRoaXMucHJvamVjdC5hZGRHaXRJZ25vcmUoYCR7Y29tbWl0dGVkID8gXCIhXCIgOiBcIlwifSR7Z2xvYlBhdHRlcm59YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChvcHRpb25zLmNvbW1pdHRlZCAhPSBudWxsKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJnaXRpZ25vcmVcIiBpcyBkaXNhYmxlZCwgc28gaXQgZG9lcyBub3QgbWFrZSBzZW5zZSB0byBzcGVjaWZ5IFwiY29tbWl0dGVkXCInXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEltcGxlbWVudGVkIGJ5IGRlcml2ZWQgY2xhc3NlcyBhbmQgcmV0dXJucyB0aGUgY29udGVudHMgb2YgdGhlIGZpbGUgdG9cbiAgICogZW1pdC5cbiAgICogQHBhcmFtIHJlc29sdmVyIENhbGwgYHJlc29sdmVyLnJlc29sdmUob2JqKWAgb24gYW55IG9iamVjdHMgaW4gb3JkZXIgdG9cbiAgICogcmVzb2x2ZSB0b2tlbiBmdW5jdGlvbnMuXG4gICAqIEByZXR1cm5zIHRoZSBjb250ZW50IHRvIHN5bnRoZXNpemUgb3IgdW5kZWZpbmVkIHRvIHNraXAgdGhlIGZpbGVcbiAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBzeW50aGVzaXplQ29udGVudChyZXNvbHZlcjogSVJlc29sdmVyKTogc3RyaW5nIHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBXcml0ZXMgdGhlIGZpbGUgdG8gdGhlIHByb2plY3QncyBvdXRwdXQgZGlyZWN0b3J5XG4gICAqL1xuICBwdWJsaWMgc3ludGhlc2l6ZSgpIHtcbiAgICBjb25zdCBvdXRkaXIgPSB0aGlzLnByb2plY3Qub3V0ZGlyO1xuICAgIGNvbnN0IGZpbGVQYXRoID0gcGF0aC5qb2luKG91dGRpciwgdGhpcy5wYXRoKTtcbiAgICBjb25zdCByZXNvbHZlcjogSVJlc29sdmVyID0ge1xuICAgICAgcmVzb2x2ZTogKG9iaiwgb3B0aW9ucykgPT4gcmVzb2x2ZShvYmosIG9wdGlvbnMpLFxuICAgIH07XG4gICAgY29uc3QgY29udGVudCA9IHRoaXMuc3ludGhlc2l6ZUNvbnRlbnQocmVzb2x2ZXIpO1xuICAgIGlmIChjb250ZW50ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIHJlbW92ZSBmaWxlIChpZiBleGlzdHMpIGFuZCBza2lwIHJlc3Qgb2Ygc3ludGhlc2lzXG4gICAgICBybVN5bmMoZmlsZVBhdGgsIHsgZm9yY2U6IHRydWUsIHJlY3Vyc2l2ZTogdHJ1ZSB9KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICAvLyBjaGVjayBpZiB0aGUgZmlsZSB3YXMgY2hhbmdlZC5cbiAgICBjb25zdCBwcmV2ID0gdHJ5UmVhZEZpbGVTeW5jKGZpbGVQYXRoKTtcbiAgICBjb25zdCBwcmV2UmVhZG9ubHkgPSAhaXNXcml0YWJsZShmaWxlUGF0aCk7XG4gICAgY29uc3Qgc3VjY2Vzc2Z1bEV4ZWN1dGFibGVBc3NlcnRpb24gPSBhc3NlcnRFeGVjdXRhYmxlUGVybWlzc2lvbnMoXG4gICAgICBmaWxlUGF0aCxcbiAgICAgIHRoaXMuZXhlY3V0YWJsZVxuICAgICk7XG4gICAgaWYgKFxuICAgICAgcHJldiAhPT0gdW5kZWZpbmVkICYmXG4gICAgICBjb250ZW50ID09PSBwcmV2ICYmXG4gICAgICBwcmV2UmVhZG9ubHkgPT09IHRoaXMucmVhZG9ubHkgJiZcbiAgICAgIHN1Y2Nlc3NmdWxFeGVjdXRhYmxlQXNzZXJ0aW9uXG4gICAgKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLmRlYnVnKGBubyBjaGFuZ2UgaW4gJHtmaWxlUGF0aH1gKTtcbiAgICAgIHRoaXMuX2NoYW5nZWQgPSBmYWxzZTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB3cml0ZUZpbGUoZmlsZVBhdGgsIGNvbnRlbnQsIHtcbiAgICAgIHJlYWRvbmx5OiB0aGlzLnJlYWRvbmx5LFxuICAgICAgZXhlY3V0YWJsZTogdGhpcy5leGVjdXRhYmxlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jaGVja0ZvclByb2plbk1hcmtlcigpO1xuXG4gICAgdGhpcy5fY2hhbmdlZCA9IHRydWU7XG4gIH1cblxuICAvKipcbiAgICogRm9yIGRlYnVnZ2luZywgY2hlY2sgd2hldGhlciBhIGZpbGUgd2FzIGluY29ycmVjdGx5IGdlbmVyYXRlZCB3aXRoXG4gICAqIG9yIHdpdGhvdXQgdGhlIHByb2plbiBtYXJrZXIuIFRoZSBwcm9qZW4gbWFya2VyIGRvZXMgbm90ICpuZWVkKiB0byBiZVxuICAgKiBpbmNsdWRlZCBvbiBwcm9qZW4tZ2VuZXJhdGVkIGZpbGVzLCBidXQgaXQncyByZWNvbW1lbmRlZCBzaW5jZSBpdCBzaWduYWxzXG4gICAqIHRoYXQgaXQgcHJvYmFibHkgc2hvdWxkIG5vdCBiZSBlZGl0ZWQgZGlyZWN0bHkuXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRm9yUHJvamVuTWFya2VyKCkge1xuICAgIGNvbnN0IGZpbGVQYXRoID0gcGF0aC5qb2luKHRoaXMucHJvamVjdC5vdXRkaXIsIHRoaXMucGF0aCk7XG4gICAgY29uc3QgY29udGVudHMgPSB0cnlSZWFkRmlsZVN5bmMoZmlsZVBhdGgpO1xuICAgIGNvbnN0IGNvbnRhaW5zTWFya2VyID0gY29udGVudHM/LmluY2x1ZGVzKFBST0pFTl9NQVJLRVIpO1xuICAgIGlmICh0aGlzLm1hcmtlciAmJiAhY29udGFpbnNNYXJrZXIpIHtcbiAgICAgIHRoaXMucHJvamVjdC5sb2dnZXIuZGVidWcoXG4gICAgICAgIGBub3RlOiBleHBlY3RlZCAke3RoaXMucGF0aH0gdG8gY29udGFpbiBtYXJrZXIgYnV0IGZvdW5kIG5vbmUuYFxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKCF0aGlzLm1hcmtlciAmJiBjb250YWluc01hcmtlcikge1xuICAgICAgdGhpcy5wcm9qZWN0LmxvZ2dlci5kZWJ1ZyhcbiAgICAgICAgYG5vdGU6IGV4cGVjdGVkICR7dGhpcy5wYXRofSB0byBub3QgY29udGFpbiBtYXJrZXIgYnV0IGZvdW5kIG9uZSBhbnl3YXkuYFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoZSBmaWxlIGhhcyBiZWVuIGNoYW5nZWQgZHVyaW5nIHN5bnRoZXNpcy4gVGhpcyBwcm9wZXJ0eSBpc1xuICAgKiBvbmx5IGF2YWlsYWJsZSBpbiBgcG9zdFN5bnRoZXNpemUoKWAgaG9va3MuIElmIHRoaXMgaXMgYHVuZGVmaW5lZGAsIHRoZVxuICAgKiBmaWxlIGhhcyBub3QgYmVlbiBzeW50aGVzaXplZCB5ZXQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGNoYW5nZWQoKTogYm9vbGVhbiB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX2NoYW5nZWQ7XG4gIH1cbn1cblxuLyoqXG4gKiBBUEkgZm9yIHJlc29sdmluZyB0b2tlbnMgd2hlbiBzeW50aGVzaXppbmcgZmlsZSBjb250ZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZlciB7XG4gIC8qKlxuICAgKiBHaXZlbiBhIHZhbHVlIChvYmplY3Qvc3RyaW5nL2FycmF5L3doYXRldmVyLCBsb29rcyB1cCBhbnkgZnVuY3Rpb25zIGluc2lkZVxuICAgKiB0aGUgb2JqZWN0IGFuZCByZXR1cm5zIGFuIG9iamVjdCB3aGVyZSBhbGwgZnVuY3Rpb25zIGFyZSBjYWxsZWQuXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUgdG8gcmVzb2x2ZVxuICAgKiBAcGFja2FnZSBvcHRpb25zIFJlc29sdmUgb3B0aW9uc1xuICAgKi9cbiAgcmVzb2x2ZSh2YWx1ZTogYW55LCBvcHRpb25zPzogUmVzb2x2ZU9wdGlvbnMpOiBhbnk7XG59XG5cbi8qKlxuICogUmVzb2x2ZSBvcHRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlc29sdmVPcHRpb25zIHtcbiAgLyoqXG4gICAqIE9taXRzIGVtcHR5IGFycmF5cyBhbmQgb2JqZWN0cy5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9taXRFbXB0eT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENvbnRleHQgYXJndW1lbnRzLlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgYXJncz86IGFueVtdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZhYmxlIHtcbiAgLyoqXG4gICAqIFJlc29sdmVzIGFuZCByZXR1cm5zIGNvbnRlbnQuXG4gICAqL1xuICB0b0pTT04oKTogYW55O1xufVxuIl19