"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constants_1 = require("./constants");
const util_1 = require("./private/util");
const workflow_steps_1 = require("./workflow-steps");
const REPO = (0, util_1.context)("github.repository");
const RUN_ID = (0, util_1.context)("github.run_id");
const SERVER_URL = (0, util_1.context)("github.server_url");
const RUN_URL = `${SERVER_URL}/${REPO}/actions/runs/${RUN_ID}`;
const GIT_PATCH_FILE_DEFAULT = "repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded, the action can optionally fail the job.
     *
     * @param options Options
     * @returns Job steps
     */
    static uploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        const steps = [
            {
                id: options.stepId,
                name: options.stepName ?? "Find mutations",
                shell: "bash",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "${options.outputName}=true" >> $GITHUB_OUTPUT`,
                ].join("\n"),
                // always run from root of repository
                // overrides default working directory which is set by some workflows using this function
                workingDirectory: "./",
            },
            workflow_steps_1.WorkflowSteps.uploadArtifact({
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                with: {
                    name: GIT_PATCH_FILE,
                    path: GIT_PATCH_FILE,
                    includeHiddenFiles: (0, util_1.isHiddenPath)(GIT_PATCH_FILE) ? true : undefined,
                },
            }),
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `uploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        const { patchFile, ...restOfOptions } = options;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        return [
            workflow_steps_1.WorkflowSteps.checkout({ with: restOfOptions }),
            {
                name: "Download patch",
                uses: "actions/download-artifact@v5",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * A step that creates a pull request based on the current repo state.
     *
     * @param options Options
     * @returns Job steps
     */
    static createPullRequest(options) {
        const workflowName = options.workflowName;
        const branchName = options.branchName ?? `github-actions/${workflowName}`;
        const stepId = options.stepId ?? "create-pr";
        const stepName = options.stepName ?? "Create Pull Request";
        const gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        const committer = `${gitIdentity.name} <${gitIdentity.email}>`;
        const pullRequestDescription = options.pullRequestDescription
            .trimEnd()
            .endsWith(".")
            ? options.pullRequestDescription.trimEnd()
            : `${options.pullRequestDescription.trimEnd()}.`;
        const title = options.pullRequestTitle;
        const description = [
            `${pullRequestDescription} See details in [workflow run].`,
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflowName}" workflow*`,
        ].join("\n");
        return [
            {
                name: stepName,
                id: stepId,
                uses: "peter-evans/create-pull-request@v7",
                with: {
                    token: options.credentials?.tokenRef,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    base: options.baseBranch,
                    title: title,
                    labels: options.labels?.join(",") || undefined,
                    assignees: options.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: options.signoff ?? true,
                },
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     *
     * @deprecated use `WorkflowSteps.setupGitIdentity` instead
     */
    static setupGitIdentity(id) {
        return [workflow_steps_1.WorkflowSteps.setupGitIdentity({ gitIdentity: id })];
    }
}
exports.WorkflowActions = WorkflowActions;
_a = JSII_RTTI_SYMBOL_1;
WorkflowActions[_a] = { fqn: "projen.github.WorkflowActions", version: "0.96.2" };
//# sourceMappingURL=data:application/json;base64,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