"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const node_path_1 = require("node:path");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    /**
     * All current jobs of the workflow.
     *
     * This is a read-only copy, use the respective helper methods to add, update or remove jobs.
     */
    get jobs() {
        return { ...this._jobs };
    }
    /**
     * @param github The GitHub component of the project this workflow belongs to.
     * @param name The name of the workflow, displayed under the repository's "Actions" tab.
     * @param options Additional options to configure the workflow.
     */
    constructor(github, name, options = {}) {
        super(github.project, `${new.target.name}#${name}`);
        this._jobs = {};
        this.events = {};
        const defaultConcurrency = {
            cancelInProgress: false,
            group: "${{ github.workflow }}",
        };
        this.name = name;
        this.concurrency = options.limitConcurrency
            ? (0, util_1.deepMerge)([
                defaultConcurrency,
                options.concurrencyOptions,
            ])
            : undefined;
        this.projenCredentials = github.projenCredentials;
        this.actions = github.actions;
        this.env = options.env;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            const fileName = options.fileName ?? `${name.toLocaleLowerCase()}.yml`;
            const extension = (0, node_path_1.extname)(fileName).toLowerCase();
            if (![".yml", ".yaml"].includes(extension)) {
                throw new Error(`GitHub Workflow files must have either a .yml or .yaml file extension, got: ${fileName}`);
            }
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${fileName}`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        verifyJobConstraints(jobs);
        Object.assign(this._jobs, { ...jobs });
    }
    /**
     * Get a single job from the workflow.
     * @param id The job name (unique within the workflow)
     */
    getJob(id) {
        return this._jobs[id];
    }
    /**
     * Updates a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    updateJob(id, job) {
        this.updateJobs({ [id]: job });
    }
    /**
     * Updates jobs for this workflow
     * Does a complete replace, it does not try to merge the jobs
     *
     * @param jobs Jobs to update.
     */
    updateJobs(jobs) {
        verifyJobConstraints(jobs);
        Object.assign(this._jobs, { ...jobs });
    }
    /**
     * Removes a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    removeJob(id) {
        delete this._jobs[id];
    }
    renderWorkflow() {
        return {
            name: this.name,
            "run-name": this.runName,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency
                ? {
                    group: this.concurrency?.group,
                    "cancel-in-progress": this.concurrency.cancelInProgress,
                }
                : undefined,
            env: this.env,
            jobs: renderJobs(this._jobs, this.actions),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.96.2" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[(0, case_1.snake)(k)] = v;
    }
    return result;
}
function renderJobs(jobs, actions) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: (0, util_1.kebabCaseKeys)(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
                strategy: renderJobStrategy(job.strategy),
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = (0, util_1.kebabCaseKeys)((0, _resolve_1.resolve)(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOnGroup) ?? arrayOrScalar(job.runsOn),
            permissions: (0, util_1.kebabCaseKeys)(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: (0, util_1.kebabCaseKeys)(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses && actions.get(step.uses),
            env: step.env,
            run: step.run,
            shell: step.shell,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
            "working-directory": step.workingDirectory,
        };
    }
}
function arrayOrScalar(arr) {
    if (!Array.isArray(arr)) {
        return arr;
    }
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v5",
            with: { distribution: "corretto", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v5",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v6",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v6",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v5",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
function verifyJobConstraints(jobs) {
    // verify that job has a "permissions" statement to ensure workflow can
    // operate in repos with default tokens set to readonly
    for (const [id, job] of Object.entries(jobs)) {
        if (!job.permissions) {
            throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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