"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BundleLogLevel = exports.Charset = exports.SourceMapMode = exports.RunBundleTask = exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const path_1 = require("path");
const util_1 = require("./util");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = options.assetsDir ?? "assets";
        this.loaders = options.loaders;
        this.runBundleTask =
            options.runBundleTask ??
                (options.addToPreCompile === false
                    ? RunBundleTask.MANUAL
                    : RunBundleTask.PRE_COMPILE);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.runBundleTask === RunBundleTask.PRE_COMPILE) {
                this.project.preCompileTask.spawn(this._task);
            }
            else if (this.runBundleTask === RunBundleTask.POST_COMPILE) {
                this.project.postCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        const name = (0, util_1.renderBundleName)(entrypoint);
        const outdir = path.posix.join(this.bundledir, name);
        const outfile = path.posix.join(outdir, options.outfile ?? "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        if (options.tsconfigPath) {
            args.push(`--tsconfig="${options.tsconfigPath}"`);
        }
        for (const x of options.externals ?? []) {
            args.push(`--external:${x}`);
        }
        if (options.sourcemap || options.sourceMapMode) {
            const sourceMapMode = options.sourceMapMode ?? SourceMapMode.DEFAULT;
            const sourceMapValue = sourceMapMode === SourceMapMode.DEFAULT
                ? ""
                : `=${options.sourceMapMode}`;
            args.push(`--sourcemap${sourceMapValue}`);
            if (options.sourcesContent === true) {
                args.push(`--sources-content=${options.sourcesContent}`);
            }
        }
        const format = options.format;
        if (format) {
            args.push(`--format=${format}`);
        }
        const loaders = options.loaders ?? false ? options.loaders : this.loaders ?? false;
        if (loaders) {
            for (let [extension, loader] of Object.entries(loaders)) {
                args.push(`--loader:.${extension}=${loader}`);
            }
        }
        const defines = Object.entries(options.define ?? {});
        for (const [key, value] of defines) {
            args.push(`--define:${key}=${JSON.stringify(value)}`);
        }
        if (options.minify) {
            args.push("--minify");
        }
        if (options.logLevel) {
            args.push(`--log-level=${options.logLevel}`);
        }
        if (options.keepNames) {
            args.push("--keep-names");
        }
        if (options.metafile) {
            args.push(`--metafile=${(0, path_1.join)(outdir, "index.meta.json")}`);
        }
        if (options.banner) {
            args.push(`--banner:js=${JSON.stringify(options.banner)}`);
        }
        if (options.footer) {
            args.push(`--footer:js=${JSON.stringify(options.footer)}`);
        }
        if (options.mainFields) {
            args.push(`--main-fields=${options.mainFields.join(",")}`);
        }
        if (options.inject) {
            args.push(...options.inject.map((i) => `--inject:${i}`));
        }
        if (options.esbuildArgs) {
            const subArgs = new Array();
            for (const [key, value] of Object.entries(options.esbuildArgs)) {
                if (value === true || value === "") {
                    subArgs.push(key);
                }
                else if (value) {
                    subArgs.push(`${key}="${value}"`);
                }
            }
            args.push(subArgs.join(" "));
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        if (options.executable ?? false) {
            bundleTask.exec(`chmod +x ${outfile}`);
        }
        let watchTask;
        const watch = options.watchTask ?? true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outdir: outdir,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.96.2" };
/**
 * Options for BundlerOptions.runBundleTask
 */
var RunBundleTask;
(function (RunBundleTask) {
    /**
     * Don't bundle automatically as part of the build.
     */
    RunBundleTask["MANUAL"] = "manual";
    /**
     * Bundle automatically before compilation.
     */
    RunBundleTask["PRE_COMPILE"] = "pre_compile";
    /**
     * Bundle automatically after compilation. This is useful if you want to
     * bundle the compiled results.
     *
     * Thus will run compilation tasks (using tsc, etc.) before running file
     * through bundling step.
     *
     * This is only required unless you are using new experimental features that
     * are not supported by `esbuild` but are supported by typescript's `tsc`
     * compiler. One example of such feature is `emitDecoratorMetadata`.
     *
     * ```typescript
     * // In a TypeScript project with output configured
     * // to go to the "lib" directory:
     * const project = new TypeScriptProject({
     *   name: "test",
     *   defaultReleaseBranch: "main",
     *   tsconfig: {
     *     compilerOptions: {
     *       outDir: "lib",
     *     },
     *   },
     *   bundlerOptions: {
     *     // ensure we compile with `tsc` before bundling
     *     runBundleTask: RunBundleTask.POST_COMPILE,
     *   },
     * });
     *
     * // Tell the bundler to bundle the compiled results (from the "lib" directory)
     * project.bundler.addBundle("./lib/index.js", {
     *   platform: "node",
     *   target: "node18",
     *   sourcemap: false,
     *   format: "esm",
     * });
     * ```
     **/
    RunBundleTask["POST_COMPILE"] = "post_compile";
})(RunBundleTask || (exports.RunBundleTask = RunBundleTask = {}));
/**
 * SourceMap mode for esbuild
 * @see https://esbuild.github.io/api/#sourcemap
 */
var SourceMapMode;
(function (SourceMapMode) {
    /**
     * Default sourceMap mode - will generate a .js.map file alongside any generated .js file and add a special //# sourceMappingURL=
     * comment to the bottom of the .js file pointing to the .js.map file
     */
    SourceMapMode["DEFAULT"] = "default";
    /**
     *  External sourceMap mode - If you want to omit the special //# sourceMappingURL= comment from the generated .js file but you still
     *  want to generate the .js.map files
     */
    SourceMapMode["EXTERNAL"] = "external";
    /**
     * Inline sourceMap mode - If you want to insert the entire source map into the .js file instead of generating a separate .js.map file
     */
    SourceMapMode["INLINE"] = "inline";
    /**
     * Both sourceMap mode - If you want to have the effect of both inline and external simultaneously
     */
    SourceMapMode["BOTH"] = "both";
})(SourceMapMode || (exports.SourceMapMode = SourceMapMode = {}));
/**
 * Charset for esbuild's output
 */
var Charset;
(function (Charset) {
    /**
     * ASCII
     *
     * Any non-ASCII characters are escaped using backslash escape sequences
     */
    Charset["ASCII"] = "ascii";
    /**
     * UTF-8
     *
     * Keep original characters without using escape sequences
     */
    Charset["UTF8"] = "utf8";
})(Charset || (exports.Charset = Charset = {}));
/**
 * Log levels for esbuild and package managers' install commands.
 */
var BundleLogLevel;
(function (BundleLogLevel) {
    /** Show everything */
    BundleLogLevel["VERBOSE"] = "verbose";
    /** Show everything from info and some additional messages for debugging */
    BundleLogLevel["DEBUG"] = "debug";
    /** Show warnings, errors, and an output file summary */
    BundleLogLevel["INFO"] = "info";
    /** Show warnings and errors */
    BundleLogLevel["WARNING"] = "warning";
    /** Show errors only */
    BundleLogLevel["ERROR"] = "error";
    /** Show nothing */
    BundleLogLevel["SILENT"] = "silent";
})(BundleLogLevel || (exports.BundleLogLevel = BundleLogLevel = {}));
//# sourceMappingURL=data:application/json;base64,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