"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const license_checker_1 = require("./license-checker");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const consts_1 = require("../build/private/consts");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const biome_1 = require("./biome/biome");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const projenrc_json_1 = require("../projenrc-json");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const path_2 = require("../util/path");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease || (exports.AutoRelease = AutoRelease = {}));
/**
 * Node.js project.
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    /**
     * The .npmrc file
     */
    get npmrc() {
        if (!this._npmrc) {
            this._npmrc = new javascript_1.NpmConfig(this, { omitEmpty: true });
        }
        return this._npmrc;
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * The minimum node version required by this package to function.
     *
     * This value indicates the package is incompatible with older versions.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version supported by this package.
     *
     * The value indicates the package is incompatible with newer versions.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    constructor(options) {
        super({
            ...options,
            // Node projects have the specific projen version locked via lockfile, so we can skip the @<VERSION> part of the top-level project
            projenCommand: options.projenCommand ?? "npx projen",
        });
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.workflowPackageCache = options.workflowPackageCache ?? false;
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        const normalizedArtifactsDirectory = (0, util_2.normalizePersistedPath)(this.artifactsDirectory);
        this.artifactsJavascriptDirectory = path_1.posix.join(normalizedArtifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                case node_package_1.NodePackageManager.YARN_CLASSIC:
                case node_package_1.NodePackageManager.YARN_BERRY:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                case node_package_1.NodePackageManager.BUN:
                    return "bun run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        const envCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.PNPM:
                    return '$(pnpm -c exec "node --print process.env.PATH")';
                case node_package_1.NodePackageManager.BUN:
                    return '$(bun --eval "console.log(process.env.PATH)")';
                default:
                    return '$(npx -c "node --print process.env.PATH")';
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", envCommand);
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        const projen = options.projenDevDependency ?? (this.parent ? false : true);
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
            if (!this.deps.isDependencySatisfied("constructs", dependencies_1.DependencyType.BUILD, "^10.0.0")) {
                this.addDevDeps(`constructs@^10.0.0`);
            }
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const workflowPermissions = {
            idToken: this.determineIdTokenPermissions(options),
        };
        const buildWorkflowOptions = options.buildWorkflowOptions ?? {};
        if (buildEnabled && (this.github || github_1.GitHub.of(this.root))) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
                ...buildWorkflowOptions,
                preBuildSteps: this.renderWorkflowSetup({
                    installStepConfiguration: {
                        workingDirectory: this.determineInstallWorkingDirectory(),
                    },
                    mutable: buildWorkflowOptions.mutableBuild ?? options.mutableBuild ?? true,
                }).concat(buildWorkflowOptions.preBuildSteps ?? []),
                postBuildSteps: [...(options.postBuildSteps ?? [])],
                ...(0, runner_options_1.filteredRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
            this.maybeAddCodecovIgnores(options);
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.release = new release_1.Release(this, {
                versionFile: "package.json", // this is where "version" is set after bump
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                ...options,
                artifactsDirectory: this.artifactsDirectory,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({
                        installStepConfiguration: {
                            workingDirectory: this.determineInstallWorkingDirectory(),
                        },
                        mutable: false,
                    }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.maybeAddCodecovIgnores(options);
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = (0, release_1.isAwsCodeArtifactRegistry)(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    npmProvenance: this.package.npmProvenance,
                    trustedPublishing: options.npmTrustedPublishing ?? false,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot((0, util_2.deepMerge)([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, (0, util_2.deepMerge)([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            const projenrcJsFile = new projenrc_1.Projenrc(this, options.projenrcJsOptions);
            this.npmignore?.exclude(`/${projenrcJsFile.filePath}`);
        }
        else if (options.projenrcJson) {
            const projenrcJsonFile = projenrc_json_1.ProjenrcJson.of(this);
            if (projenrcJsonFile) {
                this.npmignore?.exclude(`/${projenrcJsonFile.filePath}`);
            }
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        const shouldPackage = options.package ?? true;
        if (release && !shouldPackage) {
            this.logger.warn("When `release` is enabled, `package` must also be enabled as it is required by release. Force enabling `package`.");
        }
        if (release || shouldPackage) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            const pkgMgr = this.package.packageManager === node_package_1.NodePackageManager.PNPM
                ? "pnpm"
                : "npm";
            this.packageTask.exec(`${pkgMgr} pack --pack-destination ${this.artifactsJavascriptDirectory}`);
        }
        if ((0, util_2.multipleSelected)([options.biome, options.prettier])) {
            throw new Error("Only one of biome and prettier can be enabled.");
        }
        if (options.biome ?? false) {
            this.biome = new biome_1.Biome(this, { ...options.biomeOptions });
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
        // For PNPM, the default resolution mode is "lowest", which leads to any non-versioned (ie '*') dependencies being
        // resolved to the lowest available version, which is unlikely to be expected behaviour for users. We set resolution
        // mode to "highest" to match the behaviour of yarn and npm.
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("resolution-mode", "highest");
        }
        if (options.checkLicenses) {
            new license_checker_1.LicenseChecker(this, options.checkLicenses);
        }
    }
    determineInstallWorkingDirectory() {
        if (this.parent) {
            return (0, path_2.ensureRelativePathStartsWithDot)((0, path_1.relative)(".", this.root.outdir));
        }
        return;
    }
    determineIdTokenPermissions(options) {
        const { codeCovTokenSecret, scopedPackagesOptions, codeArtifactOptions } = options;
        const useCodeCoveOidc = this.useCodecov(options) && !codeCovTokenSecret;
        const hasScopedPackages = (scopedPackagesOptions ?? []).length > 0;
        const useCodeArtifactOidc = codeArtifactOptions?.authProvider ===
            release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        if (useCodeCoveOidc || (useCodeArtifactOidc && hasScopedPackages)) {
            return workflows_model_1.JobPermission.WRITE;
        }
        return undefined;
    }
    useCodecov(options) {
        // Use Codecov when it is enabled or if or a secret token name is passed in
        // AND jest must be configured
        return (options.codeCov || options.codeCovTokenSecret) && this.jest?.config;
    }
    maybeAddCodecovIgnores(options) {
        if (this.useCodecov(options)) {
            this.addGitIgnore("codecov");
            this.addGitIgnore("codecov.*");
        }
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if (this.useCodecov(options)) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v5",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest?.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest?.config.coverageDirectory,
                            use_oidc: true,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts) {
        for (const [name, command] of Object.entries(scripts)) {
            this.package.setScript(name, command);
        }
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-access-key-id": (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v4",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === node_package_1.NodePackageManager.BUN) {
            install.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        if (this.package.packageManager !== node_package_1.NodePackageManager.BUN) {
            if (this.nodeVersion || this.workflowPackageCache) {
                const cache = this.package.packageManager === node_package_1.NodePackageManager.YARN
                    ? "yarn"
                    : this.package.packageManager === node_package_1.NodePackageManager.YARN2
                        ? "yarn"
                        : this.package.packageManager === node_package_1.NodePackageManager.YARN_CLASSIC
                            ? "yarn"
                            : this.package.packageManager === node_package_1.NodePackageManager.YARN_BERRY
                                ? "yarn"
                                : this.packageManager === node_package_1.NodePackageManager.BUN
                                    ? "bun"
                                    : this.package.packageManager === node_package_1.NodePackageManager.PNPM
                                        ? "pnpm"
                                        : "npm";
                install.push({
                    name: "Setup Node.js",
                    uses: "actions/setup-node@v5",
                    with: {
                        ...(this.nodeVersion && {
                            "node-version": this.nodeVersion,
                        }),
                        ...(this.workflowPackageCache && {
                            cache,
                        }),
                    },
                });
            }
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
            ...(options.installStepConfiguration ?? {}),
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.96.2" };
//# sourceMappingURL=data:application/json;base64,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