import { Component } from "../component";
import { DependencyType } from "../dependencies";
import { GithubCredentials, GithubWorkflow, GitIdentity, workflows } from "../github";
import { ContainerOptions, JobStep, JobPermissions } from "../github/workflows-model";
import { NodeProject } from "../javascript";
import { GroupRunnerOptions } from "../runner-options";
import { Task } from "../task";
/**
 * Options for `UpgradeDependencies`.
 */
export interface UpgradeDependenciesOptions {
    /**
     * List of package names to exclude during the upgrade.
     *
     * @default - Nothing is excluded.
     */
    readonly exclude?: string[];
    /**
     * List of package names to include during the upgrade.
     *
     * @default - Everything is included.
     */
    readonly include?: string[];
    /**
     * Determines the target version to upgrade dependencies to.
     *
     * @see https://github.com/raineorshine/npm-check-updates#target
     *
     * @default "minor"
     */
    readonly target?: string;
    /**
     * Check peer dependencies of installed packages and filter updates to compatible versions.
     *
     * By default, the upgrade workflow will adhere to version constraints from peer dependencies.
     * Sometimes this is not desirable and can be disabled.
     *
     * @see https://github.com/raineorshine/npm-check-updates#peer
     *
     * @default true
     */
    readonly satisfyPeerDependencies?: boolean;
    /**
     * Include deprecated packages.
     *
     * By default, deprecated versions will be excluded from upgrades.
     *
     * @see https://github.com/raineorshine/npm-check-updates?tab=readme-ov-file#options
     *
     * @default false
     */
    readonly includeDeprecatedVersions?: boolean;
    /**
     * Include a github workflow for creating PR's that upgrades the
     * required dependencies, either by manual dispatch, or by a schedule.
     *
     * If this is `false`, only a local projen task is created, which can be executed manually to
     * upgrade the dependencies.
     *
     * @default - true for root projects, false for subprojects.
     */
    readonly workflow?: boolean;
    /**
     * Options for the github workflow. Only applies if `workflow` is true.
     *
     * @default - default options.
     */
    readonly workflowOptions?: UpgradeDependenciesWorkflowOptions;
    /**
     * The name of the task that will be created.
     * This will also be the workflow name.
     *
     * @default "upgrade".
     */
    readonly taskName?: string;
    /**
     * Title of the pull request to use (should be all lower-case).
     *
     * @default "upgrade dependencies"
     */
    readonly pullRequestTitle?: string;
    /**
     * The semantic commit type.
     *
     * @default 'chore'
     */
    readonly semanticCommit?: string;
    /**
     * Add Signed-off-by line by the committer at the end of the commit log message.
     *
     * @default true
     */
    readonly signoff?: boolean;
    /**
     * Specify which dependency types the upgrade should operate on.
     *
     * @default - All dependency types.
     */
    readonly types?: DependencyType[];
}
/**
 * Upgrade node project dependencies.
 */
export declare class UpgradeDependencies extends Component {
    /**
     * The workflows that execute the upgrades. One workflow per branch.
     */
    readonly workflows: GithubWorkflow[];
    private readonly options;
    private readonly _project;
    private readonly pullRequestTitle;
    /**
     * Container definitions for the upgrade workflow.
     */
    containerOptions?: ContainerOptions;
    /**
     * The upgrade task.
     */
    readonly upgradeTask: Task;
    /**
     * A task run after the upgrade task.
     */
    readonly postUpgradeTask: Task;
    private readonly gitIdentity;
    private readonly postBuildSteps;
    private readonly permissions;
    private readonly depTypes;
    private readonly upgradeTarget;
    private readonly satisfyPeerDependencies;
    private readonly includeDeprecatedVersions;
    constructor(project: NodeProject, options?: UpgradeDependenciesOptions);
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps: JobStep[]): void;
    private renderTaskSteps;
    /**
     * Render projen dependencies types to a list of ncu compatible types
     */
    private renderNcuDependencyTypes;
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    private renderUpgradePackagesCommand;
    private buildDependencyList;
    private filterDependencies;
    /**
     * Projen can alter a package's version in package.json when either the version is omitted, or set to "*".
     * Otherwise, the exact version selected is placed in the package.json file and upgrading is handled through the package manager
     * rather than npm-check-updates.
     *
     * @param version semver from DependencyCoordinates.version, may be undefined
     * @returns whether the version is the default versioning behavior
     */
    private packageCanBeUpgradedInPackageJson;
    private createWorkflow;
    private createUpgrade;
    private createPr;
}
/**
 * Options for `UpgradeDependencies.workflowOptions`.
 */
export interface UpgradeDependenciesWorkflowOptions {
    /**
     * Schedule to run on.
     *
     * @default UpgradeDependenciesSchedule.DAILY
     */
    readonly schedule?: UpgradeDependenciesSchedule;
    /**
     * Choose a method for authenticating with GitHub for creating the PR.
     *
     * When using the default github token, PR's created by this workflow
     * will not trigger any subsequent workflows (i.e the build workflow), so
     * projen requires API access to be provided through e.g. a personal
     * access token or other method.
     *
     * @see https://github.com/peter-evans/create-pull-request/issues/48
     * @default - personal access token named PROJEN_GITHUB_TOKEN
     */
    readonly projenCredentials?: GithubCredentials;
    /**
     * Labels to apply on the PR.
     *
     * @default - no labels.
     */
    readonly labels?: string[];
    /**
     * Assignees to add on the PR.
     *
     * @default - no assignees
     */
    readonly assignees?: string[];
    /**
     * Job container options.
     *
     * @default - defaults
     */
    readonly container?: workflows.ContainerOptions;
    /**
     * List of branches to create PR's for.
     *
     * @default - All release branches configured for the project.
     */
    readonly branches?: string[];
    /**
     * The git identity to use for commits.
     * @default "github-actions@github.com"
     */
    readonly gitIdentity?: GitIdentity;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     * @description Defines a target Runner by labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOn?: string[];
    /**
     * Github Runner Group selection options
     * @description Defines a target Runner Group by name and/or labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOnGroup?: GroupRunnerOptions;
    /**
     * Permissions granted to the upgrade job
     * To limit job permissions for `contents`, the desired permissions have to be explicitly set, e.g.: `{ contents: JobPermission.NONE }`
     * @default `{ contents: JobPermission.READ }`
     */
    readonly permissions?: JobPermissions;
}
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
export declare class UpgradeDependenciesSchedule {
    readonly cron: string[];
    /**
     * Disables automatic upgrades.
     */
    static readonly NEVER: UpgradeDependenciesSchedule;
    /**
     * At 00:00.
     */
    static readonly DAILY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on every day-of-week from Monday through Friday.
     */
    static readonly WEEKDAY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on Monday.
     */
    static readonly WEEKLY: UpgradeDependenciesSchedule;
    /**
     * At 00:00 on day-of-month 1.
     */
    static readonly MONTHLY: UpgradeDependenciesSchedule;
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron: string[]): UpgradeDependenciesSchedule;
    private constructor();
}
