"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogLevel = exports.Logger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const chalk = require("chalk");
const common_1 = require("./common");
const component_1 = require("./component");
const logging_1 = require("./logging");
const util_1 = require("./util");
const constructs_1 = require("./util/constructs");
/**
 * Project-level logging utilities.
 */
class Logger extends component_1.Component {
    constructor(scope, options = {}) {
        const project = (0, constructs_1.findClosestProject)(scope);
        super(scope, `${new.target.name}#${project.name}`);
        // if we are running inside a test, default to no logs
        const defaultLevel = common_1.IS_TEST_RUN ? LogLevel.OFF : LogLevel.INFO;
        this.level = options.level ?? defaultLevel;
        this.usePrefix = options.usePrefix ?? false;
    }
    /**
     * Log a message to stderr with a given logging level. The message will be
     * printed as long as `logger.level` is set to the message's severity or higher.
     *
     * @param level Logging verbosity
     * @param text strings or objects to print
     */
    log(level, ...text) {
        if (level === LogLevel.OFF) {
            throw new Error("Cannot log a message with level LogLevel.OFF");
        }
        let maxLevel = this.level;
        // logging level can be overridden through the --debug CLI option
        if ((0, util_1.isTruthy)(process.env.DEBUG) && maxLevel < LogLevel.DEBUG) {
            maxLevel = LogLevel.DEBUG;
        }
        if (level <= maxLevel) {
            const color = this.colorForLogLevel(level) ??
                ((...values) => values.join(" "));
            const prefix = this.usePrefix ? `[${this.project.name}] ` : "";
            console.error(`${logging_1.ICON} ${prefix}${color(...text)}`);
        }
    }
    /**
     * Log a message to stderr with VERBOSE severity
     * @param text strings or objects to print
     */
    verbose(...text) {
        this.log(LogLevel.VERBOSE, ...text);
    }
    /**
     * Log a message to stderr with DEBUG severity
     * @param text strings or objects to print
     */
    debug(...text) {
        this.log(LogLevel.DEBUG, ...text);
    }
    /**
     * Log a message to stderr with INFO severity
     * @param text strings or objects to print
     */
    info(...text) {
        this.log(LogLevel.INFO, ...text);
    }
    /**
     * Log a message to stderr with WARN severity
     * @param text strings or objects to print
     */
    warn(...text) {
        this.log(LogLevel.WARN, ...text);
    }
    /**
     * Log a message to stderr with ERROR severity
     * @param text strings or objects to print
     */
    error(...text) {
        this.log(LogLevel.ERROR, ...text);
    }
    colorForLogLevel(level) {
        switch (level) {
            case LogLevel.ERROR:
                return chalk.red;
            case LogLevel.WARN:
                return chalk.yellow;
            case LogLevel.INFO:
                return chalk.cyan;
            case LogLevel.DEBUG:
                return chalk.gray;
            case LogLevel.VERBOSE:
                return chalk.white;
            default:
                return chalk.white;
        }
    }
}
exports.Logger = Logger;
_a = JSII_RTTI_SYMBOL_1;
Logger[_a] = { fqn: "projen.Logger", version: "0.96.2" };
/**
 * Logging verbosity.
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["OFF"] = "00.off";
    LogLevel["ERROR"] = "10.error";
    LogLevel["WARN"] = "20.warn";
    LogLevel["INFO"] = "30.info";
    LogLevel["DEBUG"] = "40.debug";
    LogLevel["VERBOSE"] = "50.verbose";
})(LogLevel || (exports.LogLevel = LogLevel = {}));
//# sourceMappingURL=data:application/json;base64,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