"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const constructs_1 = require("constructs");
const glob = require("fast-glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
const constructs_2 = require("./util/constructs");
/**
 * The default output directory for a project if none is specified.
 */
const DEFAULT_OUTDIR = ".";
/**
 * Base project
 */
class Project extends constructs_1.Construct {
    /**
     * Test whether the given construct is a project.
     */
    static isProject(x) {
        return (0, constructs_2.isProject)(x);
    }
    /**
     * Find the closest ancestor project for given construct.
     * When given a project, this it the project itself.
     *
     * @throws when no project is found in the path to the root
     */
    static of(construct) {
        return (0, constructs_2.findClosestProject)(construct);
    }
    /**
     * The command to use in order to run the projen CLI.
     */
    get projenCommand() {
        return this._projenCommand ?? "npx projen";
    }
    constructor(options) {
        const outdir = determineOutdir(options.parent, options.outdir);
        const autoId = `${new.target.name}#${options.name}@${path.normalize(options.outdir ?? "<root>")}`;
        if (options.parent?.subprojects.find((p) => p.outdir === outdir)) {
            throw new Error(`There is already a subproject with "outdir": ${outdir}`);
        }
        super(options.parent, autoId);
        this.tips = new Array();
        (0, constructs_2.tagAsProject)(this);
        this.node.addMetadata("type", "project");
        this.node.addMetadata("construct", new.target.name);
        this.initProject = (0, render_options_1.resolveInitProject)(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = (0, util_1.isTruthy)(process.env.PROJEN_EJECTING);
        this._projenCommand = options.projenCommand;
        if (this.ejected) {
            this._projenCommand = "scripts/run-task.cjs";
        }
        this.outdir = outdir;
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this, {
            endOfLine: options.gitOptions?.endOfLine,
        });
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore", options.gitIgnoreOptions);
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injection but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            // Subtasks should call the root task for synth
            if (this.parent) {
                const cwd = path.relative(this.outdir, this.root.outdir);
                const normalizedCwd = (0, util_1.normalizePersistedPath)(cwd);
                this.defaultTask.exec(`${this.projenCommand} ${Project.DEFAULT_TASK}`, {
                    cwd: normalizedCwd,
                });
            }
            if (!this.parent) {
                this.ejectTask = this.tasks.addTask("eject", {
                    description: "Remove projen from the project",
                    env: {
                        PROJEN_EJECTING: "true",
                    },
                });
                this.ejectTask.spawn(this.defaultTask);
            }
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (!this.parent && projenrcJson) {
            new projenrc_json_1.ProjenrcJson(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => (0, util_1.normalizePersistedPath)(f.path)),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * The root project.
     */
    get root() {
        return (0, constructs_2.isProject)(this.node.root) ? this.node.root : this;
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return this.node
            .findAll()
            .filter((c) => (0, constructs_2.isComponent)(c) && c.project.node.path === this.node.path);
    }
    /**
     * Returns all the subprojects within this project.
     */
    get subprojects() {
        return this.node.children.filter(constructs_2.isProject);
    }
    /**
     * All files in this project.
     */
    get files() {
        return this.components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const candidate = this.node
            .findAll()
            .find((c) => (0, constructs_2.isComponent)(c) && isFile(c) && c.absolutePath === absolute);
        return candidate;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const candidate = this.tryFindFile(filePath);
        if (candidate) {
            candidate.node.scope?.node.tryRemoveChild(candidate.node.id);
            return candidate;
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        const pj = this._projenCommand ?? `npx projen@${common_1.PROJEN_VERSION}`;
        return `${pj} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all subprojects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this.components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        (0, cleanup_1.cleanup)(outdir, this.files.map((f) => (0, util_1.normalizePersistedPath)(f.path)), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this.components) {
            comp.synthesize();
        }
        if (!(0, util_1.isTruthy)(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this.components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                onlyFiles: true,
                followSymbolicLinks: false,
                absolute: true,
            });
            for (const file of files) {
                (0, fs_1.renameSync)(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.96.2" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType || (exports.ProjectType = ProjectType = {}));
/**
 * Resolves the project's output directory.
 */
function determineOutdir(parent, outdirOption) {
    if (parent && outdirOption && path.isAbsolute(outdirOption)) {
        throw new Error('"outdir" must be a relative path');
    }
    // if this is a subproject, it is relative to the parent
    if (parent) {
        if (!outdirOption) {
            throw new Error('"outdir" must be specified for subprojects');
        }
        return path.resolve(parent.outdir, outdirOption);
    }
    // if this is running inside a test and outdir is not explicitly set
    // use a temp directory (unless cwd is already under tmp)
    if (common_1.IS_TEST_RUN && !outdirOption) {
        const realCwd = (0, fs_1.realpathSync)(process.cwd());
        const realTmp = (0, fs_1.realpathSync)((0, os_1.tmpdir)());
        if (realCwd.startsWith(realTmp)) {
            return path.resolve(realCwd, outdirOption ?? DEFAULT_OUTDIR);
        }
        return (0, fs_1.mkdtempSync)(path.join((0, os_1.tmpdir)(), "projen."));
    }
    return path.resolve(outdirOption ?? DEFAULT_OUTDIR);
}
function isFile(c) {
    return c instanceof file_1.FileBase;
}
//# sourceMappingURL=data:application/json;base64,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