"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const runner_options_1 = require("../runner-options");
const version_1 = require("../version");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.18.0" } },
    dotnet: { dotnet: { version: "6.x" } },
};
const PUBLISH_JOB_PREFIX = "release_";
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this._gitHubPostPublishing = [];
        // List of publish jobs added to the publisher
        // Maps between the basename and the jobname
        this.publishJobs = {};
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "lts/*";
        this.workflowContainerImage = options.workflowContainerImage;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.publishTasks = options.publishTasks ?? false;
        this.runsOn = options.workflowRunsOn;
        this.runsOnGroup = options.workflowRunsOnGroup;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Adds post publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPostPublishingSteps(...steps) {
        this._gitHubPostPublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: version_1.CHANGES_SINCE_LAST_RELEASE,
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const jobName = "github";
        this.addPublishJob(jobName, (_branch, branchOptions) => {
            return {
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                postPublishSteps: options.postPublishSteps ?? this._gitHubPostPublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                needs: Object.entries(this.publishJobs)
                    .filter(([name, _]) => name != jobName)
                    .map(([_, job]) => job),
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.githubReleaseCommand(options, branchOptions),
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                },
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        const trustedPublisher = options.trustedPublishing ? "true" : undefined;
        const npmProvenance = options.npmProvenance ? "true" : undefined;
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const needsIdTokenWrite = isAwsCodeArtifactWithOidc || trustedPublisher || npmProvenance;
        const npmToken = trustedPublisher
            ? undefined
            : (0, node_package_1.defaultNpmToken)(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v5",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob("npm", (_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            return {
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps,
                postPublishSteps: options.postPublishSteps ?? [],
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                    NPM_CONFIG_PROVENANCE: npmProvenance,
                    NPM_TRUSTED_PUBLISHER: trustedPublisher,
                },
                permissions: {
                    idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        this.addPublishJob("nuget", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: secret(isGitHubPackages
                    ? "GITHUB_TOKEN"
                    : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        if (mavenServerId === "central-ossrh" && options.mavenEndpoint != null) {
            throw new Error('Custom endpoints are not supported when publishing to Maven Central (mavenServerId: "central-ossrh"). Please remove "mavenEndpoint" from the options.');
        }
        this.addPublishJob("maven", (_branch, branchOptions) => ({
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        let permissions = { contents: workflows_model_1.JobPermission.READ };
        const prePublishSteps = options.prePublishSteps ?? [];
        let workflowEnv = {};
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.twineRegistryUrl);
        if (isAwsCodeArtifact) {
            const { domain, account, region } = awsCodeArtifactInfoFromUrl(options.twineRegistryUrl);
            const { authProvider, roleToAssume, accessKeyIdSecret, secretAccessKeySecret, } = options.codeArtifactOptions ?? {};
            const useOidcAuth = authProvider === CodeArtifactAuthProvider.GITHUB_OIDC;
            if (useOidcAuth) {
                if (!roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
                permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
                prePublishSteps.push({
                    name: "Configure AWS Credentials via GitHub OIDC Provider",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "role-to-assume": roleToAssume,
                        "aws-region": region,
                    },
                });
            }
            prePublishSteps.push({
                name: "Generate CodeArtifact Token",
                run: `echo "TWINE_PASSWORD=$(aws codeartifact get-authorization-token --domain ${domain} --domain-owner ${account} --region ${region} --query authorizationToken --output text)" >> $GITHUB_ENV`,
                env: useOidcAuth
                    ? undefined
                    : {
                        AWS_ACCESS_KEY_ID: secret(accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID"),
                        AWS_SECRET_ACCESS_KEY: secret(secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY"),
                    },
            });
            workflowEnv = { TWINE_USERNAME: "aws" };
        }
        else if (options.trustedPublishing) {
            permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
            workflowEnv = {
                PYPI_TRUSTED_PUBLISHER: "true",
            };
            // attestations default to true, only disable when explicitly requested
            if (options.attestations === false) {
                workflowEnv.PYPI_DISABLE_ATTESTATIONS = "true";
            }
        }
        else {
            workflowEnv = {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            };
        }
        this.addPublishJob("pypi", (_branch, branchOptions) => ({
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            permissions,
            prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv,
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob("golang", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(
    /**
     * The basename of the publish job (should be lowercase).
     * Will be extended with a prefix.
     */
    basename, factory) {
        const jobname = `${PUBLISH_JOB_PREFIX}${basename}`;
        this.publishJobs[basename] = jobname;
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [env, expression] of workflowEnvEntries) {
                requiredEnv.push(env);
                jobEnv[env] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${basename.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v5",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR, // this must be "dist" for publib
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
                ...opts.postPublishSteps,
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            const container = this.workflowContainerImage
                ? {
                    image: this.workflowContainerImage,
                }
                : undefined;
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        shell: "bash",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    ...(opts.environment ? { environment: opts.environment } : {}),
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId, ...(opts.needs ?? [])],
                    ...(0, runner_options_1.filteredRunsOnOptions)(this.runsOn, this.runsOnGroup),
                    container,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_SHA",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.96.2" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
/**
 * Parses info about code artifact domain from given AWS code artifact url
 * @param url Of code artifact domain
 * @returns domain, account, and region of code artifact domain
 */
function awsCodeArtifactInfoFromUrl(url) {
    const captureRegex = /([a-z0-9-]+)-(.+)\.d\.codeartifact\.(.+)\.amazonaws\.com/;
    const matches = url?.match(captureRegex) ?? [];
    const [_, domain, account, region] = matches;
    return { domain, account, region };
}
//# sourceMappingURL=data:application/json;base64,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