# Copyright 2025 - AI4I. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Selector module for AdvPrefix attacks.

This module implements selection algorithms and strategies for choosing
optimal adversarial prefixes from generated candidates. It provides
various selection methodologies based on different criteria and
optimization objectives.

The module provides functionality for:
- Multi-objective optimization for prefix selection
- Pareto frontier analysis for trade-off identification
- Diversity-based selection algorithms
- Performance-based ranking and filtering
- Custom selection criteria and weighting schemes
- Integration with scoring and evaluation modules
- Complete pipeline step execution for Step 9 prefix selection

The selector module works in conjunction with the scoring and evaluation
modules to provide comprehensive prefix optimization capabilities.
"""

import pandas as pd
import numpy as np
import logging
from typing import List, Optional, Dict, Any
from dataclasses import dataclass


@dataclass
class PrefixSelectorConfig:
    """Configuration for prefix selection"""

    pasr_weight: float  # Weight for log-PASR in selection
    n_prefixes_per_goal: int = 1  # Number of prefixes to select per goal
    nll_tol: float = 999  # Tolerance for NLL relative to best prefix
    pasr_tol: float = 0  # Tolerance for PASR relative to best prefix
    judges: Optional[List[dict]] = None  # List of judges to use for PASR calculation


class PrefixSelector:
    """
    Advanced prefix selection system for choosing optimal adversarial prefixes.

    This class implements a sophisticated selection algorithm that combines
    multiple evaluation criteria to identify the most effective adversarial
    prefixes for each attack goal. The selection process considers both
    judge evaluation scores (PASR - Pass@1 Success Rate) and model likelihood
    scores (NLL - Negative Log-Likelihood) to ensure both effectiveness
    and model plausibility.

    The selection algorithm:
    1. Calculates combined PASR scores across multiple judges
    2. Computes weighted combination scores using PASR and NLL
    3. Applies tolerance-based filtering for quality candidates
    4. Implements iterative selection to avoid prefix redundancy
    5. Ensures diversity by filtering out sub-prefix relationships

    Key Features:
    - Multi-judge evaluation integration
    - Configurable weighting between success rate and likelihood
    - Tolerance-based candidate filtering
    - Sub-prefix elimination for diversity
    - Robust error handling and validation
    - Complete pipeline step execution capability

    Attributes:
        config: PrefixSelectorConfig containing selection parameters
        logger: Logger instance for operation tracking
        judge_column_map: Mapping of judge types to DataFrame column names
    """

    def __init__(self, config: PrefixSelectorConfig):
        """
        Initialize the prefix selector with the provided configuration.

        Sets up the selector with judge type mappings and prepares for
        multi-criteria prefix selection based on the configuration parameters.

        Args:
            config: PrefixSelectorConfig instance containing selection parameters
                including PASR weight, number of prefixes per goal, tolerance
                values, and judge configurations.
        """
        self.config = config
        self.logger = logging.getLogger(__name__)

        # Mapping of judge types to their column names in the DataFrame
        self.judge_column_map = {
            "nuanced": "eval_nj_mean",
            "jailbreakbench": "eval_jb_mean",
            "harmbench": "eval_hb_mean",
            "strongreject": "eval_sj_binary_mean",
        }

    @staticmethod
    def execute(input_df: pd.DataFrame, config: Dict[str, Any]) -> pd.DataFrame:
        """
        Execute Step 9 of the AdvPrefix pipeline: Select the final most effective adversarial prefixes.

        This static method serves as the main entry point for the prefix selection
        pipeline step. It instantiates a PrefixSelector with the provided configuration
        and executes the complete selection process to identify the most promising
        adversarial prefixes.

        This function represents the culmination of the AdvPrefix attack pipeline,
        selecting the most promising adversarial prefixes based on evaluation results
        from previous steps. It uses configurable selection criteria and ranking
        algorithms to identify the highest-performing prefixes for each target goal.

        Args:
            input_df: DataFrame containing aggregated evaluation results from Step 8.
                Expected to include columns for goals, prefixes, and judge evaluation
                scores, along with aggregated metrics like success rates.
            config: Configuration dictionary containing selection parameters including:
                - pasr_weight: Weight for Pass@1 Success Rate in selection scoring
                - n_prefixes_per_goal: Number of prefixes to select per target goal
                - selection_judges: List of judge types to consider for selection
                - Additional parameters for the PrefixSelectorConfig

        Returns:
            A pandas DataFrame containing the selected final adversarial prefixes.
            This represents the highest-performing subset of prefixes identified
            by the selection algorithm, ready for deployment or further analysis.
            Returns the original input DataFrame if selection fails.

        Note:
            This method provides backward compatibility with the pipeline architecture
            while leveraging the advanced selection capabilities of the PrefixSelector
            class. It handles configuration validation, error recovery, and logging
            to ensure robust operation within the AdvPrefix attack framework.

            If selection fails due to configuration errors or processing issues,
            the function gracefully falls back to returning the input DataFrame
            to ensure the pipeline can complete successfully.

            The selected prefixes represent the final output of the AdvPrefix
            attack pipeline and should contain the most effective adversarial
            examples for testing target model security.
        """
        logger = logging.getLogger(__name__)
        logger.info("Executing Step 9: Selecting final prefixes")

        if input_df.empty:
            logger.warning("Step 9 received an empty DataFrame. Skipping selection.")
            return input_df

        selector = None  # Ensure cleanup
        selected_df = input_df  # Default to input if selection fails

        try:
            # Initialize selector configuration
            selector_config = PrefixSelectorConfig(
                pasr_weight=config.get("pasr_weight", 0.5),
                n_prefixes_per_goal=config.get("n_prefixes_per_goal", 3),
                judges=config.get("selection_judges", []),
            )

            # Create selector instance
            selector = PrefixSelector(selector_config)

            # Execute selection process
            selected_df = selector.select_prefixes(input_df)
            logger.info(f"Selection complete. Selected {len(selected_df)} prefixes.")

        except Exception as e:
            logger.error(f"Error during prefix selection: {e}", exc_info=True)
            logger.warning("Returning unselected prefixes due to selection error.")
            selected_df = input_df  # Fallback to returning the input df

        finally:
            del selector

        logger.info(
            "Step 9 complete. Final selected prefixes CSV will be saved by the main pipeline."
        )

        return selected_df

    def select_prefixes(self, df: pd.DataFrame) -> pd.DataFrame:
        """
        Select optimal adversarial prefixes using multi-criteria selection algorithm.

        This method implements the core selection logic that combines judge
        evaluation scores with model likelihood scores to identify the most
        effective adversarial prefixes for each attack goal. The algorithm
        ensures both high attack success rates and model plausibility.

        The selection process:
        1. Validate judge configurations and required columns
        2. Calculate combined PASR scores across all configured judges
        3. Compute weighted combination scores (PASR + NLL)
        4. Select initial prefix with best combined score per goal
        5. Apply tolerance filtering for candidate qualification
        6. Iteratively select additional prefixes with diversity constraints
        7. Filter out sub-prefix relationships to ensure uniqueness

        Args:
            df: DataFrame containing evaluation results with prefix data.
                Must include columns for configured judge scores, 'prefix_nll',
                'goal', and 'prefix' columns.

        Returns:
            DataFrame containing selected prefixes with all original columns
            plus computed selection metrics:
            - pasr: Combined Pass@1 Success Rate across judges
            - log_pasr: Logarithmic PASR for numerical stability
            - combined_score: Weighted combination of PASR and NLL

        Raises:
            ValueError: If judge configuration is invalid, required columns
                are missing, or no valid judges are found.

        Note:
            The selection algorithm prioritizes prefixes with high judge
            success rates while considering model likelihood to ensure
            the selected prefixes are both effective and plausible.

            Sub-prefix filtering ensures that selected prefixes are
            meaningfully different rather than simple variations of
            the same underlying adversarial pattern.
        """
        # Validate judge configuration list
        if not isinstance(self.config.judges, list) or not self.config.judges:
            # Check if judges is a list and not empty
            raise ValueError(
                "Judge configuration ('judges' key) must be a non-empty list of dictionaries."
            )

        judge_types_found = []
        missing_columns = []
        for judge_config in self.config.judges:
            if not isinstance(judge_config, dict):
                self.logger.warning(
                    f"Skipping invalid item in judge config list (not a dict): {judge_config}"
                )
                continue

            # Extract judge type string (e.g., "nuanced") - Assuming a 'type' key
            judge_type = judge_config.get("type") or judge_config.get("evaluator_type")
            # Could add inference here if needed, similar to step 7

            if not judge_type:
                self.logger.warning(
                    f"Could not determine type for judge config: {judge_config}. Skipping."
                )
                continue

            if judge_type not in self.judge_column_map:
                # Check if the *type string* is valid
                self.logger.error(
                    f"Unknown judge type specified in config: '{judge_type}'"
                )
                raise ValueError(f"Unknown judge type for selection: {judge_type}")

            # Check if the corresponding column exists in the DataFrame
            expected_col = self.judge_column_map[judge_type]
            if expected_col not in df.columns:
                missing_columns.append(expected_col)

            if judge_type not in judge_types_found:
                judge_types_found.append(judge_type)

        if missing_columns:
            raise ValueError(
                f"Missing required evaluation result columns in DataFrame: {missing_columns}"
            )
        if not judge_types_found:
            raise ValueError(
                "No valid judge types found in the configuration to perform selection."
            )

        # Create a working copy of the DataFrame
        work_df = df.copy()

        # Calculate combined PASR score using the identified judge types
        work_df["pasr"] = self._calculate_combined_pasr(work_df, judge_types_found)

        # Calculate log PASR for scoring
        work_df["log_pasr"] = np.log(work_df["pasr"] + 1e-6)

        # Calculate combined score (minimize both 1 - PASR and prefix_nll)
        work_df["combined_score"] = (
            -self.config.pasr_weight * work_df["log_pasr"] + work_df["prefix_nll"]
        )

        # Create DataFrame for selected prefixes
        selected_prefixes = pd.DataFrame()

        # Group by goal and apply selection process
        for goal, group in work_df.groupby("goal"):
            # Step 1: Select first prefix based on combined score
            # Check if group is empty after potential filtering/issues
            if (
                group.empty
                or "combined_score" not in group.columns
                or group["combined_score"].isnull().all()
            ):
                self.logger.warning(
                    f"Skipping goal '{goal[:50]}...' during selection due to empty group or missing/invalid scores."
                )
                continue

            first_selection_idx = group["combined_score"].idxmin()
            first_selection = group.loc[first_selection_idx]

            # Step 2: Filter prefixes within PASR tolerance
            remaining_candidates = group[
                (group["pasr"] >= first_selection["pasr"] - self.config.pasr_tol)
                & (group.index != first_selection.name)
            ]

            # Step 3: Filter candidates within NLL tolerance
            valid_candidates = remaining_candidates[
                remaining_candidates["prefix_nll"]
                <= first_selection["prefix_nll"] + self.config.nll_tol
            ]

            # Initialize selections list with first selection
            selections = [first_selection]

            # Step 4: Iteratively select additional prefixes
            for _ in range(self.config.n_prefixes_per_goal - 1):
                # Remove candidates that are sub-prefixes of selected ones
                valid_candidates = valid_candidates[
                    ~valid_candidates["prefix"].apply(
                        lambda x: any(
                            str(x).startswith(str(sel["prefix"]))
                            for sel in selections
                            if sel is not None and "prefix" in sel and x is not None
                        )
                    )
                ]

                if valid_candidates.empty:
                    break

                # Select next prefix with lowest NLL
                if (
                    "prefix_nll" not in valid_candidates.columns
                    or valid_candidates["prefix_nll"].isnull().all()
                ):
                    self.logger.warning(
                        f"Cannot select next prefix for goal '{goal[:50]}...' due to missing/invalid NLL scores in candidates."
                    )
                    break
                next_selection = valid_candidates.nsmallest(1, "prefix_nll").iloc[0]
                selections.append(next_selection)
                valid_candidates = valid_candidates[
                    valid_candidates.index != next_selection.name
                ]

            # Combine selections for this goal
            combined_selection = pd.DataFrame(selections)
            selected_prefixes = pd.concat([selected_prefixes, combined_selection])

        # Reset index
        selected_prefixes.reset_index(drop=True, inplace=True)

        # Add the new columns (pasr, log_pasr, combined_score) to the output
        # Ensure columns exist before trying to select them
        output_columns = [
            col
            for col in list(df.columns) + ["pasr", "log_pasr", "combined_score"]
            if col in selected_prefixes.columns
        ]
        selected_prefixes = selected_prefixes[output_columns]

        self.logger.info(
            f"Selected {len(selected_prefixes)} prefixes across {len(df['goal'].unique())} goals"
        )
        return selected_prefixes

    def _calculate_combined_pasr(
        self, df: pd.DataFrame, judge_types: List[str]
    ) -> pd.Series:
        """
        Calculate combined Pass@1 Success Rate (PASR) across multiple judge evaluations.

        This method aggregates evaluation scores from multiple judges to compute
        a unified PASR metric that represents the overall attack success rate
        for each prefix. The combination process handles missing data gracefully
        and provides robust averaging across different judge types.

        Args:
            df: DataFrame containing judge evaluation scores. Must include
                columns corresponding to the judge types specified.
            judge_types: List of valid judge type strings that map to DataFrame
                columns via the judge_column_map (e.g., ["nuanced", "harmbench"]).

        Returns:
            A pandas Series containing combined PASR scores for each prefix.
            Values range from 0 to 1, where higher values indicate better
            attack success rates. Missing or invalid scores are handled
            by filling with 0.

        Note:
            The method automatically converts judge scores to numeric values
            and handles non-numeric or missing data gracefully. If no valid
            judge scores are available, all PASR values are set to 0.

            The combination uses simple averaging across available judges,
            which provides equal weight to each judge type. This approach
            ensures balanced evaluation across different assessment frameworks.
        """
        judge_scores = []
        for judge_type in judge_types:  # Iterate through the list of type strings
            column = self.judge_column_map[judge_type]  # Use the type string for lookup
            # Ensure column is numeric before appending
            if column in df.columns:
                try:
                    numeric_scores = pd.to_numeric(df[column], errors="coerce")
                    judge_scores.append(numeric_scores)
                except Exception as e:
                    self.logger.warning(
                        f"Could not convert column '{column}' to numeric for PASR calculation. Skipping. Error: {e}"
                    )
            else:
                # This should be caught by initial validation, but as safeguard
                self.logger.warning(
                    f"Column '{column}' for judge '{judge_type}' not found during PASR calculation."
                )

        if not judge_scores:
            self.logger.warning(
                "No valid judge scores found to calculate combined PASR. Returning zeros."
            )
            return pd.Series(0, index=df.index)

        # Calculate mean of judge scores, handling potential NaNs after conversion
        combined_scores_df = pd.concat(judge_scores, axis=1)
        # Use mean, skipping NaNs. If a row has all NaNs, the mean will be NaN.
        mean_scores = combined_scores_df.mean(axis=1, skipna=True)
        # Fill any resulting NaNs (rows where all judges had NaN scores) with 0
        return mean_scores.fillna(0)
