from typing import Callable, Optional

from chalk._lsp.error_builder import get_resolver_error_builder
from chalk.features import DataFrame
from chalk.features.feature_field import Feature
from chalk.features.feature_set import Features
from chalk.features.resolver import RESOLVER_REGISTRY, OnlineResolver, ResourceHint
from chalk.features.underscore import Underscore
from chalk.ml.model_reference import MODEL_REFERENCE_REGISTRY
from chalk.ml.model_version import ModelVersion
from chalk.utils.collections import ensure_tuple


def build_inference_function(
    model_version: ModelVersion, pkey: Feature, output_feature: Optional[Feature] = None, text_model: bool = False
) -> Callable[[DataFrame], DataFrame]:
    """Build the core inference function that takes a DataFrame and returns predictions.

    This is the function that gets called by the inference resolver. It excludes the primary key
    from the input, makes predictions, and optionally adds them back to the DataFrame.

    Parameters
    ----------
    model_version
        The model version to use for prediction
    pkey
        The primary key feature to exclude from predictions
    output_feature
        Optional output feature to add predictions to the DataFrame. If None, just returns raw predictions.

    Returns
    -------
    Callable[[DataFrame], DataFrame]
        Function that takes a DataFrame and returns predictions
    """
    pkey_string = str(pkey)
    if text_model:
        # For text models, we expect a single column of string data
        def fn(inp: DataFrame):
            texts = inp[[c for c in inp.columns if c != pkey_string]].to_pyarrow().column(0).to_numpy()
            result = model_version.predict(texts)
            if output_feature is not None:
                return inp[pkey_string].with_columns({output_feature: result})
            return result

        return fn

    def fn(inp: DataFrame):
        # Convert features to PyArrow table, excluding primary key
        arr = inp[[c for c in inp.columns if c != pkey_string]].to_pyarrow().__array__()

        result = model_version.predict(arr)
        if output_feature is not None:
            return inp[pkey_string].with_columns({output_feature: result})
        return result

    return fn


def generate_inference_resolver(
    inputs: list[Underscore], model_version: ModelVersion, resource_hint: Optional[ResourceHint] = None
) -> Feature:
    output_feature = Feature()
    previous_hook = output_feature.hook

    def hook(features: type[Features]) -> None:
        if previous_hook:
            previous_hook(features)

        pkey = features.__chalk_primary__
        if pkey is None:
            raise ValueError(f"Feature class {features} does not have a primary key defined")

        def resolver_factory():
            # Use the extracted build_inference_function

            cleaned_inputs = []
            for i in inputs:
                try:
                    cleaned_inputs.append(Feature.from_root_fqn(output_feature.namespace + str(i)[1:]))
                except Exception as e:
                    raise ValueError(f"Could not find feature for input {i}: {e}")

            try:
                text_model = len(cleaned_inputs) == 1 and cleaned_inputs[0].typ.parsed_annotation is str
            except Exception:
                # Fallback in case of any issues determining the type
                text_model = False

            fn = build_inference_function(model_version, pkey, output_feature, text_model=text_model)

            identifier = model_version.identifier or ""
            model_reference = MODEL_REFERENCE_REGISTRY.get((model_version.name, identifier), None)
            if model_reference is not None:
                model_reference.relations.append(([i.fqn for i in cleaned_inputs], output_feature.fqn))

            return OnlineResolver(
                function_definition="",
                filename="",
                fqn=f"{model_version.name}__{output_feature.namespace}_{output_feature.name}",
                doc=None,
                inputs=[DataFrame[[pkey, *ensure_tuple(cleaned_inputs)]]],
                state=None,
                output=Features[DataFrame[output_feature, pkey]],
                fn=fn,
                environment=None,
                machine_type=None,
                default_args=[None],
                timeout=None,
                cron=None,
                when=None,
                tags=None,
                owner=None,
                resource_hint=resource_hint,
                data_sources=None,
                is_sql_file_resolver=False,
                source_line=None,
                lsp_builder=get_resolver_error_builder(fn),
                parse=None,
                static=False,
                total=False,
                autogenerated=False,
                unique_on=None,
                partitioned_by=None,
                data_lineage=None,
                sql_settings=None,
            )

        RESOLVER_REGISTRY.add_to_deferred_registry(resolver_factory, override=False)

    output_feature.hook = hook

    return output_feature
