"""This module contains methods to featurize molecules."""

import numpy as np
from rdkit.Chem import rdmolops
from rdkit import Chem
from rdkit import DataStructs
from rdkit.Chem import rdFingerprintGenerator
from rdkit.Chem.Fingerprints import FingerprintMols
from rdkit.Chem import MACCSkeys
from rdkit.Chem.AtomPairs import Pairs, Torsions
from rdkit.DataStructs import cDataStructs
from aimsim.utils.ccbmlib_fingerprints import generate_fingerprints
from padelpy import from_smiles
from mhfp.encoder import MHFPEncoder
from mordred import Calculator, descriptors


from ..exceptions import (
    InvalidConfigurationError,
    NotInitializedError,
    MordredCalculatorError,
)


class Descriptor:
    """Class for descriptors.

    Attributes:
        label_ (str): Label used to denote the type of descriptor being used.
        numpy_ (np.ndarray): Value of the descriptor in the numpy format.
        rdkit_ (rdkit.DataStructs.cDataStructs.UIntSparseIntVec):  Value of
            the descriptor in the rdkit format.

    Methods:
        to_numpy: Get the numpy_ attribute. If it does not exist,
            it is created.
        to_rdkit: Get the rdkit_ attribute. If it does not exist,
            it is created.
        check_init: check if the Descriptor object is initialized. This is
            done by checking the existence of the numpy_ or rdkit_ attribute.


    """

    def __init__(self, value=None):
        if value is not None:
            self.set_manually(arbitrary_descriptor_val=value)

    def to_numpy(self):
        """Return numpy_ attribute of Descriptor.
        Attribute will be initialized if not present.

        Returns:
            (np.array): Numpy vector of descriptor.

        """
        if self.check_init() is False:
            raise NotInitializedError("Descriptor value not generated. Use " "make_fingerprint() to initialize it.")
        if not hasattr(self, "numpy_"):
            self.numpy_ = np.zeros((0,), dtype=np.int8)
            DataStructs.ConvertToNumpyArray(self.rdkit_, self.numpy_)
        self.numpy_ = self.numpy_.ravel()
        return self.numpy_

    def to_rdkit(self):
        """Return rdkit_ attribute of Descriptor.

        Returns:
            (DataStructs.ExplicitBitVect): Fingerprint value as a bit vector.

        Raises:
            (NotInitializedError): If object not initialized with a fingerprint.
            (ValueError): If only arbitrary numpy descriptor is used to
                initialize the object. This cannot be converted to bit vectors.

        """
        if self.check_init() is False:
            raise NotInitializedError("Descriptor value not generated. Use " "make_fingerprint() to initialize it.")
        if not hasattr(self, "rdkit_"):
            raise ValueError("Attempting to convert arbitrary numpy array " "to rdkit bit vector is not supported")
        return self.rdkit_

    def check_init(self):
        """Check initialization status of the Descriptor object.

        Returns:
            (bool): True if object is initialized.

        """
        return getattr(self, "numpy_", None) is not None or getattr(self, "rdkit_", None) is not None

    def _set_morgan_fingerprint(self, molecule_graph, radius, n_bits, **kwargs):
        """Set the descriptor to a morgan fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.
            radius (int): Radius of fingerprint, 3 corresponds to ECFP6.
            n_bits (int):Number of bits to use if Morgan Fingerprint wanted as
                a bit vector. If set to None, Morgan fingerprint returned
                as count.

        """
        fpg = rdFingerprintGenerator.GetMorganGenerator(radius=radius, fpSize=n_bits, **kwargs)
        self.rdkit_ = fpg.GetFingerprint(molecule_graph)
        self.label_ = "morgan_fingerprint"
        self.params_ = {"radius": radius, "n_bits": n_bits}

    def _set_rdkit_topological_fingerprint(self, molecule_graph, min_path, max_path, **kwargs):
        """Set the descriptor to a topological fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.
            min_path (int): Minimum path used to generate the
                topological fingerprint.
            max_path (int): Maximum path used to generate the topological
                fingerprint.

        """
        if molecule_graph.GetNumAtoms() <= min_path:
            raise InvalidConfigurationError(
                f"# atoms in molecule: "
                f"{molecule_graph.GetNumAtoms()}, "
                f"min_path: {min_path}. "
                f"For topological fingerprint, "
                f"the number of atoms has to be "
                f"greater than the minimum path "
                f"used for fingerprint."
            )
        fpg = rdFingerprintGenerator.GetRDKitFPGenerator(minPath=min_path, maxPath=max_path, **kwargs)
        self.rdkit_ = fpg.GetFingerprint(molecule_graph)
        self.label_ = "topological_fingerprint"
        self.params_ = {"min_path": min_path, "max_path": max_path}

    def _set_padelpy_descriptor(self, molecule_graph, descriptor, **kwargs):
        """Set the value of numpy_ to the descriptor as indicated by descriptor.

        Args:
            molecule_graph (RDKit object): Graph of the molecule of interest.
            descriptor (string): Name of descriptor, as implemented in PaDEL.

        """
        try:
            local_smi = Chem.MolToSmiles(molecule_graph)
            all_desc = from_smiles(local_smi, timeout=kwargs.get("timeout", 60))
            self.numpy_ = np.array(all_desc[descriptor])
            self.label_ = descriptor
        except Exception as e:
            raise RuntimeError(f"Unable to retrieve PaDELPy Descriptor '{descriptor}'. Is it a valid descriptor?")

    def _set_mordred_descriptor(self, molecule_graph, descriptor, **kwargs):
        """Set the value of numpy_ to the descriptor as indicated by descriptor.

        Args:
            molecule_graph (RDKit object): Graph of the molecule of interest.
            descriptor (string): Name of descriptor, as implemented in Mordred.

        Raises:
            MordredCalculatorError: If Morded is unable to calculate a value
                for the indicated descriptor, this exception will be raised.
        """
        try:
            calc = Calculator(descriptors, ignore_3D=False)
            res = calc(molecule_graph)
            res.drop_missing()
            self.numpy_ = np.array(res[descriptor])
            self.label_ = descriptor
        except KeyError:
            raise MordredCalculatorError(
                """Mordred descriptor calculator unable to calculate descriptor \"{}\",
                ensure correct name is used (https://jacksonburns.github.io/mordred-community/descriptors.html).""".format(
                    descriptor
                )
            )

    def _set_daylight_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to a daylight fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns an ExplicitBitVect
        self.rdkit_ = FingerprintMols.FingerprintMol(molecule_graph, **kwargs)
        self.label_ = "daylight_fingerprint"
        self.params_ = {}

    def _set_maccs_keys(self, molecule_graph, **kwargs):
        """Set the descriptor to MACCS keys.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns an ExplicitBitVect
        self.rdkit_ = MACCSkeys.GenMACCSKeys(molecule_graph)
        self.label_ = "maccs_keys"
        self.params_ = {}

    def _set_atom_pair_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to an atom-pair fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns a SparseBitVect
        fp = Pairs.GetAtomPairFingerprintAsBitVect(molecule_graph)
        fp2 = cDataStructs.ConvertToExplicit(fp)
        self.rdkit_ = fp2
        self.label_ = "atom-pair_fingerprint"
        self.params_ = {}

    def _set_torsion_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to a torsion fingerprint.

        Args:
            molecule_graph (RDKIT object): Graph of molecule to be
                fingerprinted.

        """
        # returns a long sparse int vector
        fp = Torsions.GetTopologicalTorsionFingerprintAsIntVect(molecule_graph)
        self.rdkit_ = fp
        self.label_ = "torsion_fingerprint"
        self.params_ = {}

    def _set_ccbmlib_fingerprint(self, molecule_graph, descriptor, **kwargs):
        """Set the descriptor to fingerprint from ccbmlib.

        Args:
        molecule_graph (RDKIT object): Graph of molecule to be fingerprinted.

        """
        # returns a list of ints which represent the on bits (features)
        fp = generate_fingerprints(molecule_graph, descriptor, **kwargs)
        self.numpy_ = fp
        self.label_ = descriptor
        self.params_ = {}

    def _set_minhash_fingerprint(self, molecule_graph, **kwargs):
        """Set the descriptor to the minhash fingerprint.

        Args:
        molecule_graph (RDKIT object): Graph of molecule to be fingerprinted.

        """
        mhfp_encoder = MHFPEncoder(
            n_permutations=kwargs["n_permutations"],
            seed=kwargs["seed"],
        )
        fp = mhfp_encoder.encode_mol(
            molecule_graph,
            radius=kwargs["radius"],
            rings=kwargs["rings"],
            kekulize=kwargs["kekulize"],
        )
        self.numpy_ = fp
        self.label_ = "minhash_fingerprint"
        self.params_ = kwargs

    def make_fingerprint(self, molecule_graph, fingerprint_type, fingerprint_params=None):
        """Make fingerprint of a molecule based on a graph representation.
        Set the state of the descriptor to this fingerprint.

        Args:
            molecule_graph (RDKIT object): The graph object used to make a
                fingerprint.
            fingerprint_type (str): label for the type of fingerprint.
                Invokes get_supported_descriptors()['fingerprints']
                for list of supported fingerprints.
            fingerprint_params (dict): Keyword arguments used to modify
                parameters of fingerprint. Default is None.
        """
        if fingerprint_params is None:
            fingerprint_params = {}
        if fingerprint_type == "morgan_fingerprint":
            morgan_params = {"radius": 3, "n_bits": 1024}
            morgan_params.update(fingerprint_params)
            self._set_morgan_fingerprint(molecule_graph=molecule_graph, **morgan_params)
        elif fingerprint_type == "minhash_fingerprint":
            minhash_params = {
                "n_permutations": 2048,
                "seed": 42,
                "radius": 3,
                "rings": True,
                "kekulize": True,
            }
            minhash_params.update(fingerprint_params)
            self._set_minhash_fingerprint(molecule_graph=molecule_graph, **minhash_params)
        elif fingerprint_type == "topological_fingerprint":
            topological_params = {"min_path": 1, "max_path": 7}
            topological_params.update(fingerprint_params)
            self._set_rdkit_topological_fingerprint(molecule_graph=molecule_graph, **topological_params)
        elif fingerprint_type == "daylight_fingerprint":
            daylight_params = {
                "minPath": 1,
                "maxPath": 7,
                "fpSize": 2048,
                "bitsPerHash": 2,
                "useHs": 0,
                "tgtDensity": 0.3,
                "minSize": 64,
            }
            daylight_params.update(fingerprint_params)
            self._set_daylight_fingerprint(molecule_graph=molecule_graph, **daylight_params)
        elif fingerprint_type == "maccs_keys":
            maccs_params = {}
            maccs_params.update(fingerprint_params)
            self._set_maccs_keys(molecule_graph=molecule_graph, **maccs_params)
        elif fingerprint_type == "atom-pair_fingerprint":
            atom_pair_fp_params = {}
            atom_pair_fp_params.update(fingerprint_params)
            self._set_atom_pair_fingerprint(molecule_graph=molecule_graph, **atom_pair_fp_params)
        elif fingerprint_type == "torsion_fingerprint":
            torsion_params = {}
            torsion_params.update(fingerprint_params)
            self._set_torsion_fingerprint(molecule_graph=molecule_graph, **torsion_params)
        elif fingerprint_type.split(":")[0] == "mordred":
            mordred_params = {}
            self._set_mordred_descriptor(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **mordred_params,
            )
        elif fingerprint_type.split(":")[0] == "padelpy":
            padelpy_params = {}
            padelpy_params.update(fingerprint_params)
            self._set_padelpy_descriptor(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **padelpy_params,
            )
        elif fingerprint_type.split(":")[0] == "ccbmlib":
            ccbmlib_params = {}
            self._set_ccbmlib_fingerprint(
                molecule_graph=molecule_graph,
                descriptor=fingerprint_type.split(":")[1],
                **ccbmlib_params,
            )
        else:
            raise ValueError(f"{fingerprint_type} not supported")

    def set_manually(self, arbitrary_descriptor_val):
        """Set the descriptor value manually based on user specified value.

        Args:
            arbitrary_descriptor_val (np.ndarray or list): Vectorized
                representation of descriptor values.
        """
        self.label_ = "arbitrary"
        self.numpy_ = np.array(arbitrary_descriptor_val)

    def get_label(self):
        if not self.check_init():
            raise NotInitializedError
        else:
            return self.label_

    def get_params(self):
        if not self.check_init():
            raise NotInitializedError
        else:
            return self.params_

    def is_fingerprint(self):
        return "fingerprint" in self.get_label()

    def get_folded_fprint(self, fold_to_length):
        """
        Get the folded value of a fingerprint to a specified length.
        Args:
            fold_to_length (int): Number of bits to fold to.

        Returns:
            (np.ndarray): Folded fingerprint.
        """
        if not self.is_fingerprint():
            raise ValueError("Can only fold fingerprints")
        fingerprint = self.to_numpy()
        if len(fingerprint) < fold_to_length:
            raise InvalidConfigurationError(f"Cannot fold fingerprint of " f"length {len(fingerprint)}to a " f"higher length {fold_to_length}")
        n_folds = np.log2(len(fingerprint) / fold_to_length)
        if n_folds - int(n_folds) > 0.0:
            raise InvalidConfigurationError(
                f"Fingerprint length " f"{len(fingerprint)} not " f"a 2-multiple of required " f"folded length {fold_to_length}"
            )
        for _ in range(int(n_folds)):
            mid_point = int(len(fingerprint) / 2)
            assert mid_point - (len(fingerprint) / 2) == 0.0
            fingerprint = fingerprint[:mid_point] | fingerprint[mid_point:]
        assert len(fingerprint) == fold_to_length
        return fingerprint

    @staticmethod
    def shorten_label(label):
        """Shorten the label of a fingerprint. Useful for plotting purposes.

        Args:
            label (str): Label of fingerprint to shorten.

        Returns:
            (str): Shortened label.

        Raises:
            InvalidConfigurationError: if label not in
                get_supported_descriptors()

        Currently implemented shortening strategies:
            1. Fingerprints: remove '_fingerprint' from the label

        """
        if label not in Descriptor.get_all_supported_descriptors():
            raise InvalidConfigurationError(f"{label} not a " f"supported descriptor")
        if label in Descriptor.get_supported_fprints():
            return label.replace("_fingerprint", "")
        return label

    @staticmethod
    def fold_to_equal_length(fingerprint1, fingerprint2):
        """
        Get back two fingerprint arrays of equal length. The longer fingerprint
        is folded to the size of the smaller one.

        Args:
            fingerprint1 (Descriptor): Fingerprint one.
            fingerprint2 (Descriptor): Fingerprint two.

        Returns:
            (np.ndarray, np.ndarray)

        """
        fprint1_arr = fingerprint1.to_numpy()
        fprint2_arr = fingerprint2.to_numpy()
        if len(fprint1_arr) > len(fprint2_arr):
            return (
                fingerprint1.get_folded_fprint(fold_to_length=len(fprint2_arr)),
                fprint2_arr,
            )
        else:
            return fprint1_arr, fingerprint2.get_folded_fprint(fold_to_length=len(fprint1_arr))

    @staticmethod
    def get_supported_fprints():
        """Return a list of strings for the currently implemented
        molecular fingerprints.
        Returns:
            List: List of strings.
        """
        return [
            "morgan_fingerprint",
            "topological_fingerprint",
            "daylight_fingerprint",
        ]

    @staticmethod
    def get_all_supported_descriptors():
        """Returns a list of descriptors which _can_ be used with AIMSim
        but are considered experimental or are complex/rarely used and
        are therefore excluded from the UI.

        Returns:
            List: List of strings of all supported descriptors.
        """
        return [
            "morgan_fingerprint",
            "topological_fingerprint",
            "daylight_fingerprint",
            "minhash_fingerprint",
            "maccs_keys",
            "atom-pair_fingerprint",
            "torsion_fingerprint",
            "ccbmlib:atom_pairs",
            "ccbmlib:hashed_atom_pairs",
            "ccbmlib:avalon",
            "ccbmlib:maccs_keys",
            "ccbmlib:morgan",
            "ccbmlib:hashed_morgan",
            "ccbmlib:rdkit_fingerprint",
            "ccbmlib:torsions",
            "ccbmlib:hashed_torsions",
            "padelpy:nAcid",
            "padelpy:ALogP",
            "padelpy:ALogp2",
            "padelpy:AMR",
            "padelpy:apol",
            "padelpy:naAromAtom",
            "padelpy:nAromBond",
            "padelpy:nAtom",
            "padelpy:nHeavyAtom",
            "padelpy:nH",
            "padelpy:nB",
            "padelpy:nC",
            "padelpy:nN",
            "padelpy:nO",
            "padelpy:nS",
            "padelpy:nP",
            "padelpy:nF",
            "padelpy:nCl",
            "padelpy:nBr",
            "padelpy:nI",
            "padelpy:nX",
            "padelpy:ATS0m",
            "padelpy:ATS1m",
            "padelpy:ATS2m",
            "padelpy:ATS3m",
            "padelpy:ATS4m",
            "padelpy:ATS5m",
            "padelpy:ATS6m",
            "padelpy:ATS7m",
            "padelpy:ATS8m",
            "padelpy:ATS0v",
            "padelpy:ATS1v",
            "padelpy:ATS2v",
            "padelpy:ATS3v",
            "padelpy:ATS4v",
            "padelpy:ATS5v",
            "padelpy:ATS6v",
            "padelpy:ATS7v",
            "padelpy:ATS8v",
            "padelpy:ATS0e",
            "padelpy:ATS1e",
            "padelpy:ATS2e",
            "padelpy:ATS3e",
            "padelpy:ATS4e",
            "padelpy:ATS5e",
            "padelpy:ATS6e",
            "padelpy:ATS7e",
            "padelpy:ATS8e",
            "padelpy:ATS0p",
            "padelpy:ATS1p",
            "padelpy:ATS2p",
            "padelpy:ATS3p",
            "padelpy:ATS4p",
            "padelpy:ATS5p",
            "padelpy:ATS6p",
            "padelpy:ATS7p",
            "padelpy:ATS8p",
            "padelpy:ATS0i",
            "padelpy:ATS1i",
            "padelpy:ATS2i",
            "padelpy:ATS3i",
            "padelpy:ATS4i",
            "padelpy:ATS5i",
            "padelpy:ATS6i",
            "padelpy:ATS7i",
            "padelpy:ATS8i",
            "padelpy:ATS0s",
            "padelpy:ATS1s",
            "padelpy:ATS2s",
            "padelpy:ATS3s",
            "padelpy:ATS4s",
            "padelpy:ATS5s",
            "padelpy:ATS6s",
            "padelpy:ATS7s",
            "padelpy:ATS8s",
            "padelpy:AATS0m",
            "padelpy:AATS1m",
            "padelpy:AATS2m",
            "padelpy:AATS3m",
            "padelpy:AATS4m",
            "padelpy:AATS5m",
            "padelpy:AATS6m",
            "padelpy:AATS7m",
            "padelpy:AATS8m",
            "padelpy:AATS0v",
            "padelpy:AATS1v",
            "padelpy:AATS2v",
            "padelpy:AATS3v",
            "padelpy:AATS4v",
            "padelpy:AATS5v",
            "padelpy:AATS6v",
            "padelpy:AATS7v",
            "padelpy:AATS8v",
            "padelpy:AATS0e",
            "padelpy:AATS1e",
            "padelpy:AATS2e",
            "padelpy:AATS3e",
            "padelpy:AATS4e",
            "padelpy:AATS5e",
            "padelpy:AATS6e",
            "padelpy:AATS7e",
            "padelpy:AATS8e",
            "padelpy:AATS0p",
            "padelpy:AATS1p",
            "padelpy:AATS2p",
            "padelpy:AATS3p",
            "padelpy:AATS4p",
            "padelpy:AATS5p",
            "padelpy:AATS6p",
            "padelpy:AATS7p",
            "padelpy:AATS8p",
            "padelpy:AATS0i",
            "padelpy:AATS1i",
            "padelpy:AATS2i",
            "padelpy:AATS3i",
            "padelpy:AATS4i",
            "padelpy:AATS5i",
            "padelpy:AATS6i",
            "padelpy:AATS7i",
            "padelpy:AATS8i",
            "padelpy:AATS0s",
            "padelpy:AATS1s",
            "padelpy:AATS2s",
            "padelpy:AATS3s",
            "padelpy:AATS4s",
            "padelpy:AATS5s",
            "padelpy:AATS6s",
            "padelpy:AATS7s",
            "padelpy:AATS8s",
            "padelpy:ATSC0c",
            "padelpy:ATSC1c",
            "padelpy:ATSC2c",
            "padelpy:ATSC3c",
            "padelpy:ATSC4c",
            "padelpy:ATSC5c",
            "padelpy:ATSC6c",
            "padelpy:ATSC7c",
            "padelpy:ATSC8c",
            "padelpy:ATSC0m",
            "padelpy:ATSC1m",
            "padelpy:ATSC2m",
            "padelpy:ATSC3m",
            "padelpy:ATSC4m",
            "padelpy:ATSC5m",
            "padelpy:ATSC6m",
            "padelpy:ATSC7m",
            "padelpy:ATSC8m",
            "padelpy:ATSC0v",
            "padelpy:ATSC1v",
            "padelpy:ATSC2v",
            "padelpy:ATSC3v",
            "padelpy:ATSC4v",
            "padelpy:ATSC5v",
            "padelpy:ATSC6v",
            "padelpy:ATSC7v",
            "padelpy:ATSC8v",
            "padelpy:ATSC0e",
            "padelpy:ATSC1e",
            "padelpy:ATSC2e",
            "padelpy:ATSC3e",
            "padelpy:ATSC4e",
            "padelpy:ATSC5e",
            "padelpy:ATSC6e",
            "padelpy:ATSC7e",
            "padelpy:ATSC8e",
            "padelpy:ATSC0p",
            "padelpy:ATSC1p",
            "padelpy:ATSC2p",
            "padelpy:ATSC3p",
            "padelpy:ATSC4p",
            "padelpy:ATSC5p",
            "padelpy:ATSC6p",
            "padelpy:ATSC7p",
            "padelpy:ATSC8p",
            "padelpy:ATSC0i",
            "padelpy:ATSC1i",
            "padelpy:ATSC2i",
            "padelpy:ATSC3i",
            "padelpy:ATSC4i",
            "padelpy:ATSC5i",
            "padelpy:ATSC6i",
            "padelpy:ATSC7i",
            "padelpy:ATSC8i",
            "padelpy:ATSC0s",
            "padelpy:ATSC1s",
            "padelpy:ATSC2s",
            "padelpy:ATSC3s",
            "padelpy:ATSC4s",
            "padelpy:ATSC5s",
            "padelpy:ATSC6s",
            "padelpy:ATSC7s",
            "padelpy:ATSC8s",
            "padelpy:AATSC0c",
            "padelpy:AATSC1c",
            "padelpy:AATSC2c",
            "padelpy:AATSC3c",
            "padelpy:AATSC4c",
            "padelpy:AATSC5c",
            "padelpy:AATSC6c",
            "padelpy:AATSC7c",
            "padelpy:AATSC8c",
            "padelpy:AATSC0m",
            "padelpy:AATSC1m",
            "padelpy:AATSC2m",
            "padelpy:AATSC3m",
            "padelpy:AATSC4m",
            "padelpy:AATSC5m",
            "padelpy:AATSC6m",
            "padelpy:AATSC7m",
            "padelpy:AATSC8m",
            "padelpy:AATSC0v",
            "padelpy:AATSC1v",
            "padelpy:AATSC2v",
            "padelpy:AATSC3v",
            "padelpy:AATSC4v",
            "padelpy:AATSC5v",
            "padelpy:AATSC6v",
            "padelpy:AATSC7v",
            "padelpy:AATSC8v",
            "padelpy:AATSC0e",
            "padelpy:AATSC1e",
            "padelpy:AATSC2e",
            "padelpy:AATSC3e",
            "padelpy:AATSC4e",
            "padelpy:AATSC5e",
            "padelpy:AATSC6e",
            "padelpy:AATSC7e",
            "padelpy:AATSC8e",
            "padelpy:AATSC0p",
            "padelpy:AATSC1p",
            "padelpy:AATSC2p",
            "padelpy:AATSC3p",
            "padelpy:AATSC4p",
            "padelpy:AATSC5p",
            "padelpy:AATSC6p",
            "padelpy:AATSC7p",
            "padelpy:AATSC8p",
            "padelpy:AATSC0i",
            "padelpy:AATSC1i",
            "padelpy:AATSC2i",
            "padelpy:AATSC3i",
            "padelpy:AATSC4i",
            "padelpy:AATSC5i",
            "padelpy:AATSC6i",
            "padelpy:AATSC7i",
            "padelpy:AATSC8i",
            "padelpy:AATSC0s",
            "padelpy:AATSC1s",
            "padelpy:AATSC2s",
            "padelpy:AATSC3s",
            "padelpy:AATSC4s",
            "padelpy:AATSC5s",
            "padelpy:AATSC6s",
            "padelpy:AATSC7s",
            "padelpy:AATSC8s",
            "padelpy:MATS1c",
            "padelpy:MATS2c",
            "padelpy:MATS3c",
            "padelpy:MATS4c",
            "padelpy:MATS5c",
            "padelpy:MATS6c",
            "padelpy:MATS7c",
            "padelpy:MATS8c",
            "padelpy:MATS1m",
            "padelpy:MATS2m",
            "padelpy:MATS3m",
            "padelpy:MATS4m",
            "padelpy:MATS5m",
            "padelpy:MATS6m",
            "padelpy:MATS7m",
            "padelpy:MATS8m",
            "padelpy:MATS1v",
            "padelpy:MATS2v",
            "padelpy:MATS3v",
            "padelpy:MATS4v",
            "padelpy:MATS5v",
            "padelpy:MATS6v",
            "padelpy:MATS7v",
            "padelpy:MATS8v",
            "padelpy:MATS1e",
            "padelpy:MATS2e",
            "padelpy:MATS3e",
            "padelpy:MATS4e",
            "padelpy:MATS5e",
            "padelpy:MATS6e",
            "padelpy:MATS7e",
            "padelpy:MATS8e",
            "padelpy:MATS1p",
            "padelpy:MATS2p",
            "padelpy:MATS3p",
            "padelpy:MATS4p",
            "padelpy:MATS5p",
            "padelpy:MATS6p",
            "padelpy:MATS7p",
            "padelpy:MATS8p",
            "padelpy:MATS1i",
            "padelpy:MATS2i",
            "padelpy:MATS3i",
            "padelpy:MATS4i",
            "padelpy:MATS5i",
            "padelpy:MATS6i",
            "padelpy:MATS7i",
            "padelpy:MATS8i",
            "padelpy:MATS1s",
            "padelpy:MATS2s",
            "padelpy:MATS3s",
            "padelpy:MATS4s",
            "padelpy:MATS5s",
            "padelpy:MATS6s",
            "padelpy:MATS7s",
            "padelpy:MATS8s",
            "padelpy:GATS1c",
            "padelpy:GATS2c",
            "padelpy:GATS3c",
            "padelpy:GATS4c",
            "padelpy:GATS5c",
            "padelpy:GATS6c",
            "padelpy:GATS7c",
            "padelpy:GATS8c",
            "padelpy:GATS1m",
            "padelpy:GATS2m",
            "padelpy:GATS3m",
            "padelpy:GATS4m",
            "padelpy:GATS5m",
            "padelpy:GATS6m",
            "padelpy:GATS7m",
            "padelpy:GATS8m",
            "padelpy:GATS1v",
            "padelpy:GATS2v",
            "padelpy:GATS3v",
            "padelpy:GATS4v",
            "padelpy:GATS5v",
            "padelpy:GATS6v",
            "padelpy:GATS7v",
            "padelpy:GATS8v",
            "padelpy:GATS1e",
            "padelpy:GATS2e",
            "padelpy:GATS3e",
            "padelpy:GATS4e",
            "padelpy:GATS5e",
            "padelpy:GATS6e",
            "padelpy:GATS7e",
            "padelpy:GATS8e",
            "padelpy:GATS1p",
            "padelpy:GATS2p",
            "padelpy:GATS3p",
            "padelpy:GATS4p",
            "padelpy:GATS5p",
            "padelpy:GATS6p",
            "padelpy:GATS7p",
            "padelpy:GATS8p",
            "padelpy:GATS1i",
            "padelpy:GATS2i",
            "padelpy:GATS3i",
            "padelpy:GATS4i",
            "padelpy:GATS5i",
            "padelpy:GATS6i",
            "padelpy:GATS7i",
            "padelpy:GATS8i",
            "padelpy:GATS1s",
            "padelpy:GATS2s",
            "padelpy:GATS3s",
            "padelpy:GATS4s",
            "padelpy:GATS5s",
            "padelpy:GATS6s",
            "padelpy:GATS7s",
            "padelpy:GATS8s",
            "padelpy:SpAbs_DzZ",
            "padelpy:SpMax_DzZ",
            "padelpy:SpDiam_DzZ",
            "padelpy:SpAD_DzZ",
            "padelpy:SpMAD_DzZ",
            "padelpy:EE_DzZ",
            "padelpy:SM1_DzZ",
            "padelpy:VE1_DzZ",
            "padelpy:VE2_DzZ",
            "padelpy:VE3_DzZ",
            "padelpy:VR1_DzZ",
            "padelpy:VR2_DzZ",
            "padelpy:VR3_DzZ",
            "padelpy:SpAbs_Dzm",
            "padelpy:SpMax_Dzm",
            "padelpy:SpDiam_Dzm",
            "padelpy:SpAD_Dzm",
            "padelpy:SpMAD_Dzm",
            "padelpy:EE_Dzm",
            "padelpy:SM1_Dzm",
            "padelpy:VE1_Dzm",
            "padelpy:VE2_Dzm",
            "padelpy:VE3_Dzm",
            "padelpy:VR1_Dzm",
            "padelpy:VR2_Dzm",
            "padelpy:VR3_Dzm",
            "padelpy:SpAbs_Dzv",
            "padelpy:SpMax_Dzv",
            "padelpy:SpDiam_Dzv",
            "padelpy:SpAD_Dzv",
            "padelpy:SpMAD_Dzv",
            "padelpy:EE_Dzv",
            "padelpy:SM1_Dzv",
            "padelpy:VE1_Dzv",
            "padelpy:VE2_Dzv",
            "padelpy:VE3_Dzv",
            "padelpy:VR1_Dzv",
            "padelpy:VR2_Dzv",
            "padelpy:VR3_Dzv",
            "padelpy:SpAbs_Dze",
            "padelpy:SpMax_Dze",
            "padelpy:SpDiam_Dze",
            "padelpy:SpAD_Dze",
            "padelpy:SpMAD_Dze",
            "padelpy:EE_Dze",
            "padelpy:SM1_Dze",
            "padelpy:VE1_Dze",
            "padelpy:VE2_Dze",
            "padelpy:VE3_Dze",
            "padelpy:VR1_Dze",
            "padelpy:VR2_Dze",
            "padelpy:VR3_Dze",
            "padelpy:SpAbs_Dzp",
            "padelpy:SpMax_Dzp",
            "padelpy:SpDiam_Dzp",
            "padelpy:SpAD_Dzp",
            "padelpy:SpMAD_Dzp",
            "padelpy:EE_Dzp",
            "padelpy:SM1_Dzp",
            "padelpy:VE1_Dzp",
            "padelpy:VE2_Dzp",
            "padelpy:VE3_Dzp",
            "padelpy:VR1_Dzp",
            "padelpy:VR2_Dzp",
            "padelpy:VR3_Dzp",
            "padelpy:SpAbs_Dzi",
            "padelpy:SpMax_Dzi",
            "padelpy:SpDiam_Dzi",
            "padelpy:SpAD_Dzi",
            "padelpy:SpMAD_Dzi",
            "padelpy:EE_Dzi",
            "padelpy:SM1_Dzi",
            "padelpy:VE1_Dzi",
            "padelpy:VE2_Dzi",
            "padelpy:VE3_Dzi",
            "padelpy:VR1_Dzi",
            "padelpy:VR2_Dzi",
            "padelpy:VR3_Dzi",
            "padelpy:SpAbs_Dzs",
            "padelpy:SpMax_Dzs",
            "padelpy:SpDiam_Dzs",
            "padelpy:SpAD_Dzs",
            "padelpy:SpMAD_Dzs",
            "padelpy:EE_Dzs",
            "padelpy:SM1_Dzs",
            "padelpy:VE1_Dzs",
            "padelpy:VE2_Dzs",
            "padelpy:VE3_Dzs",
            "padelpy:VR1_Dzs",
            "padelpy:VR2_Dzs",
            "padelpy:VR3_Dzs",
            "padelpy:nBase",
            "padelpy:BCUTw-1l",
            "padelpy:BCUTw-1h",
            "padelpy:BCUTc-1l",
            "padelpy:BCUTc-1h",
            "padelpy:BCUTp-1l",
            "padelpy:BCUTp-1h",
            "padelpy:nBonds",
            "padelpy:nBonds2",
            "padelpy:nBondsS",
            "padelpy:nBondsS2",
            "padelpy:nBondsS3",
            "padelpy:nBondsD",
            "padelpy:nBondsD2",
            "padelpy:nBondsT",
            "padelpy:nBondsQ",
            "padelpy:nBondsM",
            "padelpy:bpol",
            "padelpy:SpMax1_Bhm",
            "padelpy:SpMax2_Bhm",
            "padelpy:SpMax3_Bhm",
            "padelpy:SpMax4_Bhm",
            "padelpy:SpMax5_Bhm",
            "padelpy:SpMax6_Bhm",
            "padelpy:SpMax7_Bhm",
            "padelpy:SpMax8_Bhm",
            "padelpy:SpMin1_Bhm",
            "padelpy:SpMin2_Bhm",
            "padelpy:SpMin3_Bhm",
            "padelpy:SpMin4_Bhm",
            "padelpy:SpMin5_Bhm",
            "padelpy:SpMin6_Bhm",
            "padelpy:SpMin7_Bhm",
            "padelpy:SpMin8_Bhm",
            "padelpy:SpMax1_Bhv",
            "padelpy:SpMax2_Bhv",
            "padelpy:SpMax3_Bhv",
            "padelpy:SpMax4_Bhv",
            "padelpy:SpMax5_Bhv",
            "padelpy:SpMax6_Bhv",
            "padelpy:SpMax7_Bhv",
            "padelpy:SpMax8_Bhv",
            "padelpy:SpMin1_Bhv",
            "padelpy:SpMin2_Bhv",
            "padelpy:SpMin3_Bhv",
            "padelpy:SpMin4_Bhv",
            "padelpy:SpMin5_Bhv",
            "padelpy:SpMin6_Bhv",
            "padelpy:SpMin7_Bhv",
            "padelpy:SpMin8_Bhv",
            "padelpy:SpMax1_Bhe",
            "padelpy:SpMax2_Bhe",
            "padelpy:SpMax3_Bhe",
            "padelpy:SpMax4_Bhe",
            "padelpy:SpMax5_Bhe",
            "padelpy:SpMax6_Bhe",
            "padelpy:SpMax7_Bhe",
            "padelpy:SpMax8_Bhe",
            "padelpy:SpMin1_Bhe",
            "padelpy:SpMin2_Bhe",
            "padelpy:SpMin3_Bhe",
            "padelpy:SpMin4_Bhe",
            "padelpy:SpMin5_Bhe",
            "padelpy:SpMin6_Bhe",
            "padelpy:SpMin7_Bhe",
            "padelpy:SpMin8_Bhe",
            "padelpy:SpMax1_Bhp",
            "padelpy:SpMax2_Bhp",
            "padelpy:SpMax3_Bhp",
            "padelpy:SpMax4_Bhp",
            "padelpy:SpMax5_Bhp",
            "padelpy:SpMax6_Bhp",
            "padelpy:SpMax7_Bhp",
            "padelpy:SpMax8_Bhp",
            "padelpy:SpMin1_Bhp",
            "padelpy:SpMin2_Bhp",
            "padelpy:SpMin3_Bhp",
            "padelpy:SpMin4_Bhp",
            "padelpy:SpMin5_Bhp",
            "padelpy:SpMin6_Bhp",
            "padelpy:SpMin7_Bhp",
            "padelpy:SpMin8_Bhp",
            "padelpy:SpMax1_Bhi",
            "padelpy:SpMax2_Bhi",
            "padelpy:SpMax3_Bhi",
            "padelpy:SpMax4_Bhi",
            "padelpy:SpMax5_Bhi",
            "padelpy:SpMax6_Bhi",
            "padelpy:SpMax7_Bhi",
            "padelpy:SpMax8_Bhi",
            "padelpy:SpMin1_Bhi",
            "padelpy:SpMin2_Bhi",
            "padelpy:SpMin3_Bhi",
            "padelpy:SpMin4_Bhi",
            "padelpy:SpMin5_Bhi",
            "padelpy:SpMin6_Bhi",
            "padelpy:SpMin7_Bhi",
            "padelpy:SpMin8_Bhi",
            "padelpy:SpMax1_Bhs",
            "padelpy:SpMax2_Bhs",
            "padelpy:SpMax3_Bhs",
            "padelpy:SpMax4_Bhs",
            "padelpy:SpMax5_Bhs",
            "padelpy:SpMax6_Bhs",
            "padelpy:SpMax7_Bhs",
            "padelpy:SpMax8_Bhs",
            "padelpy:SpMin1_Bhs",
            "padelpy:SpMin2_Bhs",
            "padelpy:SpMin3_Bhs",
            "padelpy:SpMin4_Bhs",
            "padelpy:SpMin5_Bhs",
            "padelpy:SpMin6_Bhs",
            "padelpy:SpMin7_Bhs",
            "padelpy:SpMin8_Bhs",
            "padelpy:C1SP1",
            "padelpy:C2SP1",
            "padelpy:C1SP2",
            "padelpy:C2SP2",
            "padelpy:C3SP2",
            "padelpy:C1SP3",
            "padelpy:C2SP3",
            "padelpy:C3SP3",
            "padelpy:C4SP3",
            "padelpy:SCH-3",
            "padelpy:SCH-4",
            "padelpy:SCH-5",
            "padelpy:SCH-6",
            "padelpy:SCH-7",
            "padelpy:VCH-3",
            "padelpy:VCH-4",
            "padelpy:VCH-5",
            "padelpy:VCH-6",
            "padelpy:VCH-7",
            "padelpy:SC-3",
            "padelpy:SC-4",
            "padelpy:SC-5",
            "padelpy:SC-6",
            "padelpy:VC-3",
            "padelpy:VC-4",
            "padelpy:VC-5",
            "padelpy:VC-6",
            "padelpy:SPC-4",
            "padelpy:SPC-5",
            "padelpy:SPC-6",
            "padelpy:VPC-4",
            "padelpy:VPC-5",
            "padelpy:VPC-6",
            "padelpy:SP-0",
            "padelpy:SP-1",
            "padelpy:SP-2",
            "padelpy:SP-3",
            "padelpy:SP-4",
            "padelpy:SP-5",
            "padelpy:SP-6",
            "padelpy:SP-7",
            "padelpy:ASP-0",
            "padelpy:ASP-1",
            "padelpy:ASP-2",
            "padelpy:ASP-3",
            "padelpy:ASP-4",
            "padelpy:ASP-5",
            "padelpy:ASP-6",
            "padelpy:ASP-7",
            "padelpy:VP-0",
            "padelpy:VP-1",
            "padelpy:VP-2",
            "padelpy:VP-3",
            "padelpy:VP-4",
            "padelpy:VP-5",
            "padelpy:VP-6",
            "padelpy:VP-7",
            "padelpy:AVP-0",
            "padelpy:AVP-1",
            "padelpy:AVP-2",
            "padelpy:AVP-3",
            "padelpy:AVP-4",
            "padelpy:AVP-5",
            "padelpy:AVP-6",
            "padelpy:AVP-7",
            "padelpy:Sv",
            "padelpy:Sse",
            "padelpy:Spe",
            "padelpy:Sare",
            "padelpy:Sp",
            "padelpy:Si",
            "padelpy:Mv",
            "padelpy:Mse",
            "padelpy:Mpe",
            "padelpy:Mare",
            "padelpy:Mp",
            "padelpy:Mi",
            "padelpy:CrippenLogP",
            "padelpy:CrippenMR",
            "padelpy:SpMax_Dt",
            "padelpy:SpDiam_Dt",
            "padelpy:SpAD_Dt",
            "padelpy:SpMAD_Dt",
            "padelpy:EE_Dt",
            "padelpy:VE1_Dt",
            "padelpy:VE2_Dt",
            "padelpy:VE3_Dt",
            "padelpy:VR1_Dt",
            "padelpy:VR2_Dt",
            "padelpy:VR3_Dt",
            "padelpy:ECCEN",
            "padelpy:nHBd",
            "padelpy:nwHBd",
            "padelpy:nHBa",
            "padelpy:nwHBa",
            "padelpy:nHBint2",
            "padelpy:nHBint3",
            "padelpy:nHBint4",
            "padelpy:nHBint5",
            "padelpy:nHBint6",
            "padelpy:nHBint7",
            "padelpy:nHBint8",
            "padelpy:nHBint9",
            "padelpy:nHBint10",
            "padelpy:nHsOH",
            "padelpy:nHdNH",
            "padelpy:nHsSH",
            "padelpy:nHsNH2",
            "padelpy:nHssNH",
            "padelpy:nHaaNH",
            "padelpy:nHsNH3p",
            "padelpy:nHssNH2p",
            "padelpy:nHsssNHp",
            "padelpy:nHtCH",
            "padelpy:nHdCH2",
            "padelpy:nHdsCH",
            "padelpy:nHaaCH",
            "padelpy:nHCHnX",
            "padelpy:nHCsats",
            "padelpy:nHCsatu",
            "padelpy:nHAvin",
            "padelpy:nHother",
            "padelpy:nHmisc",
            "padelpy:nsLi",
            "padelpy:nssBe",
            "padelpy:nssssBem",
            "padelpy:nsBH2",
            "padelpy:nssBH",
            "padelpy:nsssB",
            "padelpy:nssssBm",
            "padelpy:nsCH3",
            "padelpy:ndCH2",
            "padelpy:nssCH2",
            "padelpy:ntCH",
            "padelpy:ndsCH",
            "padelpy:naaCH",
            "padelpy:nsssCH",
            "padelpy:nddC",
            "padelpy:ntsC",
            "padelpy:ndssC",
            "padelpy:naasC",
            "padelpy:naaaC",
            "padelpy:nssssC",
            "padelpy:nsNH3p",
            "padelpy:nsNH2",
            "padelpy:nssNH2p",
            "padelpy:ndNH",
            "padelpy:nssNH",
            "padelpy:naaNH",
            "padelpy:ntN",
            "padelpy:nsssNHp",
            "padelpy:ndsN",
            "padelpy:naaN",
            "padelpy:nsssN",
            "padelpy:nddsN",
            "padelpy:naasN",
            "padelpy:nssssNp",
            "padelpy:nsOH",
            "padelpy:ndO",
            "padelpy:nssO",
            "padelpy:naaO",
            "padelpy:naOm",
            "padelpy:nsOm",
            "padelpy:nsF",
            "padelpy:nsSiH3",
            "padelpy:nssSiH2",
            "padelpy:nsssSiH",
            "padelpy:nssssSi",
            "padelpy:nsPH2",
            "padelpy:nssPH",
            "padelpy:nsssP",
            "padelpy:ndsssP",
            "padelpy:nddsP",
            "padelpy:nsssssP",
            "padelpy:nsSH",
            "padelpy:ndS",
            "padelpy:nssS",
            "padelpy:naaS",
            "padelpy:ndssS",
            "padelpy:nddssS",
            "padelpy:nssssssS",
            "padelpy:nSm",
            "padelpy:nsCl",
            "padelpy:nsGeH3",
            "padelpy:nssGeH2",
            "padelpy:nsssGeH",
            "padelpy:nssssGe",
            "padelpy:nsAsH2",
            "padelpy:nssAsH",
            "padelpy:nsssAs",
            "padelpy:ndsssAs",
            "padelpy:nddsAs",
            "padelpy:nsssssAs",
            "padelpy:nsSeH",
            "padelpy:ndSe",
            "padelpy:nssSe",
            "padelpy:naaSe",
            "padelpy:ndssSe",
            "padelpy:nssssssSe",
            "padelpy:nddssSe",
            "padelpy:nsBr",
            "padelpy:nsSnH3",
            "padelpy:nssSnH2",
            "padelpy:nsssSnH",
            "padelpy:nssssSn",
            "padelpy:nsI",
            "padelpy:nsPbH3",
            "padelpy:nssPbH2",
            "padelpy:nsssPbH",
            "padelpy:nssssPb",
            "padelpy:SHBd",
            "padelpy:SwHBd",
            "padelpy:SHBa",
            "padelpy:SwHBa",
            "padelpy:SHBint2",
            "padelpy:SHBint3",
            "padelpy:SHBint4",
            "padelpy:SHBint5",
            "padelpy:SHBint6",
            "padelpy:SHBint7",
            "padelpy:SHBint8",
            "padelpy:SHBint9",
            "padelpy:SHBint10",
            "padelpy:SHsOH",
            "padelpy:SHdNH",
            "padelpy:SHsSH",
            "padelpy:SHsNH2",
            "padelpy:SHssNH",
            "padelpy:SHaaNH",
            "padelpy:SHsNH3p",
            "padelpy:SHssNH2p",
            "padelpy:SHsssNHp",
            "padelpy:SHtCH",
            "padelpy:SHdCH2",
            "padelpy:SHdsCH",
            "padelpy:SHaaCH",
            "padelpy:SHCHnX",
            "padelpy:SHCsats",
            "padelpy:SHCsatu",
            "padelpy:SHAvin",
            "padelpy:SHother",
            "padelpy:SHmisc",
            "padelpy:SsLi",
            "padelpy:SssBe",
            "padelpy:SssssBem",
            "padelpy:SsBH2",
            "padelpy:SssBH",
            "padelpy:SsssB",
            "padelpy:SssssBm",
            "padelpy:SsCH3",
            "padelpy:SdCH2",
            "padelpy:SssCH2",
            "padelpy:StCH",
            "padelpy:SdsCH",
            "padelpy:SaaCH",
            "padelpy:SsssCH",
            "padelpy:SddC",
            "padelpy:StsC",
            "padelpy:SdssC",
            "padelpy:SaasC",
            "padelpy:SaaaC",
            "padelpy:SssssC",
            "padelpy:SsNH3p",
            "padelpy:SsNH2",
            "padelpy:SssNH2p",
            "padelpy:SdNH",
            "padelpy:SssNH",
            "padelpy:SaaNH",
            "padelpy:StN",
            "padelpy:SsssNHp",
            "padelpy:SdsN",
            "padelpy:SaaN",
            "padelpy:SsssN",
            "padelpy:SddsN",
            "padelpy:SaasN",
            "padelpy:SssssNp",
            "padelpy:SsOH",
            "padelpy:SdO",
            "padelpy:SssO",
            "padelpy:SaaO",
            "padelpy:SaOm",
            "padelpy:SsOm",
            "padelpy:SsF",
            "padelpy:SsSiH3",
            "padelpy:SssSiH2",
            "padelpy:SsssSiH",
            "padelpy:SssssSi",
            "padelpy:SsPH2",
            "padelpy:SssPH",
            "padelpy:SsssP",
            "padelpy:SdsssP",
            "padelpy:SddsP",
            "padelpy:SsssssP",
            "padelpy:SsSH",
            "padelpy:SdS",
            "padelpy:SssS",
            "padelpy:SaaS",
            "padelpy:SdssS",
            "padelpy:SddssS",
            "padelpy:SssssssS",
            "padelpy:SSm",
            "padelpy:SsCl",
            "padelpy:SsGeH3",
            "padelpy:SssGeH2",
            "padelpy:SsssGeH",
            "padelpy:SssssGe",
            "padelpy:SsAsH2",
            "padelpy:SssAsH",
            "padelpy:SsssAs",
            "padelpy:SdsssAs",
            "padelpy:SddsAs",
            "padelpy:SsssssAs",
            "padelpy:SsSeH",
            "padelpy:SdSe",
            "padelpy:SssSe",
            "padelpy:SaaSe",
            "padelpy:SdssSe",
            "padelpy:SssssssSe",
            "padelpy:SddssSe",
            "padelpy:SsBr",
            "padelpy:SsSnH3",
            "padelpy:SssSnH2",
            "padelpy:SsssSnH",
            "padelpy:SssssSn",
            "padelpy:SsI",
            "padelpy:SsPbH3",
            "padelpy:SssPbH2",
            "padelpy:SsssPbH",
            "padelpy:SssssPb",
            "padelpy:minHBd",
            "padelpy:minwHBd",
            "padelpy:minHBa",
            "padelpy:minwHBa",
            "padelpy:minHBint2",
            "padelpy:minHBint3",
            "padelpy:minHBint4",
            "padelpy:minHBint5",
            "padelpy:minHBint6",
            "padelpy:minHBint7",
            "padelpy:minHBint8",
            "padelpy:minHBint9",
            "padelpy:minHBint10",
            "padelpy:minHsOH",
            "padelpy:minHdNH",
            "padelpy:minHsSH",
            "padelpy:minHsNH2",
            "padelpy:minHssNH",
            "padelpy:minHaaNH",
            "padelpy:minHsNH3p",
            "padelpy:minHssNH2p",
            "padelpy:minHsssNHp",
            "padelpy:minHtCH",
            "padelpy:minHdCH2",
            "padelpy:minHdsCH",
            "padelpy:minHaaCH",
            "padelpy:minHCHnX",
            "padelpy:minHCsats",
            "padelpy:minHCsatu",
            "padelpy:minHAvin",
            "padelpy:minHother",
            "padelpy:minHmisc",
            "padelpy:minsLi",
            "padelpy:minssBe",
            "padelpy:minssssBem",
            "padelpy:minsBH2",
            "padelpy:minssBH",
            "padelpy:minsssB",
            "padelpy:minssssBm",
            "padelpy:minsCH3",
            "padelpy:mindCH2",
            "padelpy:minssCH2",
            "padelpy:mintCH",
            "padelpy:mindsCH",
            "padelpy:minaaCH",
            "padelpy:minsssCH",
            "padelpy:minddC",
            "padelpy:mintsC",
            "padelpy:mindssC",
            "padelpy:minaasC",
            "padelpy:minaaaC",
            "padelpy:minssssC",
            "padelpy:minsNH3p",
            "padelpy:minsNH2",
            "padelpy:minssNH2p",
            "padelpy:mindNH",
            "padelpy:minssNH",
            "padelpy:minaaNH",
            "padelpy:mintN",
            "padelpy:minsssNHp",
            "padelpy:mindsN",
            "padelpy:minaaN",
            "padelpy:minsssN",
            "padelpy:minddsN",
            "padelpy:minaasN",
            "padelpy:minssssNp",
            "padelpy:minsOH",
            "padelpy:mindO",
            "padelpy:minssO",
            "padelpy:minaaO",
            "padelpy:minaOm",
            "padelpy:minsOm",
            "padelpy:minsF",
            "padelpy:minsSiH3",
            "padelpy:minssSiH2",
            "padelpy:minsssSiH",
            "padelpy:minssssSi",
            "padelpy:minsPH2",
            "padelpy:minssPH",
            "padelpy:minsssP",
            "padelpy:mindsssP",
            "padelpy:minddsP",
            "padelpy:minsssssP",
            "padelpy:minsSH",
            "padelpy:mindS",
            "padelpy:minssS",
            "padelpy:minaaS",
            "padelpy:mindssS",
            "padelpy:minddssS",
            "padelpy:minssssssS",
            "padelpy:minSm",
            "padelpy:minsCl",
            "padelpy:minsGeH3",
            "padelpy:minssGeH2",
            "padelpy:minsssGeH",
            "padelpy:minssssGe",
            "padelpy:minsAsH2",
            "padelpy:minssAsH",
            "padelpy:minsssAs",
            "padelpy:mindsssAs",
            "padelpy:minddsAs",
            "padelpy:minsssssAs",
            "padelpy:minsSeH",
            "padelpy:mindSe",
            "padelpy:minssSe",
            "padelpy:minaaSe",
            "padelpy:mindssSe",
            "padelpy:minssssssSe",
            "padelpy:minddssSe",
            "padelpy:minsBr",
            "padelpy:minsSnH3",
            "padelpy:minssSnH2",
            "padelpy:minsssSnH",
            "padelpy:minssssSn",
            "padelpy:minsI",
            "padelpy:minsPbH3",
            "padelpy:minssPbH2",
            "padelpy:minsssPbH",
            "padelpy:minssssPb",
            "padelpy:maxHBd",
            "padelpy:maxwHBd",
            "padelpy:maxHBa",
            "padelpy:maxwHBa",
            "padelpy:maxHBint2",
            "padelpy:maxHBint3",
            "padelpy:maxHBint4",
            "padelpy:maxHBint5",
            "padelpy:maxHBint6",
            "padelpy:maxHBint7",
            "padelpy:maxHBint8",
            "padelpy:maxHBint9",
            "padelpy:maxHBint10",
            "padelpy:maxHsOH",
            "padelpy:maxHdNH",
            "padelpy:maxHsSH",
            "padelpy:maxHsNH2",
            "padelpy:maxHssNH",
            "padelpy:maxHaaNH",
            "padelpy:maxHsNH3p",
            "padelpy:maxHssNH2p",
            "padelpy:maxHsssNHp",
            "padelpy:maxHtCH",
            "padelpy:maxHdCH2",
            "padelpy:maxHdsCH",
            "padelpy:maxHaaCH",
            "padelpy:maxHCHnX",
            "padelpy:maxHCsats",
            "padelpy:maxHCsatu",
            "padelpy:maxHAvin",
            "padelpy:maxHother",
            "padelpy:maxHmisc",
            "padelpy:maxsLi",
            "padelpy:maxssBe",
            "padelpy:maxssssBem",
            "padelpy:maxsBH2",
            "padelpy:maxssBH",
            "padelpy:maxsssB",
            "padelpy:maxssssBm",
            "padelpy:maxsCH3",
            "padelpy:maxdCH2",
            "padelpy:maxssCH2",
            "padelpy:maxtCH",
            "padelpy:maxdsCH",
            "padelpy:maxaaCH",
            "padelpy:maxsssCH",
            "padelpy:maxddC",
            "padelpy:maxtsC",
            "padelpy:maxdssC",
            "padelpy:maxaasC",
            "padelpy:maxaaaC",
            "padelpy:maxssssC",
            "padelpy:maxsNH3p",
            "padelpy:maxsNH2",
            "padelpy:maxssNH2p",
            "padelpy:maxdNH",
            "padelpy:maxssNH",
            "padelpy:maxaaNH",
            "padelpy:maxtN",
            "padelpy:maxsssNHp",
            "padelpy:maxdsN",
            "padelpy:maxaaN",
            "padelpy:maxsssN",
            "padelpy:maxddsN",
            "padelpy:maxaasN",
            "padelpy:maxssssNp",
            "padelpy:maxsOH",
            "padelpy:maxdO",
            "padelpy:maxssO",
            "padelpy:maxaaO",
            "padelpy:maxaOm",
            "padelpy:maxsOm",
            "padelpy:maxsF",
            "padelpy:maxsSiH3",
            "padelpy:maxssSiH2",
            "padelpy:maxsssSiH",
            "padelpy:maxssssSi",
            "padelpy:maxsPH2",
            "padelpy:maxssPH",
            "padelpy:maxsssP",
            "padelpy:maxdsssP",
            "padelpy:maxddsP",
            "padelpy:maxsssssP",
            "padelpy:maxsSH",
            "padelpy:maxdS",
            "padelpy:maxssS",
            "padelpy:maxaaS",
            "padelpy:maxdssS",
            "padelpy:maxddssS",
            "padelpy:maxssssssS",
            "padelpy:maxSm",
            "padelpy:maxsCl",
            "padelpy:maxsGeH3",
            "padelpy:maxssGeH2",
            "padelpy:maxsssGeH",
            "padelpy:maxssssGe",
            "padelpy:maxsAsH2",
            "padelpy:maxssAsH",
            "padelpy:maxsssAs",
            "padelpy:maxdsssAs",
            "padelpy:maxddsAs",
            "padelpy:maxsssssAs",
            "padelpy:maxsSeH",
            "padelpy:maxdSe",
            "padelpy:maxssSe",
            "padelpy:maxaaSe",
            "padelpy:maxdssSe",
            "padelpy:maxssssssSe",
            "padelpy:maxddssSe",
            "padelpy:maxsBr",
            "padelpy:maxsSnH3",
            "padelpy:maxssSnH2",
            "padelpy:maxsssSnH",
            "padelpy:maxssssSn",
            "padelpy:maxsI",
            "padelpy:maxsPbH3",
            "padelpy:maxssPbH2",
            "padelpy:maxsssPbH",
            "padelpy:maxssssPb",
            "padelpy:sumI",
            "padelpy:meanI",
            "padelpy:hmax",
            "padelpy:gmax",
            "padelpy:hmin",
            "padelpy:gmin",
            "padelpy:LipoaffinityIndex",
            "padelpy:MAXDN",
            "padelpy:MAXDP",
            "padelpy:DELS",
            "padelpy:MAXDN2",
            "padelpy:MAXDP2",
            "padelpy:DELS2",
            "padelpy:ETA_Alpha",
            "padelpy:ETA_AlphaP",
            "padelpy:ETA_dAlpha_A",
            "padelpy:ETA_dAlpha_B",
            "padelpy:ETA_Epsilon_1",
            "padelpy:ETA_Epsilon_2",
            "padelpy:ETA_Epsilon_3",
            "padelpy:ETA_Epsilon_4",
            "padelpy:ETA_Epsilon_5",
            "padelpy:ETA_dEpsilon_A",
            "padelpy:ETA_dEpsilon_B",
            "padelpy:ETA_dEpsilon_C",
            "padelpy:ETA_dEpsilon_D",
            "padelpy:ETA_Psi_1",
            "padelpy:ETA_dPsi_A",
            "padelpy:ETA_dPsi_B",
            "padelpy:ETA_Shape_P",
            "padelpy:ETA_Shape_Y",
            "padelpy:ETA_Shape_X",
            "padelpy:ETA_Beta",
            "padelpy:ETA_BetaP",
            "padelpy:ETA_Beta_s",
            "padelpy:ETA_BetaP_s",
            "padelpy:ETA_Beta_ns",
            "padelpy:ETA_BetaP_ns",
            "padelpy:ETA_dBeta",
            "padelpy:ETA_dBetaP",
            "padelpy:ETA_Beta_ns_d",
            "padelpy:ETA_BetaP_ns_d",
            "padelpy:ETA_Eta",
            "padelpy:ETA_EtaP",
            "padelpy:ETA_Eta_R",
            "padelpy:ETA_Eta_F",
            "padelpy:ETA_EtaP_F",
            "padelpy:ETA_Eta_L",
            "padelpy:ETA_EtaP_L",
            "padelpy:ETA_Eta_R_L",
            "padelpy:ETA_Eta_F_L",
            "padelpy:ETA_EtaP_F_L",
            "padelpy:ETA_Eta_B",
            "padelpy:ETA_EtaP_B",
            "padelpy:ETA_Eta_B_RC",
            "padelpy:ETA_EtaP_B_RC",
            "padelpy:FMF",
            "padelpy:fragC",
            "padelpy:nHBAcc",
            "padelpy:nHBAcc2",
            "padelpy:nHBAcc3",
            "padelpy:nHBAcc_Lipinski",
            "padelpy:nHBDon",
            "padelpy:nHBDon_Lipinski",
            "padelpy:HybRatio",
            "padelpy:IC0",
            "padelpy:IC1",
            "padelpy:IC2",
            "padelpy:IC3",
            "padelpy:IC4",
            "padelpy:IC5",
            "padelpy:TIC0",
            "padelpy:TIC1",
            "padelpy:TIC2",
            "padelpy:TIC3",
            "padelpy:TIC4",
            "padelpy:TIC5",
            "padelpy:SIC0",
            "padelpy:SIC1",
            "padelpy:SIC2",
            "padelpy:SIC3",
            "padelpy:SIC4",
            "padelpy:SIC5",
            "padelpy:CIC0",
            "padelpy:CIC1",
            "padelpy:CIC2",
            "padelpy:CIC3",
            "padelpy:CIC4",
            "padelpy:CIC5",
            "padelpy:BIC0",
            "padelpy:BIC1",
            "padelpy:BIC2",
            "padelpy:BIC3",
            "padelpy:BIC4",
            "padelpy:BIC5",
            "padelpy:MIC0",
            "padelpy:MIC1",
            "padelpy:MIC2",
            "padelpy:MIC3",
            "padelpy:MIC4",
            "padelpy:MIC5",
            "padelpy:ZMIC0",
            "padelpy:ZMIC1",
            "padelpy:ZMIC2",
            "padelpy:ZMIC3",
            "padelpy:ZMIC4",
            "padelpy:ZMIC5",
            "padelpy:Kier1",
            "padelpy:Kier2",
            "padelpy:Kier3",
            "padelpy:nAtomLC",
            "padelpy:nAtomP",
            "padelpy:nAtomLAC",
            "padelpy:MLogP",
            "padelpy:McGowan_Volume",
            "padelpy:MDEC-11",
            "padelpy:MDEC-12",
            "padelpy:MDEC-13",
            "padelpy:MDEC-14",
            "padelpy:MDEC-22",
            "padelpy:MDEC-23",
            "padelpy:MDEC-24",
            "padelpy:MDEC-33",
            "padelpy:MDEC-34",
            "padelpy:MDEC-44",
            "padelpy:MDEO-11",
            "padelpy:MDEO-12",
            "padelpy:MDEO-22",
            "padelpy:MDEN-11",
            "padelpy:MDEN-12",
            "padelpy:MDEN-13",
            "padelpy:MDEN-22",
            "padelpy:MDEN-23",
            "padelpy:MDEN-33",
            "padelpy:MLFER_A",
            "padelpy:MLFER_BH",
            "padelpy:MLFER_BO",
            "padelpy:MLFER_S",
            "padelpy:MLFER_E",
            "padelpy:MLFER_L",
            "padelpy:MPC2",
            "padelpy:MPC3",
            "padelpy:MPC4",
            "padelpy:MPC5",
            "padelpy:MPC6",
            "padelpy:MPC7",
            "padelpy:MPC8",
            "padelpy:MPC9",
            "padelpy:MPC10",
            "padelpy:TPC",
            "padelpy:piPC1",
            "padelpy:piPC2",
            "padelpy:piPC3",
            "padelpy:piPC4",
            "padelpy:piPC5",
            "padelpy:piPC6",
            "padelpy:piPC7",
            "padelpy:piPC8",
            "padelpy:piPC9",
            "padelpy:piPC10",
            "padelpy:TpiPC",
            "padelpy:R_TpiPCTPC",
            "padelpy:PetitjeanNumber",
            "padelpy:nRing",
            "padelpy:n3Ring",
            "padelpy:n4Ring",
            "padelpy:n5Ring",
            "padelpy:n6Ring",
            "padelpy:n7Ring",
            "padelpy:n8Ring",
            "padelpy:n9Ring",
            "padelpy:n10Ring",
            "padelpy:n11Ring",
            "padelpy:n12Ring",
            "padelpy:nG12Ring",
            "padelpy:nFRing",
            "padelpy:nF4Ring",
            "padelpy:nF5Ring",
            "padelpy:nF6Ring",
            "padelpy:nF7Ring",
            "padelpy:nF8Ring",
            "padelpy:nF9Ring",
            "padelpy:nF10Ring",
            "padelpy:nF11Ring",
            "padelpy:nF12Ring",
            "padelpy:nFG12Ring",
            "padelpy:nTRing",
            "padelpy:nT4Ring",
            "padelpy:nT5Ring",
            "padelpy:nT6Ring",
            "padelpy:nT7Ring",
            "padelpy:nT8Ring",
            "padelpy:nT9Ring",
            "padelpy:nT10Ring",
            "padelpy:nT11Ring",
            "padelpy:nT12Ring",
            "padelpy:nTG12Ring",
            "padelpy:nHeteroRing",
            "padelpy:n3HeteroRing",
            "padelpy:n4HeteroRing",
            "padelpy:n5HeteroRing",
            "padelpy:n6HeteroRing",
            "padelpy:n7HeteroRing",
            "padelpy:n8HeteroRing",
            "padelpy:n9HeteroRing",
            "padelpy:n10HeteroRing",
            "padelpy:n11HeteroRing",
            "padelpy:n12HeteroRing",
            "padelpy:nG12HeteroRing",
            "padelpy:nFHeteroRing",
            "padelpy:nF4HeteroRing",
            "padelpy:nF5HeteroRing",
            "padelpy:nF6HeteroRing",
            "padelpy:nF7HeteroRing",
            "padelpy:nF8HeteroRing",
            "padelpy:nF9HeteroRing",
            "padelpy:nF10HeteroRing",
            "padelpy:nF11HeteroRing",
            "padelpy:nF12HeteroRing",
            "padelpy:nFG12HeteroRing",
            "padelpy:nTHeteroRing",
            "padelpy:nT4HeteroRing",
            "padelpy:nT5HeteroRing",
            "padelpy:nT6HeteroRing",
            "padelpy:nT7HeteroRing",
            "padelpy:nT8HeteroRing",
            "padelpy:nT9HeteroRing",
            "padelpy:nT10HeteroRing",
            "padelpy:nT11HeteroRing",
            "padelpy:nT12HeteroRing",
            "padelpy:nTG12HeteroRing",
            "padelpy:nRotB",
            "padelpy:RotBFrac",
            "padelpy:nRotBt",
            "padelpy:RotBtFrac",
            "padelpy:LipinskiFailures",
            "padelpy:topoRadius",
            "padelpy:topoDiameter",
            "padelpy:topoShape",
            "padelpy:GGI1",
            "padelpy:GGI2",
            "padelpy:GGI3",
            "padelpy:GGI4",
            "padelpy:GGI5",
            "padelpy:GGI6",
            "padelpy:GGI7",
            "padelpy:GGI8",
            "padelpy:GGI9",
            "padelpy:GGI10",
            "padelpy:JGI1",
            "padelpy:JGI2",
            "padelpy:JGI3",
            "padelpy:JGI4",
            "padelpy:JGI5",
            "padelpy:JGI6",
            "padelpy:JGI7",
            "padelpy:JGI8",
            "padelpy:JGI9",
            "padelpy:JGI10",
            "padelpy:JGT",
            "padelpy:SpMax_D",
            "padelpy:SpDiam_D",
            "padelpy:SpAD_D",
            "padelpy:SpMAD_D",
            "padelpy:EE_D",
            "padelpy:VE1_D",
            "padelpy:VE2_D",
            "padelpy:VE3_D",
            "padelpy:VR1_D",
            "padelpy:VR2_D",
            "padelpy:VR3_D",
            "padelpy:TopoPSA",
            "padelpy:VABC",
            "padelpy:VAdjMat",
            "padelpy:MWC2",
            "padelpy:MWC3",
            "padelpy:MWC4",
            "padelpy:MWC5",
            "padelpy:MWC6",
            "padelpy:MWC7",
            "padelpy:MWC8",
            "padelpy:MWC9",
            "padelpy:MWC10",
            "padelpy:TWC",
            "padelpy:SRW2",
            "padelpy:SRW3",
            "padelpy:SRW4",
            "padelpy:SRW5",
            "padelpy:SRW6",
            "padelpy:SRW7",
            "padelpy:SRW8",
            "padelpy:SRW9",
            "padelpy:SRW10",
            "padelpy:TSRW",
            "padelpy:MW",
            "padelpy:AMW",
            "padelpy:WTPT-1",
            "padelpy:WTPT-2",
            "padelpy:WTPT-3",
            "padelpy:WTPT-4",
            "padelpy:WTPT-5",
            "padelpy:WPATH",
            "padelpy:WPOL",
            "padelpy:XLogP",
            "padelpy:Zagreb",
            "padelpy:TDB1u",
            "padelpy:TDB2u",
            "padelpy:TDB3u",
            "padelpy:TDB4u",
            "padelpy:TDB5u",
            "padelpy:TDB6u",
            "padelpy:TDB7u",
            "padelpy:TDB8u",
            "padelpy:TDB9u",
            "padelpy:TDB10u",
            "padelpy:TDB1m",
            "padelpy:TDB2m",
            "padelpy:TDB3m",
            "padelpy:TDB4m",
            "padelpy:TDB5m",
            "padelpy:TDB6m",
            "padelpy:TDB7m",
            "padelpy:TDB8m",
            "padelpy:TDB9m",
            "padelpy:TDB10m",
            "padelpy:TDB1v",
            "padelpy:TDB2v",
            "padelpy:TDB3v",
            "padelpy:TDB4v",
            "padelpy:TDB5v",
            "padelpy:TDB6v",
            "padelpy:TDB7v",
            "padelpy:TDB8v",
            "padelpy:TDB9v",
            "padelpy:TDB10v",
            "padelpy:TDB1e",
            "padelpy:TDB2e",
            "padelpy:TDB3e",
            "padelpy:TDB4e",
            "padelpy:TDB5e",
            "padelpy:TDB6e",
            "padelpy:TDB7e",
            "padelpy:TDB8e",
            "padelpy:TDB9e",
            "padelpy:TDB10e",
            "padelpy:TDB1p",
            "padelpy:TDB2p",
            "padelpy:TDB3p",
            "padelpy:TDB4p",
            "padelpy:TDB5p",
            "padelpy:TDB6p",
            "padelpy:TDB7p",
            "padelpy:TDB8p",
            "padelpy:TDB9p",
            "padelpy:TDB10p",
            "padelpy:TDB1i",
            "padelpy:TDB2i",
            "padelpy:TDB3i",
            "padelpy:TDB4i",
            "padelpy:TDB5i",
            "padelpy:TDB6i",
            "padelpy:TDB7i",
            "padelpy:TDB8i",
            "padelpy:TDB9i",
            "padelpy:TDB10i",
            "padelpy:TDB1s",
            "padelpy:TDB2s",
            "padelpy:TDB3s",
            "padelpy:TDB4s",
            "padelpy:TDB5s",
            "padelpy:TDB6s",
            "padelpy:TDB7s",
            "padelpy:TDB8s",
            "padelpy:TDB9s",
            "padelpy:TDB10s",
            "padelpy:TDB1r",
            "padelpy:TDB2r",
            "padelpy:TDB3r",
            "padelpy:TDB4r",
            "padelpy:TDB5r",
            "padelpy:TDB6r",
            "padelpy:TDB7r",
            "padelpy:TDB8r",
            "padelpy:TDB9r",
            "padelpy:TDB10r",
            "padelpy:PPSA-1",
            "padelpy:PPSA-2",
            "padelpy:PPSA-3",
            "padelpy:PNSA-1",
            "padelpy:PNSA-2",
            "padelpy:PNSA-3",
            "padelpy:DPSA-1",
            "padelpy:DPSA-2",
            "padelpy:DPSA-3",
            "padelpy:FPSA-1",
            "padelpy:FPSA-2",
            "padelpy:FPSA-3",
            "padelpy:FNSA-1",
            "padelpy:FNSA-2",
            "padelpy:FNSA-3",
            "padelpy:WPSA-1",
            "padelpy:WPSA-2",
            "padelpy:WPSA-3",
            "padelpy:WNSA-1",
            "padelpy:WNSA-2",
            "padelpy:WNSA-3",
            "padelpy:RPCG",
            "padelpy:RNCG",
            "padelpy:RPCS",
            "padelpy:RNCS",
            "padelpy:THSA",
            "padelpy:TPSA",
            "padelpy:RHSA",
            "padelpy:RPSA",
            "padelpy:GRAV-1",
            "padelpy:GRAV-2",
            "padelpy:GRAV-3",
            "padelpy:GRAVH-1",
            "padelpy:GRAVH-2",
            "padelpy:GRAVH-3",
            "padelpy:GRAV-4",
            "padelpy:GRAV-5",
            "padelpy:GRAV-6",
            "padelpy:LOBMAX",
            "padelpy:LOBMIN",
            "padelpy:MOMI-X",
            "padelpy:MOMI-Y",
            "padelpy:MOMI-Z",
            "padelpy:MOMI-XY",
            "padelpy:MOMI-XZ",
            "padelpy:MOMI-YZ",
            "padelpy:MOMI-R",
            "padelpy:geomRadius",
            "padelpy:geomDiameter",
            "padelpy:geomShape",
            "padelpy:RDF10u",
            "padelpy:RDF15u",
            "padelpy:RDF20u",
            "padelpy:RDF25u",
            "padelpy:RDF30u",
            "padelpy:RDF35u",
            "padelpy:RDF40u",
            "padelpy:RDF45u",
            "padelpy:RDF50u",
            "padelpy:RDF55u",
            "padelpy:RDF60u",
            "padelpy:RDF65u",
            "padelpy:RDF70u",
            "padelpy:RDF75u",
            "padelpy:RDF80u",
            "padelpy:RDF85u",
            "padelpy:RDF90u",
            "padelpy:RDF95u",
            "padelpy:RDF100u",
            "padelpy:RDF105u",
            "padelpy:RDF110u",
            "padelpy:RDF115u",
            "padelpy:RDF120u",
            "padelpy:RDF125u",
            "padelpy:RDF130u",
            "padelpy:RDF135u",
            "padelpy:RDF140u",
            "padelpy:RDF145u",
            "padelpy:RDF150u",
            "padelpy:RDF155u",
            "padelpy:RDF10m",
            "padelpy:RDF15m",
            "padelpy:RDF20m",
            "padelpy:RDF25m",
            "padelpy:RDF30m",
            "padelpy:RDF35m",
            "padelpy:RDF40m",
            "padelpy:RDF45m",
            "padelpy:RDF50m",
            "padelpy:RDF55m",
            "padelpy:RDF60m",
            "padelpy:RDF65m",
            "padelpy:RDF70m",
            "padelpy:RDF75m",
            "padelpy:RDF80m",
            "padelpy:RDF85m",
            "padelpy:RDF90m",
            "padelpy:RDF95m",
            "padelpy:RDF100m",
            "padelpy:RDF105m",
            "padelpy:RDF110m",
            "padelpy:RDF115m",
            "padelpy:RDF120m",
            "padelpy:RDF125m",
            "padelpy:RDF130m",
            "padelpy:RDF135m",
            "padelpy:RDF140m",
            "padelpy:RDF145m",
            "padelpy:RDF150m",
            "padelpy:RDF155m",
            "padelpy:RDF10v",
            "padelpy:RDF15v",
            "padelpy:RDF20v",
            "padelpy:RDF25v",
            "padelpy:RDF30v",
            "padelpy:RDF35v",
            "padelpy:RDF40v",
            "padelpy:RDF45v",
            "padelpy:RDF50v",
            "padelpy:RDF55v",
            "padelpy:RDF60v",
            "padelpy:RDF65v",
            "padelpy:RDF70v",
            "padelpy:RDF75v",
            "padelpy:RDF80v",
            "padelpy:RDF85v",
            "padelpy:RDF90v",
            "padelpy:RDF95v",
            "padelpy:RDF100v",
            "padelpy:RDF105v",
            "padelpy:RDF110v",
            "padelpy:RDF115v",
            "padelpy:RDF120v",
            "padelpy:RDF125v",
            "padelpy:RDF130v",
            "padelpy:RDF135v",
            "padelpy:RDF140v",
            "padelpy:RDF145v",
            "padelpy:RDF150v",
            "padelpy:RDF155v",
            "padelpy:RDF10e",
            "padelpy:RDF15e",
            "padelpy:RDF20e",
            "padelpy:RDF25e",
            "padelpy:RDF30e",
            "padelpy:RDF35e",
            "padelpy:RDF40e",
            "padelpy:RDF45e",
            "padelpy:RDF50e",
            "padelpy:RDF55e",
            "padelpy:RDF60e",
            "padelpy:RDF65e",
            "padelpy:RDF70e",
            "padelpy:RDF75e",
            "padelpy:RDF80e",
            "padelpy:RDF85e",
            "padelpy:RDF90e",
            "padelpy:RDF95e",
            "padelpy:RDF100e",
            "padelpy:RDF105e",
            "padelpy:RDF110e",
            "padelpy:RDF115e",
            "padelpy:RDF120e",
            "padelpy:RDF125e",
            "padelpy:RDF130e",
            "padelpy:RDF135e",
            "padelpy:RDF140e",
            "padelpy:RDF145e",
            "padelpy:RDF150e",
            "padelpy:RDF155e",
            "padelpy:RDF10p",
            "padelpy:RDF15p",
            "padelpy:RDF20p",
            "padelpy:RDF25p",
            "padelpy:RDF30p",
            "padelpy:RDF35p",
            "padelpy:RDF40p",
            "padelpy:RDF45p",
            "padelpy:RDF50p",
            "padelpy:RDF55p",
            "padelpy:RDF60p",
            "padelpy:RDF65p",
            "padelpy:RDF70p",
            "padelpy:RDF75p",
            "padelpy:RDF80p",
            "padelpy:RDF85p",
            "padelpy:RDF90p",
            "padelpy:RDF95p",
            "padelpy:RDF100p",
            "padelpy:RDF105p",
            "padelpy:RDF110p",
            "padelpy:RDF115p",
            "padelpy:RDF120p",
            "padelpy:RDF125p",
            "padelpy:RDF130p",
            "padelpy:RDF135p",
            "padelpy:RDF140p",
            "padelpy:RDF145p",
            "padelpy:RDF150p",
            "padelpy:RDF155p",
            "padelpy:RDF10i",
            "padelpy:RDF15i",
            "padelpy:RDF20i",
            "padelpy:RDF25i",
            "padelpy:RDF30i",
            "padelpy:RDF35i",
            "padelpy:RDF40i",
            "padelpy:RDF45i",
            "padelpy:RDF50i",
            "padelpy:RDF55i",
            "padelpy:RDF60i",
            "padelpy:RDF65i",
            "padelpy:RDF70i",
            "padelpy:RDF75i",
            "padelpy:RDF80i",
            "padelpy:RDF85i",
            "padelpy:RDF90i",
            "padelpy:RDF95i",
            "padelpy:RDF100i",
            "padelpy:RDF105i",
            "padelpy:RDF110i",
            "padelpy:RDF115i",
            "padelpy:RDF120i",
            "padelpy:RDF125i",
            "padelpy:RDF130i",
            "padelpy:RDF135i",
            "padelpy:RDF140i",
            "padelpy:RDF145i",
            "padelpy:RDF150i",
            "padelpy:RDF155i",
            "padelpy:RDF10s",
            "padelpy:RDF15s",
            "padelpy:RDF20s",
            "padelpy:RDF25s",
            "padelpy:RDF30s",
            "padelpy:RDF35s",
            "padelpy:RDF40s",
            "padelpy:RDF45s",
            "padelpy:RDF50s",
            "padelpy:RDF55s",
            "padelpy:RDF60s",
            "padelpy:RDF65s",
            "padelpy:RDF70s",
            "padelpy:RDF75s",
            "padelpy:RDF80s",
            "padelpy:RDF85s",
            "padelpy:RDF90s",
            "padelpy:RDF95s",
            "padelpy:RDF100s",
            "padelpy:RDF105s",
            "padelpy:RDF110s",
            "padelpy:RDF115s",
            "padelpy:RDF120s",
            "padelpy:RDF125s",
            "padelpy:RDF130s",
            "padelpy:RDF135s",
            "padelpy:RDF140s",
            "padelpy:RDF145s",
            "padelpy:RDF150s",
            "padelpy:RDF155s",
            "padelpy:L1u",
            "padelpy:L2u",
            "padelpy:L3u",
            "padelpy:P1u",
            "padelpy:P2u",
            "padelpy:E1u",
            "padelpy:E2u",
            "padelpy:E3u",
            "padelpy:Tu",
            "padelpy:Au",
            "padelpy:Vu",
            "padelpy:Ku",
            "padelpy:Du",
            "padelpy:L1m",
            "padelpy:L2m",
            "padelpy:L3m",
            "padelpy:P1m",
            "padelpy:P2m",
            "padelpy:E1m",
            "padelpy:E2m",
            "padelpy:E3m",
            "padelpy:Tm",
            "padelpy:Am",
            "padelpy:Vm",
            "padelpy:Km",
            "padelpy:Dm",
            "padelpy:L1v",
            "padelpy:L2v",
            "padelpy:L3v",
            "padelpy:P1v",
            "padelpy:P2v",
            "padelpy:E1v",
            "padelpy:E2v",
            "padelpy:E3v",
            "padelpy:Tv",
            "padelpy:Av",
            "padelpy:Vv",
            "padelpy:Kv",
            "padelpy:Dv",
            "padelpy:L1e",
            "padelpy:L2e",
            "padelpy:L3e",
            "padelpy:P1e",
            "padelpy:P2e",
            "padelpy:E1e",
            "padelpy:E2e",
            "padelpy:E3e",
            "padelpy:Te",
            "padelpy:Ae",
            "padelpy:Ve",
            "padelpy:Ke",
            "padelpy:De",
            "padelpy:L1p",
            "padelpy:L2p",
            "padelpy:L3p",
            "padelpy:P1p",
            "padelpy:P2p",
            "padelpy:E1p",
            "padelpy:E2p",
            "padelpy:E3p",
            "padelpy:Tp",
            "padelpy:Ap",
            "padelpy:Vp",
            "padelpy:Kp",
            "padelpy:Dp",
            "padelpy:L1i",
            "padelpy:L2i",
            "padelpy:L3i",
            "padelpy:P1i",
            "padelpy:P2i",
            "padelpy:E1i",
            "padelpy:E2i",
            "padelpy:E3i",
            "padelpy:Ti",
            "padelpy:Ai",
            "padelpy:Vi",
            "padelpy:Ki",
            "padelpy:Di",
            "padelpy:L1s",
            "padelpy:L2s",
            "padelpy:L3s",
            "padelpy:P1s",
            "padelpy:P2s",
            "padelpy:E1s",
            "padelpy:E2s",
            "padelpy:E3s",
            "padelpy:Ts",
            "padelpy:As",
            "padelpy:Vs",
            "padelpy:Ks",
            "padelpy:Ds",
        ]
