# Copyright (c) 2025 Microsoft Corporation.
# Licensed under the MIT License

"""
OpenAI embedding implementation.

WARNING: This code is under development and may undergo changes in future releases.
Backwards compatibility is not guaranteed at this time.
"""

import os
import asyncio
import threading
from typing import List, Optional

from openai import AsyncOpenAI
from nlweb_core.config import CONFIG


# Add lock for thread-safe client access
_client_lock = threading.Lock()
openai_client = None

def get_openai_api_key() -> str:
    """
    Retrieve the OpenAI API key from configuration.
    """
    # Get the API key from the embedding provider config
    provider_config = CONFIG.get_embedding_provider("openai")
    if provider_config and provider_config.api_key:
        api_key = provider_config.api_key
        if api_key:
            return api_key
    
    # Fallback to environment variable
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        error_msg = "OpenAI API key not found in configuration or environment"
        raise ValueError(error_msg)
    
    return api_key

def get_async_client() -> AsyncOpenAI:
    """
    Configure and return an asynchronous OpenAI client.
    """
    global openai_client
    with _client_lock:  # Thread-safe client initialization
        if openai_client is None:
            try:
                api_key = get_openai_api_key()
                openai_client = AsyncOpenAI(api_key=api_key)
            except Exception as e:
                raise
    
    return openai_client

async def get_openai_embeddings(
    text: str,
    model: Optional[str] = None,
    timeout: float = 30.0
) -> List[float]:
    """
    Generate an embedding for a single text using OpenAI API.
    
    Args:
        text: The text to embed
        model: Optional model ID to use, defaults to provider's configured model
        timeout: Maximum time to wait for the embedding response in seconds
        
    Returns:
        List of floats representing the embedding vector
    """
    # If model not provided, get it from config
    if model is None:
        provider_config = CONFIG.get_embedding_provider("openai")
        if provider_config and provider_config.model:
            model = provider_config.model
        else:
            # Default to a common embedding model
            model = "text-embedding-3-small"
    
    
    client = get_async_client()

    try:
        # Clean input text (replace newlines with spaces)
        text = text.replace("\n", " ")
        
        response = await client.embeddings.create(
            input=text,
            model=model
        )
        
        embedding = response.data[0].embedding
        return embedding
    except Exception as e:
        raise

async def get_openai_batch_embeddings(
    texts: List[str],
    model: Optional[str] = None,
    timeout: float = 60.0
) -> List[List[float]]:
    """
    Generate embeddings for multiple texts using OpenAI API.
    
    Args:
        texts: List of texts to embed
        model: Optional model ID to use, defaults to provider's configured model
        timeout: Maximum time to wait for the batch embedding response in seconds
        
    Returns:
        List of embedding vectors, each a list of floats
    """
    # If model not provided, get it from config
    if model is None:
        provider_config = CONFIG.get_embedding_provider("openai")
        if provider_config and provider_config.model:
            model = provider_config.model
        else:
            # Default to a common embedding model
            model = "text-embedding-3-small"
    
    
    client = get_async_client()

    try:
        # Clean input texts (replace newlines with spaces)
        cleaned_texts = [text.replace("\n", " ") for text in texts]
        
        response = await client.embeddings.create(
            input=cleaned_texts,
            model=model
        )
        
        # Extract embeddings in the same order as input texts
        # Use sorted to ensure correct ordering by index
        embeddings = [data.embedding for data in sorted(response.data, key=lambda x: x.index)]
        return embeddings
    except Exception as e:
        raise