# pylint: disable=line-too-long,too-many-lines,missing-docstring
"""ImageNet classification dataset."""
from os import path
from mxnet.gluon.data.vision import ImageFolderDataset

__all__ = ['ImageNet', 'ImageNet1kAttr']

class ImageNet(ImageFolderDataset):
    """Load the ImageNet classification dataset.

    Refer to :doc:`../build/examples_datasets/imagenet` for the description of
    this dataset and how to prepare it.

    Parameters
    ----------
    root : str, default '~/.mxnet/datasets/imagenet'
        Path to the folder stored the dataset.
    train : bool, default True
        Whether to load the training or validation set.
    transform : function, default None
        A function that takes data and label and transforms them. Refer to
        :doc:`./transforms` for examples. (TODO, should we restrict its datatype
        to transformer?)
    """
    def __init__(self, root=path.join('~', '.mxnet', 'datasets', 'imagenet'),
                 train=True, transform=None):
        split = 'train' if train else 'val'
        root = path.join(root, split)
        super(ImageNet, self).__init__(root=root, flag=1, transform=transform)

class ImageNet1kAttr(object):
    def __init__(self):
        self.num_class = 1000
        self.synset = []
        self.classes = []
        self.classes_long = []

        for syn, cls, cls_l in _ILSVRC2012_Attr:
            self.synset.append(syn)
            self.classes.append(cls)
            self.classes_long.append(cls_l)

_ILSVRC2012_Attr = [
    ("n01440764", "tench", "tench, Tinca tinca"),
    ("n01443537", "goldfish", "goldfish, Carassius auratus"),
    ("n01484850", "great white shark", "great white shark, white shark, man-eater, man-eating shark, Carcharodon carcharias"),
    ("n01491361", "tiger shark", "tiger shark, Galeocerdo cuvieri"),
    ("n01494475", "hammerhead", "hammerhead, hammerhead shark"),
    ("n01496331", "electric ray", "electric ray, crampfish, numbfish, torpedo"),
    ("n01498041", "stingray", "stingray"),
    ("n01514668", "cock", "cock"),
    ("n01514859", "hen", "hen"),
    ("n01518878", "ostrich", "ostrich, Struthio camelus"),
    ("n01530575", "brambling", "brambling, Fringilla montifringilla"),
    ("n01531178", "goldfinch", "goldfinch, Carduelis carduelis"),
    ("n01532829", "house finch", "house finch, linnet, Carpodacus mexicanus"),
    ("n01534433", "junco", "junco, snowbird"),
    ("n01537544", "indigo bunting", "indigo bunting, indigo finch, indigo bird, Passerina cyanea"),
    ("n01558993", "robin", "robin, American robin, Turdus migratorius"),
    ("n01560419", "bulbul", "bulbul"),
    ("n01580077", "jay", "jay"),
    ("n01582220", "magpie", "magpie"),
    ("n01592084", "chickadee", "chickadee"),
    ("n01601694", "water ouzel", "water ouzel, dipper"),
    ("n01608432", "kite", "kite"),
    ("n01614925", "bald eagle", "bald eagle, American eagle, Haliaeetus leucocephalus"),
    ("n01616318", "vulture", "vulture"),
    ("n01622779", "great grey owl", "great grey owl, great gray owl, Strix nebulosa"),
    ("n01629819", "European fire salamander", "European fire salamander, Salamandra salamandra"),
    ("n01630670", "common newt", "common newt, Triturus vulgaris"),
    ("n01631663", "eft", "eft"),
    ("n01632458", "spotted salamander", "spotted salamander, Ambystoma maculatum"),
    ("n01632777", "axolotl", "axolotl, mud puppy, Ambystoma mexicanum"),
    ("n01641577", "bullfrog", "bullfrog, Rana catesbeiana"),
    ("n01644373", "tree frog", "tree frog, tree-frog"),
    ("n01644900", "tailed frog", "tailed frog, bell toad, ribbed toad, tailed toad, Ascaphus trui"),
    ("n01664065", "loggerhead", "loggerhead, loggerhead turtle, Caretta caretta"),
    ("n01665541", "leatherback turtle", "leatherback turtle, leatherback, leathery turtle, Dermochelys coriacea"),
    ("n01667114", "mud turtle", "mud turtle"),
    ("n01667778", "terrapin", "terrapin"),
    ("n01669191", "box turtle", "box turtle, box tortoise"),
    ("n01675722", "banded gecko", "banded gecko"),
    ("n01677366", "common iguana", "common iguana, iguana, Iguana iguana"),
    ("n01682714", "American chameleon", "American chameleon, anole, Anolis carolinensis"),
    ("n01685808", "whiptail", "whiptail, whiptail lizard"),
    ("n01687978", "agama", "agama"),
    ("n01688243", "frilled lizard", "frilled lizard, Chlamydosaurus kingi"),
    ("n01689811", "alligator lizard", "alligator lizard"),
    ("n01692333", "Gila monster", "Gila monster, Heloderma suspectum"),
    ("n01693334", "green lizard", "green lizard, Lacerta viridis"),
    ("n01694178", "African chameleon", "African chameleon, Chamaeleo chamaeleon"),
    ("n01695060", "Komodo dragon", "Komodo dragon, Komodo lizard, dragon lizard, giant lizard, Varanus komodoensis"),
    ("n01697457", "African crocodile", "African crocodile, Nile crocodile, Crocodylus niloticus"),
    ("n01698640", "American alligator", "American alligator, Alligator mississipiensis"),
    ("n01704323", "triceratops", "triceratops"),
    ("n01728572", "thunder snake", "thunder snake, worm snake, Carphophis amoenus"),
    ("n01728920", "ringneck snake", "ringneck snake, ring-necked snake, ring snake"),
    ("n01729322", "hognose snake", "hognose snake, puff adder, sand viper"),
    ("n01729977", "green snake", "green snake, grass snake"),
    ("n01734418", "king snake", "king snake, kingsnake"),
    ("n01735189", "garter snake", "garter snake, grass snake"),
    ("n01737021", "water snake", "water snake"),
    ("n01739381", "vine snake", "vine snake"),
    ("n01740131", "night snake", "night snake, Hypsiglena torquata"),
    ("n01742172", "boa constrictor", "boa constrictor, Constrictor constrictor"),
    ("n01744401", "rock python", "rock python, rock snake, Python sebae"),
    ("n01748264", "Indian cobra", "Indian cobra, Naja naja"),
    ("n01749939", "green mamba", "green mamba"),
    ("n01751748", "sea snake", "sea snake"),
    ("n01753488", "horned viper", "horned viper, cerastes, sand viper, horned asp, Cerastes cornutus"),
    ("n01755581", "diamondback", "diamondback, diamondback rattlesnake, Crotalus adamanteus"),
    ("n01756291", "sidewinder", "sidewinder, horned rattlesnake, Crotalus cerastes"),
    ("n01768244", "trilobite", "trilobite"),
    ("n01770081", "harvestman", "harvestman, daddy longlegs, Phalangium opilio"),
    ("n01770393", "scorpion", "scorpion"),
    ("n01773157", "black and gold garden spider", "black and gold garden spider, Argiope aurantia"),
    ("n01773549", "barn spider", "barn spider, Araneus cavaticus"),
    ("n01773797", "garden spider", "garden spider, Aranea diademata"),
    ("n01774384", "black widow", "black widow, Latrodectus mactans"),
    ("n01774750", "tarantula", "tarantula"),
    ("n01775062", "wolf spider", "wolf spider, hunting spider"),
    ("n01776313", "tick", "tick"),
    ("n01784675", "centipede", "centipede"),
    ("n01795545", "black grouse", "black grouse"),
    ("n01796340", "ptarmigan", "ptarmigan"),
    ("n01797886", "ruffed grouse", "ruffed grouse, partridge, Bonasa umbellus"),
    ("n01798484", "prairie chicken", "prairie chicken, prairie grouse, prairie fowl"),
    ("n01806143", "peacock", "peacock"),
    ("n01806567", "quail", "quail"),
    ("n01807496", "partridge", "partridge"),
    ("n01817953", "African grey", "African grey, African gray, Psittacus erithacus"),
    ("n01818515", "macaw", "macaw"),
    ("n01819313", "sulphur-crested cockatoo", "sulphur-crested cockatoo, Kakatoe galerita, Cacatua galerita"),
    ("n01820546", "lorikeet", "lorikeet"),
    ("n01824575", "coucal", "coucal"),
    ("n01828970", "bee eater", "bee eater"),
    ("n01829413", "hornbill", "hornbill"),
    ("n01833805", "hummingbird", "hummingbird"),
    ("n01843065", "jacamar", "jacamar"),
    ("n01843383", "toucan", "toucan"),
    ("n01847000", "drake", "drake"),
    ("n01855032", "red-breasted merganser", "red-breasted merganser, Mergus serrator"),
    ("n01855672", "goose", "goose"),
    ("n01860187", "black swan", "black swan, Cygnus atratus"),
    ("n01871265", "tusker", "tusker"),
    ("n01872401", "echidna", "echidna, spiny anteater, anteater"),
    ("n01873310", "platypus", "platypus, duckbill, duckbilled platypus, duck-billed platypus, Ornithorhynchus anatinus"),
    ("n01877812", "wallaby", "wallaby, brush kangaroo"),
    ("n01882714", "koala", "koala, koala bear, kangaroo bear, native bear, Phascolarctos cinereus"),
    ("n01883070", "wombat", "wombat"),
    ("n01910747", "jellyfish", "jellyfish"),
    ("n01914609", "sea anemone", "sea anemone, anemone"),
    ("n01917289", "brain coral", "brain coral"),
    ("n01924916", "flatworm", "flatworm, platyhelminth"),
    ("n01930112", "nematode", "nematode, nematode worm, roundworm"),
    ("n01943899", "conch", "conch"),
    ("n01944390", "snail", "snail"),
    ("n01945685", "slug", "slug"),
    ("n01950731", "sea slug", "sea slug, nudibranch"),
    ("n01955084", "chiton", "chiton, coat-of-mail shell, sea cradle, polyplacophore"),
    ("n01968897", "chambered nautilus", "chambered nautilus, pearly nautilus, nautilus"),
    ("n01978287", "Dungeness crab", "Dungeness crab, Cancer magister"),
    ("n01978455", "rock crab", "rock crab, Cancer irroratus"),
    ("n01980166", "fiddler crab", "fiddler crab"),
    ("n01981276", "king crab", "king crab, Alaska crab, Alaskan king crab, Alaska king crab, Paralithodes camtschatica"),
    ("n01983481", "American lobster", "American lobster, Northern lobster, Maine lobster, Homarus americanus"),
    ("n01984695", "spiny lobster", "spiny lobster, langouste, rock lobster, crawfish, crayfish, sea crawfish"),
    ("n01985128", "crayfish", "crayfish, crawfish, crawdad, crawdaddy"),
    ("n01986214", "hermit crab", "hermit crab"),
    ("n01990800", "isopod", "isopod"),
    ("n02002556", "white stork", "white stork, Ciconia ciconia"),
    ("n02002724", "black stork", "black stork, Ciconia nigra"),
    ("n02006656", "spoonbill", "spoonbill"),
    ("n02007558", "flamingo", "flamingo"),
    ("n02009229", "little blue heron", "little blue heron, Egretta caerulea"),
    ("n02009912", "American egret", "American egret, great white heron, Egretta albus"),
    ("n02011460", "bittern", "bittern"),
    ("n02012849", "crane", "crane"),
    ("n02013706", "limpkin", "limpkin, Aramus pictus"),
    ("n02017213", "European gallinule", "European gallinule, Porphyrio porphyrio"),
    ("n02018207", "American coot", "American coot, marsh hen, mud hen, water hen, Fulica americana"),
    ("n02018795", "bustard", "bustard"),
    ("n02025239", "ruddy turnstone", "ruddy turnstone, Arenaria interpres"),
    ("n02027492", "red-backed sandpiper", "red-backed sandpiper, dunlin, Erolia alpina"),
    ("n02028035", "redshank", "redshank, Tringa totanus"),
    ("n02033041", "dowitcher", "dowitcher"),
    ("n02037110", "oystercatcher", "oystercatcher, oyster catcher"),
    ("n02051845", "pelican", "pelican"),
    ("n02056570", "king penguin", "king penguin, Aptenodytes patagonica"),
    ("n02058221", "albatross", "albatross, mollymawk"),
    ("n02066245", "grey whale", "grey whale, gray whale, devilfish, Eschrichtius gibbosus, Eschrichtius robustus"),
    ("n02071294", "killer whale", "killer whale, killer, orca, grampus, sea wolf, Orcinus orca"),
    ("n02074367", "dugong", "dugong, Dugong dugon"),
    ("n02077923", "sea lion", "sea lion"),
    ("n02085620", "Chihuahua", "Chihuahua"),
    ("n02085782", "Japanese spaniel", "Japanese spaniel"),
    ("n02085936", "Maltese dog", "Maltese dog, Maltese terrier, Maltese"),
    ("n02086079", "Pekinese", "Pekinese, Pekingese, Peke"),
    ("n02086240", "Shih-Tzu", "Shih-Tzu"),
    ("n02086646", "Blenheim spaniel", "Blenheim spaniel"),
    ("n02086910", "papillon", "papillon"),
    ("n02087046", "toy terrier", "toy terrier"),
    ("n02087394", "Rhodesian ridgeback", "Rhodesian ridgeback"),
    ("n02088094", "Afghan hound", "Afghan hound, Afghan"),
    ("n02088238", "basset", "basset, basset hound"),
    ("n02088364", "beagle", "beagle"),
    ("n02088466", "bloodhound", "bloodhound, sleuthhound"),
    ("n02088632", "bluetick", "bluetick"),
    ("n02089078", "black-and-tan coonhound", "black-and-tan coonhound"),
    ("n02089867", "Walker hound", "Walker hound, Walker foxhound"),
    ("n02089973", "English foxhound", "English foxhound"),
    ("n02090379", "redbone", "redbone"),
    ("n02090622", "borzoi", "borzoi, Russian wolfhound"),
    ("n02090721", "Irish wolfhound", "Irish wolfhound"),
    ("n02091032", "Italian greyhound", "Italian greyhound"),
    ("n02091134", "whippet", "whippet"),
    ("n02091244", "Ibizan hound", "Ibizan hound, Ibizan Podenco"),
    ("n02091467", "Norwegian elkhound", "Norwegian elkhound, elkhound"),
    ("n02091635", "otterhound", "otterhound, otter hound"),
    ("n02091831", "Saluki", "Saluki, gazelle hound"),
    ("n02092002", "Scottish deerhound", "Scottish deerhound, deerhound"),
    ("n02092339", "Weimaraner", "Weimaraner"),
    ("n02093256", "Staffordshire bullterrier", "Staffordshire bullterrier, Staffordshire bull terrier"),
    ("n02093428", "American Staffordshire terrier", "American Staffordshire terrier, Staffordshire terrier, American pit bull terrier, pit bull terrier"),
    ("n02093647", "Bedlington terrier", "Bedlington terrier"),
    ("n02093754", "Border terrier", "Border terrier"),
    ("n02093859", "Kerry blue terrier", "Kerry blue terrier"),
    ("n02093991", "Irish terrier", "Irish terrier"),
    ("n02094114", "Norfolk terrier", "Norfolk terrier"),
    ("n02094258", "Norwich terrier", "Norwich terrier"),
    ("n02094433", "Yorkshire terrier", "Yorkshire terrier"),
    ("n02095314", "wire-haired fox terrier", "wire-haired fox terrier"),
    ("n02095570", "Lakeland terrier", "Lakeland terrier"),
    ("n02095889", "Sealyham terrier", "Sealyham terrier, Sealyham"),
    ("n02096051", "Airedale", "Airedale, Airedale terrier"),
    ("n02096177", "cairn", "cairn, cairn terrier"),
    ("n02096294", "Australian terrier", "Australian terrier"),
    ("n02096437", "Dandie Dinmont", "Dandie Dinmont, Dandie Dinmont terrier"),
    ("n02096585", "Boston bull", "Boston bull, Boston terrier"),
    ("n02097047", "miniature schnauzer", "miniature schnauzer"),
    ("n02097130", "giant schnauzer", "giant schnauzer"),
    ("n02097209", "standard schnauzer", "standard schnauzer"),
    ("n02097298", "Scotch terrier", "Scotch terrier, Scottish terrier, Scottie"),
    ("n02097474", "Tibetan terrier", "Tibetan terrier, chrysanthemum dog"),
    ("n02097658", "silky terrier", "silky terrier, Sydney silky"),
    ("n02098105", "soft-coated wheaten terrier", "soft-coated wheaten terrier"),
    ("n02098286", "West Highland white terrier", "West Highland white terrier"),
    ("n02098413", "Lhasa", "Lhasa, Lhasa apso"),
    ("n02099267", "flat-coated retriever", "flat-coated retriever"),
    ("n02099429", "curly-coated retriever", "curly-coated retriever"),
    ("n02099601", "golden retriever", "golden retriever"),
    ("n02099712", "Labrador retriever", "Labrador retriever"),
    ("n02099849", "Chesapeake Bay retriever", "Chesapeake Bay retriever"),
    ("n02100236", "German short-haired pointer", "German short-haired pointer"),
    ("n02100583", "vizsla", "vizsla, Hungarian pointer"),
    ("n02100735", "English setter", "English setter"),
    ("n02100877", "Irish setter", "Irish setter, red setter"),
    ("n02101006", "Gordon setter", "Gordon setter"),
    ("n02101388", "Brittany spaniel", "Brittany spaniel"),
    ("n02101556", "clumber", "clumber, clumber spaniel"),
    ("n02102040", "English springer", "English springer, English springer spaniel"),
    ("n02102177", "Welsh springer spaniel", "Welsh springer spaniel"),
    ("n02102318", "cocker spaniel", "cocker spaniel, English cocker spaniel, cocker"),
    ("n02102480", "Sussex spaniel", "Sussex spaniel"),
    ("n02102973", "Irish water spaniel", "Irish water spaniel"),
    ("n02104029", "kuvasz", "kuvasz"),
    ("n02104365", "schipperke", "schipperke"),
    ("n02105056", "groenendael", "groenendael"),
    ("n02105162", "malinois", "malinois"),
    ("n02105251", "briard", "briard"),
    ("n02105412", "kelpie", "kelpie"),
    ("n02105505", "komondor", "komondor"),
    ("n02105641", "Old English sheepdog", "Old English sheepdog, bobtail"),
    ("n02105855", "Shetland sheepdog", "Shetland sheepdog, Shetland sheep dog, Shetland"),
    ("n02106030", "collie", "collie"),
    ("n02106166", "Border collie", "Border collie"),
    ("n02106382", "Bouvier des Flandres", "Bouvier des Flandres, Bouviers des Flandres"),
    ("n02106550", "Rottweiler", "Rottweiler"),
    ("n02106662", "German shepherd", "German shepherd, German shepherd dog, German police dog, alsatian"),
    ("n02107142", "Doberman", "Doberman, Doberman pinscher"),
    ("n02107312", "miniature pinscher", "miniature pinscher"),
    ("n02107574", "Greater Swiss Mountain dog", "Greater Swiss Mountain dog"),
    ("n02107683", "Bernese mountain dog", "Bernese mountain dog"),
    ("n02107908", "Appenzeller", "Appenzeller"),
    ("n02108000", "EntleBucher", "EntleBucher"),
    ("n02108089", "boxer", "boxer"),
    ("n02108422", "bull mastiff", "bull mastiff"),
    ("n02108551", "Tibetan mastiff", "Tibetan mastiff"),
    ("n02108915", "French bulldog", "French bulldog"),
    ("n02109047", "Great Dane", "Great Dane"),
    ("n02109525", "Saint Bernard", "Saint Bernard, St Bernard"),
    ("n02109961", "Eskimo dog", "Eskimo dog, husky"),
    ("n02110063", "malamute", "malamute, malemute, Alaskan malamute"),
    ("n02110185", "Siberian husky", "Siberian husky"),
    ("n02110341", "dalmatian", "dalmatian, coach dog, carriage dog"),
    ("n02110627", "affenpinscher", "affenpinscher, monkey pinscher, monkey dog"),
    ("n02110806", "basenji", "basenji"),
    ("n02110958", "pug", "pug, pug-dog"),
    ("n02111129", "Leonberg", "Leonberg"),
    ("n02111277", "Newfoundland", "Newfoundland, Newfoundland dog"),
    ("n02111500", "Great Pyrenees", "Great Pyrenees"),
    ("n02111889", "Samoyed", "Samoyed, Samoyede"),
    ("n02112018", "Pomeranian", "Pomeranian"),
    ("n02112137", "chow", "chow, chow chow"),
    ("n02112350", "keeshond", "keeshond"),
    ("n02112706", "Brabancon griffon", "Brabancon griffon"),
    ("n02113023", "Pembroke", "Pembroke, Pembroke Welsh corgi"),
    ("n02113186", "Cardigan", "Cardigan, Cardigan Welsh corgi"),
    ("n02113624", "toy poodle", "toy poodle"),
    ("n02113712", "miniature poodle", "miniature poodle"),
    ("n02113799", "standard poodle", "standard poodle"),
    ("n02113978", "Mexican hairless", "Mexican hairless"),
    ("n02114367", "timber wolf", "timber wolf, grey wolf, gray wolf, Canis lupus"),
    ("n02114548", "white wolf", "white wolf, Arctic wolf, Canis lupus tundrarum"),
    ("n02114712", "red wolf", "red wolf, maned wolf, Canis rufus, Canis niger"),
    ("n02114855", "coyote", "coyote, prairie wolf, brush wolf, Canis latrans"),
    ("n02115641", "dingo", "dingo, warrigal, warragal, Canis dingo"),
    ("n02115913", "dhole", "dhole, Cuon alpinus"),
    ("n02116738", "African hunting dog", "African hunting dog, hyena dog, Cape hunting dog, Lycaon pictus"),
    ("n02117135", "hyena", "hyena, hyaena"),
    ("n02119022", "red fox", "red fox, Vulpes vulpes"),
    ("n02119789", "kit fox", "kit fox, Vulpes macrotis"),
    ("n02120079", "Arctic fox", "Arctic fox, white fox, Alopex lagopus"),
    ("n02120505", "grey fox", "grey fox, gray fox, Urocyon cinereoargenteus"),
    ("n02123045", "tabby", "tabby, tabby cat"),
    ("n02123159", "tiger cat", "tiger cat"),
    ("n02123394", "Persian cat", "Persian cat"),
    ("n02123597", "Siamese cat", "Siamese cat, Siamese"),
    ("n02124075", "Egyptian cat", "Egyptian cat"),
    ("n02125311", "cougar", "cougar, puma, catamount, mountain lion, painter, panther, Felis concolor"),
    ("n02127052", "lynx", "lynx, catamount"),
    ("n02128385", "leopard", "leopard, Panthera pardus"),
    ("n02128757", "snow leopard", "snow leopard, ounce, Panthera uncia"),
    ("n02128925", "jaguar", "jaguar, panther, Panthera onca, Felis onca"),
    ("n02129165", "lion", "lion, king of beasts, Panthera leo"),
    ("n02129604", "tiger", "tiger, Panthera tigris"),
    ("n02130308", "cheetah", "cheetah, chetah, Acinonyx jubatus"),
    ("n02132136", "brown bear", "brown bear, bruin, Ursus arctos"),
    ("n02133161", "American black bear", "American black bear, black bear, Ursus americanus, Euarctos americanus"),
    ("n02134084", "ice bear", "ice bear, polar bear, Ursus Maritimus, Thalarctos maritimus"),
    ("n02134418", "sloth bear", "sloth bear, Melursus ursinus, Ursus ursinus"),
    ("n02137549", "mongoose", "mongoose"),
    ("n02138441", "meerkat", "meerkat, mierkat"),
    ("n02165105", "tiger beetle", "tiger beetle"),
    ("n02165456", "ladybug", "ladybug, ladybeetle, lady beetle, ladybird, ladybird beetle"),
    ("n02167151", "ground beetle", "ground beetle, carabid beetle"),
    ("n02168699", "long-horned beetle", "long-horned beetle, longicorn, longicorn beetle"),
    ("n02169497", "leaf beetle", "leaf beetle, chrysomelid"),
    ("n02172182", "dung beetle", "dung beetle"),
    ("n02174001", "rhinoceros beetle", "rhinoceros beetle"),
    ("n02177972", "weevil", "weevil"),
    ("n02190166", "fly", "fly"),
    ("n02206856", "bee", "bee"),
    ("n02219486", "ant", "ant, emmet, pismire"),
    ("n02226429", "grasshopper", "grasshopper, hopper"),
    ("n02229544", "cricket", "cricket"),
    ("n02231487", "walking stick", "walking stick, walkingstick, stick insect"),
    ("n02233338", "cockroach", "cockroach, roach"),
    ("n02236044", "mantis", "mantis, mantid"),
    ("n02256656", "cicada", "cicada, cicala"),
    ("n02259212", "leafhopper", "leafhopper"),
    ("n02264363", "lacewing", "lacewing, lacewing fly"),
    ("n02268443", "dragonfly", "dragonfly, darning needle, devil's darning needle, sewing needle, snake feeder, snake doctor, mosquito hawk, skeeter hawk"),
    ("n02268853", "damselfly", "damselfly"),
    ("n02276258", "admiral", "admiral"),
    ("n02277742", "ringlet", "ringlet, ringlet butterfly"),
    ("n02279972", "monarch", "monarch, monarch butterfly, milkweed butterfly, Danaus plexippus"),
    ("n02280649", "cabbage butterfly", "cabbage butterfly"),
    ("n02281406", "sulphur butterfly", "sulphur butterfly, sulfur butterfly"),
    ("n02281787", "lycaenid", "lycaenid, lycaenid butterfly"),
    ("n02317335", "starfish", "starfish, sea star"),
    ("n02319095", "sea urchin", "sea urchin"),
    ("n02321529", "sea cucumber", "sea cucumber, holothurian"),
    ("n02325366", "wood rabbit", "wood rabbit, cottontail, cottontail rabbit"),
    ("n02326432", "hare", "hare"),
    ("n02328150", "Angora", "Angora, Angora rabbit"),
    ("n02342885", "hamster", "hamster"),
    ("n02346627", "porcupine", "porcupine, hedgehog"),
    ("n02356798", "fox squirrel", "fox squirrel, eastern fox squirrel, Sciurus niger"),
    ("n02361337", "marmot", "marmot"),
    ("n02363005", "beaver", "beaver"),
    ("n02364673", "guinea pig", "guinea pig, Cavia cobaya"),
    ("n02389026", "sorrel", "sorrel"),
    ("n02391049", "zebra", "zebra"),
    ("n02395406", "hog", "hog, pig, grunter, squealer, Sus scrofa"),
    ("n02396427", "wild boar", "wild boar, boar, Sus scrofa"),
    ("n02397096", "warthog", "warthog"),
    ("n02398521", "hippopotamus", "hippopotamus, hippo, river horse, Hippopotamus amphibius"),
    ("n02403003", "ox", "ox"),
    ("n02408429", "water buffalo", "water buffalo, water ox, Asiatic buffalo, Bubalus bubalis"),
    ("n02410509", "bison", "bison"),
    ("n02412080", "ram", "ram, tup"),
    ("n02415577", "bighorn", "bighorn, bighorn sheep, cimarron, Rocky Mountain bighorn, Rocky Mountain sheep, Ovis canadensis"),
    ("n02417914", "ibex", "ibex, Capra ibex"),
    ("n02422106", "hartebeest", "hartebeest"),
    ("n02422699", "impala", "impala, Aepyceros melampus"),
    ("n02423022", "gazelle", "gazelle"),
    ("n02437312", "Arabian camel", "Arabian camel, dromedary, Camelus dromedarius"),
    ("n02437616", "llama", "llama"),
    ("n02441942", "weasel", "weasel"),
    ("n02442845", "mink", "mink"),
    ("n02443114", "polecat", "polecat, fitch, foulmart, foumart, Mustela putorius"),
    ("n02443484", "black-footed ferret", "black-footed ferret, ferret, Mustela nigripes"),
    ("n02444819", "otter", "otter"),
    ("n02445715", "skunk", "skunk, polecat, wood pussy"),
    ("n02447366", "badger", "badger"),
    ("n02454379", "armadillo", "armadillo"),
    ("n02457408", "three-toed sloth", "three-toed sloth, ai, Bradypus tridactylus"),
    ("n02480495", "orangutan", "orangutan, orang, orangutang, Pongo pygmaeus"),
    ("n02480855", "gorilla", "gorilla, Gorilla gorilla"),
    ("n02481823", "chimpanzee", "chimpanzee, chimp, Pan troglodytes"),
    ("n02483362", "gibbon", "gibbon, Hylobates lar"),
    ("n02483708", "siamang", "siamang, Hylobates syndactylus, Symphalangus syndactylus"),
    ("n02484975", "guenon", "guenon, guenon monkey"),
    ("n02486261", "patas", "patas, hussar monkey, Erythrocebus patas"),
    ("n02486410", "baboon", "baboon"),
    ("n02487347", "macaque", "macaque"),
    ("n02488291", "langur", "langur"),
    ("n02488702", "colobus", "colobus, colobus monkey"),
    ("n02489166", "proboscis monkey", "proboscis monkey, Nasalis larvatus"),
    ("n02490219", "marmoset", "marmoset"),
    ("n02492035", "capuchin", "capuchin, ringtail, Cebus capucinus"),
    ("n02492660", "howler monkey", "howler monkey, howler"),
    ("n02493509", "titi", "titi, titi monkey"),
    ("n02493793", "spider monkey", "spider monkey, Ateles geoffroyi"),
    ("n02494079", "squirrel monkey", "squirrel monkey, Saimiri sciureus"),
    ("n02497673", "Madagascar cat", "Madagascar cat, ring-tailed lemur, Lemur catta"),
    ("n02500267", "indri", "indri, indris, Indri indri, Indri brevicaudatus"),
    ("n02504013", "Indian elephant", "Indian elephant, Elephas maximus"),
    ("n02504458", "African elephant", "African elephant, Loxodonta africana"),
    ("n02509815", "lesser panda", "lesser panda, red panda, panda, bear cat, cat bear, Ailurus fulgens"),
    ("n02510455", "giant panda", "giant panda, panda, panda bear, coon bear, Ailuropoda melanoleuca"),
    ("n02514041", "barracouta", "barracouta, snoek"),
    ("n02526121", "eel", "eel"),
    ("n02536864", "coho", "coho, cohoe, coho salmon, blue jack, silver salmon, Oncorhynchus kisutch"),
    ("n02606052", "rock beauty", "rock beauty, Holocanthus tricolor"),
    ("n02607072", "anemone fish", "anemone fish"),
    ("n02640242", "sturgeon", "sturgeon"),
    ("n02641379", "gar", "gar, garfish, garpike, billfish, Lepisosteus osseus"),
    ("n02643566", "lionfish", "lionfish"),
    ("n02655020", "puffer", "puffer, pufferfish, blowfish, globefish"),
    ("n02666196", "abacus", "abacus"),
    ("n02667093", "abaya", "abaya"),
    ("n02669723", "academic gown", "academic gown, academic robe, judge's robe"),
    ("n02672831", "accordion", "accordion, piano accordion, squeeze box"),
    ("n02676566", "acoustic guitar", "acoustic guitar"),
    ("n02687172", "aircraft carrier", "aircraft carrier, carrier, flattop, attack aircraft carrier"),
    ("n02690373", "airliner", "airliner"),
    ("n02692877", "airship", "airship, dirigible"),
    ("n02699494", "altar", "altar"),
    ("n02701002", "ambulance", "ambulance"),
    ("n02704792", "amphibian", "amphibian, amphibious vehicle"),
    ("n02708093", "analog clock", "analog clock"),
    ("n02727426", "apiary", "apiary, bee house"),
    ("n02730930", "apron", "apron"),
    ("n02747177", "ashcan", "ashcan, trash can, garbage can, wastebin, ash bin, ash-bin, ashbin, dustbin, trash barrel, trash bin"),
    ("n02749479", "assault rifle", "assault rifle, assault gun"),
    ("n02769748", "backpack", "backpack, back pack, knapsack, packsack, rucksack, haversack"),
    ("n02776631", "bakery", "bakery, bakeshop, bakehouse"),
    ("n02777292", "balance beam", "balance beam, beam"),
    ("n02782093", "balloon", "balloon"),
    ("n02783161", "ballpoint", "ballpoint, ballpoint pen, ballpen, Biro"),
    ("n02786058", "Band Aid", "Band Aid"),
    ("n02787622", "banjo", "banjo"),
    ("n02788148", "bannister", "bannister, banister, balustrade, balusters, handrail"),
    ("n02790996", "barbell", "barbell"),
    ("n02791124", "barber chair", "barber chair"),
    ("n02791270", "barbershop", "barbershop"),
    ("n02793495", "barn", "barn"),
    ("n02794156", "barometer", "barometer"),
    ("n02795169", "barrel", "barrel, cask"),
    ("n02797295", "barrow", "barrow, garden cart, lawn cart, wheelbarrow"),
    ("n02799071", "baseball", "baseball"),
    ("n02802426", "basketball", "basketball"),
    ("n02804414", "bassinet", "bassinet"),
    ("n02804610", "bassoon", "bassoon"),
    ("n02807133", "bathing cap", "bathing cap, swimming cap"),
    ("n02808304", "bath towel", "bath towel"),
    ("n02808440", "bathtub", "bathtub, bathing tub, bath, tub"),
    ("n02814533", "beach wagon", "beach wagon, station wagon, wagon, estate car, beach waggon, station waggon, waggon"),
    ("n02814860", "beacon", "beacon, lighthouse, beacon light, pharos"),
    ("n02815834", "beaker", "beaker"),
    ("n02817516", "bearskin", "bearskin, busby, shako"),
    ("n02823428", "beer bottle", "beer bottle"),
    ("n02823750", "beer glass", "beer glass"),
    ("n02825657", "bell cote", "bell cote, bell cot"),
    ("n02834397", "bib", "bib"),
    ("n02835271", "bicycle-built-for-two", "bicycle-built-for-two, tandem bicycle, tandem"),
    ("n02837789", "bikini", "bikini, two-piece"),
    ("n02840245", "binder", "binder, ring-binder"),
    ("n02841315", "binoculars", "binoculars, field glasses, opera glasses"),
    ("n02843684", "birdhouse", "birdhouse"),
    ("n02859443", "boathouse", "boathouse"),
    ("n02860847", "bobsled", "bobsled, bobsleigh, bob"),
    ("n02865351", "bolo tie", "bolo tie, bolo, bola tie, bola"),
    ("n02869837", "bonnet", "bonnet, poke bonnet"),
    ("n02870880", "bookcase", "bookcase"),
    ("n02871525", "bookshop", "bookshop, bookstore, bookstall"),
    ("n02877765", "bottlecap", "bottlecap"),
    ("n02879718", "bow", "bow"),
    ("n02883205", "bow tie", "bow tie, bow-tie, bowtie"),
    ("n02892201", "brass", "brass, memorial tablet, plaque"),
    ("n02892767", "brassiere", "brassiere, bra, bandeau"),
    ("n02894605", "breakwater", "breakwater, groin, groyne, mole, bulwark, seawall, jetty"),
    ("n02895154", "breastplate", "breastplate, aegis, egis"),
    ("n02906734", "broom", "broom"),
    ("n02909870", "bucket", "bucket, pail"),
    ("n02910353", "buckle", "buckle"),
    ("n02916936", "bulletproof vest", "bulletproof vest"),
    ("n02917067", "bullet train", "bullet train, bullet"),
    ("n02927161", "butcher shop", "butcher shop, meat market"),
    ("n02930766", "cab", "cab, hack, taxi, taxicab"),
    ("n02939185", "caldron", "caldron, cauldron"),
    ("n02948072", "candle", "candle, taper, wax light"),
    ("n02950826", "cannon", "cannon"),
    ("n02951358", "canoe", "canoe"),
    ("n02951585", "can opener", "can opener, tin opener"),
    ("n02963159", "cardigan", "cardigan"),
    ("n02965783", "car mirror", "car mirror"),
    ("n02966193", "carousel", "carousel, carrousel, merry-go-round, roundabout, whirligig"),
    ("n02966687", "carpenter's kit", "carpenter's kit, tool kit"),
    ("n02971356", "carton", "carton"),
    ("n02974003", "car wheel", "car wheel"),
    ("n02977058", "cash machine", "cash machine, cash dispenser, automated teller machine, automatic teller machine, automated teller, automatic teller, ATM"),
    ("n02978881", "cassette", "cassette"),
    ("n02979186", "cassette player", "cassette player"),
    ("n02980441", "castle", "castle"),
    ("n02981792", "catamaran", "catamaran"),
    ("n02988304", "CD player", "CD player"),
    ("n02992211", "cello", "cello, violoncello"),
    ("n02992529", "cellular telephone", "cellular telephone, cellular phone, cellphone, cell, mobile phone"),
    ("n02999410", "chain", "chain"),
    ("n03000134", "chainlink fence", "chainlink fence"),
    ("n03000247", "chain mail", "chain mail, ring mail, mail, chain armor, chain armour, ring armor, ring armour"),
    ("n03000684", "chain saw", "chain saw, chainsaw"),
    ("n03014705", "chest", "chest"),
    ("n03016953", "chiffonier", "chiffonier, commode"),
    ("n03017168", "chime", "chime, bell, gong"),
    ("n03018349", "china cabinet", "china cabinet, china closet"),
    ("n03026506", "Christmas stocking", "Christmas stocking"),
    ("n03028079", "church", "church, church building"),
    ("n03032252", "cinema", "cinema, movie theater, movie theatre, movie house, picture palace"),
    ("n03041632", "cleaver", "cleaver, meat cleaver, chopper"),
    ("n03042490", "cliff dwelling", "cliff dwelling"),
    ("n03045698", "cloak", "cloak"),
    ("n03047690", "clog", "clog, geta, patten, sabot"),
    ("n03062245", "cocktail shaker", "cocktail shaker"),
    ("n03063599", "coffee mug", "coffee mug"),
    ("n03063689", "coffeepot", "coffeepot"),
    ("n03065424", "coil", "coil, spiral, volute, whorl, helix"),
    ("n03075370", "combination lock", "combination lock"),
    ("n03085013", "computer keyboard", "computer keyboard, keypad"),
    ("n03089624", "confectionery", "confectionery, confectionary, candy store"),
    ("n03095699", "container ship", "container ship, containership, container vessel"),
    ("n03100240", "convertible", "convertible"),
    ("n03109150", "corkscrew", "corkscrew, bottle screw"),
    ("n03110669", "cornet", "cornet, horn, trumpet, trump"),
    ("n03124043", "cowboy boot", "cowboy boot"),
    ("n03124170", "cowboy hat", "cowboy hat, ten-gallon hat"),
    ("n03125729", "cradle", "cradle"),
    ("n03126707", "crane", "crane"),
    ("n03127747", "crash helmet", "crash helmet"),
    ("n03127925", "crate", "crate"),
    ("n03131574", "crib", "crib, cot"),
    ("n03133878", "Crock Pot", "Crock Pot"),
    ("n03134739", "croquet ball", "croquet ball"),
    ("n03141823", "crutch", "crutch"),
    ("n03146219", "cuirass", "cuirass"),
    ("n03160309", "dam", "dam, dike, dyke"),
    ("n03179701", "desk", "desk"),
    ("n03180011", "desktop computer", "desktop computer"),
    ("n03187595", "dial telephone", "dial telephone, dial phone"),
    ("n03188531", "diaper", "diaper, nappy, napkin"),
    ("n03196217", "digital clock", "digital clock"),
    ("n03197337", "digital watch", "digital watch"),
    ("n03201208", "dining table", "dining table, board"),
    ("n03207743", "dishrag", "dishrag, dishcloth"),
    ("n03207941", "dishwasher", "dishwasher, dish washer, dishwashing machine"),
    ("n03208938", "disk brake", "disk brake, disc brake"),
    ("n03216828", "dock", "dock, dockage, docking facility"),
    ("n03218198", "dogsled", "dogsled, dog sled, dog sleigh"),
    ("n03220513", "dome", "dome"),
    ("n03223299", "doormat", "doormat, welcome mat"),
    ("n03240683", "drilling platform", "drilling platform, offshore rig"),
    ("n03249569", "drum", "drum, membranophone, tympan"),
    ("n03250847", "drumstick", "drumstick"),
    ("n03255030", "dumbbell", "dumbbell"),
    ("n03259280", "Dutch oven", "Dutch oven"),
    ("n03271574", "electric fan", "electric fan, blower"),
    ("n03272010", "electric guitar", "electric guitar"),
    ("n03272562", "electric locomotive", "electric locomotive"),
    ("n03290653", "entertainment center", "entertainment center"),
    ("n03291819", "envelope", "envelope"),
    ("n03297495", "espresso maker", "espresso maker"),
    ("n03314780", "face powder", "face powder"),
    ("n03325584", "feather boa", "feather boa, boa"),
    ("n03337140", "file", "file, file cabinet, filing cabinet"),
    ("n03344393", "fireboat", "fireboat"),
    ("n03345487", "fire engine", "fire engine, fire truck"),
    ("n03347037", "fire screen", "fire screen, fireguard"),
    ("n03355925", "flagpole", "flagpole, flagstaff"),
    ("n03372029", "flute", "flute, transverse flute"),
    ("n03376595", "folding chair", "folding chair"),
    ("n03379051", "football helmet", "football helmet"),
    ("n03384352", "forklift", "forklift"),
    ("n03388043", "fountain", "fountain"),
    ("n03388183", "fountain pen", "fountain pen"),
    ("n03388549", "four-poster", "four-poster"),
    ("n03393912", "freight car", "freight car"),
    ("n03394916", "French horn", "French horn, horn"),
    ("n03400231", "frying pan", "frying pan, frypan, skillet"),
    ("n03404251", "fur coat", "fur coat"),
    ("n03417042", "garbage truck", "garbage truck, dustcart"),
    ("n03424325", "gasmask", "gasmask, respirator, gas helmet"),
    ("n03425413", "gas pump", "gas pump, gasoline pump, petrol pump, island dispenser"),
    ("n03443371", "goblet", "goblet"),
    ("n03444034", "go-kart", "go-kart"),
    ("n03445777", "golf ball", "golf ball"),
    ("n03445924", "golfcart", "golfcart, golf cart"),
    ("n03447447", "gondola", "gondola"),
    ("n03447721", "gong", "gong, tam-tam"),
    ("n03450230", "gown", "gown"),
    ("n03452741", "grand piano", "grand piano, grand"),
    ("n03457902", "greenhouse", "greenhouse, nursery, glasshouse"),
    ("n03459775", "grille", "grille, radiator grille"),
    ("n03461385", "grocery store", "grocery store, grocery, food market, market"),
    ("n03467068", "guillotine", "guillotine"),
    ("n03476684", "hair slide", "hair slide"),
    ("n03476991", "hair spray", "hair spray"),
    ("n03478589", "half track", "half track"),
    ("n03481172", "hammer", "hammer"),
    ("n03482405", "hamper", "hamper"),
    ("n03483316", "hand blower", "hand blower, blow dryer, blow drier, hair dryer, hair drier"),
    ("n03485407", "hand-held computer", "hand-held computer, hand-held microcomputer"),
    ("n03485794", "handkerchief", "handkerchief, hankie, hanky, hankey"),
    ("n03492542", "hard disc", "hard disc, hard disk, fixed disk"),
    ("n03494278", "harmonica", "harmonica, mouth organ, harp, mouth harp"),
    ("n03495258", "harp", "harp"),
    ("n03496892", "harvester", "harvester, reaper"),
    ("n03498962", "hatchet", "hatchet"),
    ("n03527444", "holster", "holster"),
    ("n03529860", "home theater", "home theater, home theatre"),
    ("n03530642", "honeycomb", "honeycomb"),
    ("n03532672", "hook", "hook, claw"),
    ("n03534580", "hoopskirt", "hoopskirt, crinoline"),
    ("n03535780", "horizontal bar", "horizontal bar, high bar"),
    ("n03538406", "horse cart", "horse cart, horse-cart"),
    ("n03544143", "hourglass", "hourglass"),
    ("n03584254", "iPod", "iPod"),
    ("n03584829", "iron", "iron, smoothing iron"),
    ("n03590841", "jack-o'-lantern", "jack-o'-lantern"),
    ("n03594734", "jean", "jean, blue jean, denim"),
    ("n03594945", "jeep", "jeep, landrover"),
    ("n03595614", "jersey", "jersey, T-shirt, tee shirt"),
    ("n03598930", "jigsaw puzzle", "jigsaw puzzle"),
    ("n03599486", "jinrikisha", "jinrikisha, ricksha, rickshaw"),
    ("n03602883", "joystick", "joystick"),
    ("n03617480", "kimono", "kimono"),
    ("n03623198", "knee pad", "knee pad"),
    ("n03627232", "knot", "knot"),
    ("n03630383", "lab coat", "lab coat, laboratory coat"),
    ("n03633091", "ladle", "ladle"),
    ("n03637318", "lampshade", "lampshade, lamp shade"),
    ("n03642806", "laptop", "laptop, laptop computer"),
    ("n03649909", "lawn mower", "lawn mower, mower"),
    ("n03657121", "lens cap", "lens cap, lens cover"),
    ("n03658185", "letter opener", "letter opener, paper knife, paperknife"),
    ("n03661043", "library", "library"),
    ("n03662601", "lifeboat", "lifeboat"),
    ("n03666591", "lighter", "lighter, light, igniter, ignitor"),
    ("n03670208", "limousine", "limousine, limo"),
    ("n03673027", "liner", "liner, ocean liner"),
    ("n03676483", "lipstick", "lipstick, lip rouge"),
    ("n03680355", "Loafer", "Loafer"),
    ("n03690938", "lotion", "lotion"),
    ("n03691459", "loudspeaker", "loudspeaker, speaker, speaker unit, loudspeaker system, speaker system"),
    ("n03692522", "loupe", "loupe, jeweler's loupe"),
    ("n03697007", "lumbermill", "lumbermill, sawmill"),
    ("n03706229", "magnetic compass", "magnetic compass"),
    ("n03709823", "mailbag", "mailbag, postbag"),
    ("n03710193", "mailbox", "mailbox, letter box"),
    ("n03710637", "maillot", "maillot"),
    ("n03710721", "maillot", "maillot, tank suit"),
    ("n03717622", "manhole cover", "manhole cover"),
    ("n03720891", "maraca", "maraca"),
    ("n03721384", "marimba", "marimba, xylophone"),
    ("n03724870", "mask", "mask"),
    ("n03729826", "matchstick", "matchstick"),
    ("n03733131", "maypole", "maypole"),
    ("n03733281", "maze", "maze, labyrinth"),
    ("n03733805", "measuring cup", "measuring cup"),
    ("n03742115", "medicine chest", "medicine chest, medicine cabinet"),
    ("n03743016", "megalith", "megalith, megalithic structure"),
    ("n03759954", "microphone", "microphone, mike"),
    ("n03761084", "microwave", "microwave, microwave oven"),
    ("n03763968", "military uniform", "military uniform"),
    ("n03764736", "milk can", "milk can"),
    ("n03769881", "minibus", "minibus"),
    ("n03770439", "miniskirt", "miniskirt, mini"),
    ("n03770679", "minivan", "minivan"),
    ("n03773504", "missile", "missile"),
    ("n03775071", "mitten", "mitten"),
    ("n03775546", "mixing bowl", "mixing bowl"),
    ("n03776460", "mobile home", "mobile home, manufactured home"),
    ("n03777568", "Model T", "Model T"),
    ("n03777754", "modem", "modem"),
    ("n03781244", "monastery", "monastery"),
    ("n03782006", "monitor", "monitor"),
    ("n03785016", "moped", "moped"),
    ("n03786901", "mortar", "mortar"),
    ("n03787032", "mortarboard", "mortarboard"),
    ("n03788195", "mosque", "mosque"),
    ("n03788365", "mosquito net", "mosquito net"),
    ("n03791053", "motor scooter", "motor scooter, scooter"),
    ("n03792782", "mountain bike", "mountain bike, all-terrain bike, off-roader"),
    ("n03792972", "mountain tent", "mountain tent"),
    ("n03793489", "mouse", "mouse, computer mouse"),
    ("n03794056", "mousetrap", "mousetrap"),
    ("n03796401", "moving van", "moving van"),
    ("n03803284", "muzzle", "muzzle"),
    ("n03804744", "nail", "nail"),
    ("n03814639", "neck brace", "neck brace"),
    ("n03814906", "necklace", "necklace"),
    ("n03825788", "nipple", "nipple"),
    ("n03832673", "notebook", "notebook, notebook computer"),
    ("n03837869", "obelisk", "obelisk"),
    ("n03838899", "oboe", "oboe, hautboy, hautbois"),
    ("n03840681", "ocarina", "ocarina, sweet potato"),
    ("n03841143", "odometer", "odometer, hodometer, mileometer, milometer"),
    ("n03843555", "oil filter", "oil filter"),
    ("n03854065", "organ", "organ, pipe organ"),
    ("n03857828", "oscilloscope", "oscilloscope, scope, cathode-ray oscilloscope, CRO"),
    ("n03866082", "overskirt", "overskirt"),
    ("n03868242", "oxcart", "oxcart"),
    ("n03868863", "oxygen mask", "oxygen mask"),
    ("n03871628", "packet", "packet"),
    ("n03873416", "paddle", "paddle, boat paddle"),
    ("n03874293", "paddlewheel", "paddlewheel, paddle wheel"),
    ("n03874599", "padlock", "padlock"),
    ("n03876231", "paintbrush", "paintbrush"),
    ("n03877472", "pajama", "pajama, pyjama, pj's, jammies"),
    ("n03877845", "palace", "palace"),
    ("n03884397", "panpipe", "panpipe, pandean pipe, syrinx"),
    ("n03887697", "paper towel", "paper towel"),
    ("n03888257", "parachute", "parachute, chute"),
    ("n03888605", "parallel bars", "parallel bars, bars"),
    ("n03891251", "park bench", "park bench"),
    ("n03891332", "parking meter", "parking meter"),
    ("n03895866", "passenger car", "passenger car, coach, carriage"),
    ("n03899768", "patio", "patio, terrace"),
    ("n03902125", "pay-phone", "pay-phone, pay-station"),
    ("n03903868", "pedestal", "pedestal, plinth, footstall"),
    ("n03908618", "pencil box", "pencil box, pencil case"),
    ("n03908714", "pencil sharpener", "pencil sharpener"),
    ("n03916031", "perfume", "perfume, essence"),
    ("n03920288", "Petri dish", "Petri dish"),
    ("n03924679", "photocopier", "photocopier"),
    ("n03929660", "pick", "pick, plectrum, plectron"),
    ("n03929855", "pickelhaube", "pickelhaube"),
    ("n03930313", "picket fence", "picket fence, paling"),
    ("n03930630", "pickup", "pickup, pickup truck"),
    ("n03933933", "pier", "pier"),
    ("n03935335", "piggy bank", "piggy bank, penny bank"),
    ("n03937543", "pill bottle", "pill bottle"),
    ("n03938244", "pillow", "pillow"),
    ("n03942813", "ping-pong ball", "ping-pong ball"),
    ("n03944341", "pinwheel", "pinwheel"),
    ("n03947888", "pirate", "pirate, pirate ship"),
    ("n03950228", "pitcher", "pitcher, ewer"),
    ("n03954731", "plane", "plane, carpenter's plane, woodworking plane"),
    ("n03956157", "planetarium", "planetarium"),
    ("n03958227", "plastic bag", "plastic bag"),
    ("n03961711", "plate rack", "plate rack"),
    ("n03967562", "plow", "plow, plough"),
    ("n03970156", "plunger", "plunger, plumber's helper"),
    ("n03976467", "Polaroid camera", "Polaroid camera, Polaroid Land camera"),
    ("n03976657", "pole", "pole"),
    ("n03977966", "police van", "police van, police wagon, paddy wagon, patrol wagon, wagon, black Maria"),
    ("n03980874", "poncho", "poncho"),
    ("n03982430", "pool table", "pool table, billiard table, snooker table"),
    ("n03983396", "pop bottle", "pop bottle, soda bottle"),
    ("n03991062", "pot", "pot, flowerpot"),
    ("n03992509", "potter's wheel", "potter's wheel"),
    ("n03995372", "power drill", "power drill"),
    ("n03998194", "prayer rug", "prayer rug, prayer mat"),
    ("n04004767", "printer", "printer"),
    ("n04005630", "prison", "prison, prison house"),
    ("n04008634", "projectile", "projectile, missile"),
    ("n04009552", "projector", "projector"),
    ("n04019541", "puck", "puck, hockey puck"),
    ("n04023962", "punching bag", "punching bag, punch bag, punching ball, punchball"),
    ("n04026417", "purse", "purse"),
    ("n04033901", "quill", "quill, quill pen"),
    ("n04033995", "quilt", "quilt, comforter, comfort, puff"),
    ("n04037443", "racer", "racer, race car, racing car"),
    ("n04039381", "racket", "racket, racquet"),
    ("n04040759", "radiator", "radiator"),
    ("n04041544", "radio", "radio, wireless"),
    ("n04044716", "radio telescope", "radio telescope, radio reflector"),
    ("n04049303", "rain barrel", "rain barrel"),
    ("n04065272", "recreational vehicle", "recreational vehicle, RV, R.V."),
    ("n04067472", "reel", "reel"),
    ("n04069434", "reflex camera", "reflex camera"),
    ("n04070727", "refrigerator", "refrigerator, icebox"),
    ("n04074963", "remote control", "remote control, remote"),
    ("n04081281", "restaurant", "restaurant, eating house, eating place, eatery"),
    ("n04086273", "revolver", "revolver, six-gun, six-shooter"),
    ("n04090263", "rifle", "rifle"),
    ("n04099969", "rocking chair", "rocking chair, rocker"),
    ("n04111531", "rotisserie", "rotisserie"),
    ("n04116512", "rubber eraser", "rubber eraser, rubber, pencil eraser"),
    ("n04118538", "rugby ball", "rugby ball"),
    ("n04118776", "rule", "rule, ruler"),
    ("n04120489", "running shoe", "running shoe"),
    ("n04125021", "safe", "safe"),
    ("n04127249", "safety pin", "safety pin"),
    ("n04131690", "saltshaker", "saltshaker, salt shaker"),
    ("n04133789", "sandal", "sandal"),
    ("n04136333", "sarong", "sarong"),
    ("n04141076", "sax", "sax, saxophone"),
    ("n04141327", "scabbard", "scabbard"),
    ("n04141975", "scale", "scale, weighing machine"),
    ("n04146614", "school bus", "school bus"),
    ("n04147183", "schooner", "schooner"),
    ("n04149813", "scoreboard", "scoreboard"),
    ("n04152593", "screen", "screen, CRT screen"),
    ("n04153751", "screw", "screw"),
    ("n04154565", "screwdriver", "screwdriver"),
    ("n04162706", "seat belt", "seat belt, seatbelt"),
    ("n04179913", "sewing machine", "sewing machine"),
    ("n04192698", "shield", "shield, buckler"),
    ("n04200800", "shoe shop", "shoe shop, shoe-shop, shoe store"),
    ("n04201297", "shoji", "shoji"),
    ("n04204238", "shopping basket", "shopping basket"),
    ("n04204347", "shopping cart", "shopping cart"),
    ("n04208210", "shovel", "shovel"),
    ("n04209133", "shower cap", "shower cap"),
    ("n04209239", "shower curtain", "shower curtain"),
    ("n04228054", "ski", "ski"),
    ("n04229816", "ski mask", "ski mask"),
    ("n04235860", "sleeping bag", "sleeping bag"),
    ("n04238763", "slide rule", "slide rule, slipstick"),
    ("n04239074", "sliding door", "sliding door"),
    ("n04243546", "slot", "slot, one-armed bandit"),
    ("n04251144", "snorkel", "snorkel"),
    ("n04252077", "snowmobile", "snowmobile"),
    ("n04252225", "snowplow", "snowplow, snowplough"),
    ("n04254120", "soap dispenser", "soap dispenser"),
    ("n04254680", "soccer ball", "soccer ball"),
    ("n04254777", "sock", "sock"),
    ("n04258138", "solar dish", "solar dish, solar collector, solar furnace"),
    ("n04259630", "sombrero", "sombrero"),
    ("n04263257", "soup bowl", "soup bowl"),
    ("n04264628", "space bar", "space bar"),
    ("n04265275", "space heater", "space heater"),
    ("n04266014", "space shuttle", "space shuttle"),
    ("n04270147", "spatula", "spatula"),
    ("n04273569", "speedboat", "speedboat"),
    ("n04275548", "spider web", "spider web, spider's web"),
    ("n04277352", "spindle", "spindle"),
    ("n04285008", "sports car", "sports car, sport car"),
    ("n04286575", "spotlight", "spotlight, spot"),
    ("n04296562", "stage", "stage"),
    ("n04310018", "steam locomotive", "steam locomotive"),
    ("n04311004", "steel arch bridge", "steel arch bridge"),
    ("n04311174", "steel drum", "steel drum"),
    ("n04317175", "stethoscope", "stethoscope"),
    ("n04325704", "stole", "stole"),
    ("n04326547", "stone wall", "stone wall"),
    ("n04328186", "stopwatch", "stopwatch, stop watch"),
    ("n04330267", "stove", "stove"),
    ("n04332243", "strainer", "strainer"),
    ("n04335435", "streetcar", "streetcar, tram, tramcar, trolley, trolley car"),
    ("n04336792", "stretcher", "stretcher"),
    ("n04344873", "studio couch", "studio couch, day bed"),
    ("n04346328", "stupa", "stupa, tope"),
    ("n04347754", "submarine", "submarine, pigboat, sub, U-boat"),
    ("n04350905", "suit", "suit, suit of clothes"),
    ("n04355338", "sundial", "sundial"),
    ("n04355933", "sunglass", "sunglass"),
    ("n04356056", "sunglasses", "sunglasses, dark glasses, shades"),
    ("n04357314", "sunscreen", "sunscreen, sunblock, sun blocker"),
    ("n04366367", "suspension bridge", "suspension bridge"),
    ("n04367480", "swab", "swab, swob, mop"),
    ("n04370456", "sweatshirt", "sweatshirt"),
    ("n04371430", "swimming trunks", "swimming trunks, bathing trunks"),
    ("n04371774", "swing", "swing"),
    ("n04372370", "switch", "switch, electric switch, electrical switch"),
    ("n04376876", "syringe", "syringe"),
    ("n04380533", "table lamp", "table lamp"),
    ("n04389033", "tank", "tank, army tank, armored combat vehicle, armoured combat vehicle"),
    ("n04392985", "tape player", "tape player"),
    ("n04398044", "teapot", "teapot"),
    ("n04399382", "teddy", "teddy, teddy bear"),
    ("n04404412", "television", "television, television system"),
    ("n04409515", "tennis ball", "tennis ball"),
    ("n04417672", "thatch", "thatch, thatched roof"),
    ("n04418357", "theater curtain", "theater curtain, theatre curtain"),
    ("n04423845", "thimble", "thimble"),
    ("n04428191", "thresher", "thresher, thrasher, threshing machine"),
    ("n04429376", "throne", "throne"),
    ("n04435653", "tile roof", "tile roof"),
    ("n04442312", "toaster", "toaster"),
    ("n04443257", "tobacco shop", "tobacco shop, tobacconist shop, tobacconist"),
    ("n04447861", "toilet seat", "toilet seat"),
    ("n04456115", "torch", "torch"),
    ("n04458633", "totem pole", "totem pole"),
    ("n04461696", "tow truck", "tow truck, tow car, wrecker"),
    ("n04462240", "toyshop", "toyshop"),
    ("n04465501", "tractor", "tractor"),
    ("n04467665", "trailer truck", "trailer truck, tractor trailer, trucking rig, rig, articulated lorry, semi"),
    ("n04476259", "tray", "tray"),
    ("n04479046", "trench coat", "trench coat"),
    ("n04482393", "tricycle", "tricycle, trike, velocipede"),
    ("n04483307", "trimaran", "trimaran"),
    ("n04485082", "tripod", "tripod"),
    ("n04486054", "triumphal arch", "triumphal arch"),
    ("n04487081", "trolleybus", "trolleybus, trolley coach, trackless trolley"),
    ("n04487394", "trombone", "trombone"),
    ("n04493381", "tub", "tub, vat"),
    ("n04501370", "turnstile", "turnstile"),
    ("n04505470", "typewriter keyboard", "typewriter keyboard"),
    ("n04507155", "umbrella", "umbrella"),
    ("n04509417", "unicycle", "unicycle, monocycle"),
    ("n04515003", "upright", "upright, upright piano"),
    ("n04517823", "vacuum", "vacuum, vacuum cleaner"),
    ("n04522168", "vase", "vase"),
    ("n04523525", "vault", "vault"),
    ("n04525038", "velvet", "velvet"),
    ("n04525305", "vending machine", "vending machine"),
    ("n04532106", "vestment", "vestment"),
    ("n04532670", "viaduct", "viaduct"),
    ("n04536866", "violin", "violin, fiddle"),
    ("n04540053", "volleyball", "volleyball"),
    ("n04542943", "waffle iron", "waffle iron"),
    ("n04548280", "wall clock", "wall clock"),
    ("n04548362", "wallet", "wallet, billfold, notecase, pocketbook"),
    ("n04550184", "wardrobe", "wardrobe, closet, press"),
    ("n04552348", "warplane", "warplane, military plane"),
    ("n04553703", "washbasin", "washbasin, handbasin, washbowl, lavabo, wash-hand basin"),
    ("n04554684", "washer", "washer, automatic washer, washing machine"),
    ("n04557648", "water bottle", "water bottle"),
    ("n04560804", "water jug", "water jug"),
    ("n04562935", "water tower", "water tower"),
    ("n04579145", "whiskey jug", "whiskey jug"),
    ("n04579432", "whistle", "whistle"),
    ("n04584207", "wig", "wig"),
    ("n04589890", "window screen", "window screen"),
    ("n04590129", "window shade", "window shade"),
    ("n04591157", "Windsor tie", "Windsor tie"),
    ("n04591713", "wine bottle", "wine bottle"),
    ("n04592741", "wing", "wing"),
    ("n04596742", "wok", "wok"),
    ("n04597913", "wooden spoon", "wooden spoon"),
    ("n04599235", "wool", "wool, woolen, woollen"),
    ("n04604644", "worm fence", "worm fence, snake fence, snake-rail fence, Virginia fence"),
    ("n04606251", "wreck", "wreck"),
    ("n04612504", "yawl", "yawl"),
    ("n04613696", "yurt", "yurt"),
    ("n06359193", "web site", "web site, website, internet site, site"),
    ("n06596364", "comic book", "comic book"),
    ("n06785654", "crossword puzzle", "crossword puzzle, crossword"),
    ("n06794110", "street sign", "street sign"),
    ("n06874185", "traffic light", "traffic light, traffic signal, stoplight"),
    ("n07248320", "book jacket", "book jacket, dust cover, dust jacket, dust wrapper"),
    ("n07565083", "menu", "menu"),
    ("n07579787", "plate", "plate"),
    ("n07583066", "guacamole", "guacamole"),
    ("n07584110", "consomme", "consomme"),
    ("n07590611", "hot pot", "hot pot, hotpot"),
    ("n07613480", "trifle", "trifle"),
    ("n07614500", "ice cream", "ice cream, icecream"),
    ("n07615774", "ice lolly", "ice lolly, lolly, lollipop, popsicle"),
    ("n07684084", "French loaf", "French loaf"),
    ("n07693725", "bagel", "bagel, beigel"),
    ("n07695742", "pretzel", "pretzel"),
    ("n07697313", "cheeseburger", "cheeseburger"),
    ("n07697537", "hotdog", "hotdog, hot dog, red hot"),
    ("n07711569", "mashed potato", "mashed potato"),
    ("n07714571", "head cabbage", "head cabbage"),
    ("n07714990", "broccoli", "broccoli"),
    ("n07715103", "cauliflower", "cauliflower"),
    ("n07716358", "zucchini", "zucchini, courgette"),
    ("n07716906", "spaghetti squash", "spaghetti squash"),
    ("n07717410", "acorn squash", "acorn squash"),
    ("n07717556", "butternut squash", "butternut squash"),
    ("n07718472", "cucumber", "cucumber, cuke"),
    ("n07718747", "artichoke", "artichoke, globe artichoke"),
    ("n07720875", "bell pepper", "bell pepper"),
    ("n07730033", "cardoon", "cardoon"),
    ("n07734744", "mushroom", "mushroom"),
    ("n07742313", "Granny Smith", "Granny Smith"),
    ("n07745940", "strawberry", "strawberry"),
    ("n07747607", "orange", "orange"),
    ("n07749582", "lemon", "lemon"),
    ("n07753113", "fig", "fig"),
    ("n07753275", "pineapple", "pineapple, ananas"),
    ("n07753592", "banana", "banana"),
    ("n07754684", "jackfruit", "jackfruit, jak, jack"),
    ("n07760859", "custard apple", "custard apple"),
    ("n07768694", "pomegranate", "pomegranate"),
    ("n07802026", "hay", "hay"),
    ("n07831146", "carbonara", "carbonara"),
    ("n07836838", "chocolate sauce", "chocolate sauce, chocolate syrup"),
    ("n07860988", "dough", "dough"),
    ("n07871810", "meat loaf", "meat loaf, meatloaf"),
    ("n07873807", "pizza", "pizza, pizza pie"),
    ("n07875152", "potpie", "potpie"),
    ("n07880968", "burrito", "burrito"),
    ("n07892512", "red wine", "red wine"),
    ("n07920052", "espresso", "espresso"),
    ("n07930864", "cup", "cup"),
    ("n07932039", "eggnog", "eggnog"),
    ("n09193705", "alp", "alp"),
    ("n09229709", "bubble", "bubble"),
    ("n09246464", "cliff", "cliff, drop, drop-off"),
    ("n09256479", "coral reef", "coral reef"),
    ("n09288635", "geyser", "geyser"),
    ("n09332890", "lakeside", "lakeside, lakeshore"),
    ("n09399592", "promontory", "promontory, headland, head, foreland"),
    ("n09421951", "sandbar", "sandbar, sand bar"),
    ("n09428293", "seashore", "seashore, coast, seacoast, sea-coast"),
    ("n09468604", "valley", "valley, vale"),
    ("n09472597", "volcano", "volcano"),
    ("n09835506", "ballplayer", "ballplayer, baseball player"),
    ("n10148035", "groom", "groom, bridegroom"),
    ("n10565667", "scuba diver", "scuba diver"),
    ("n11879895", "rapeseed", "rapeseed"),
    ("n11939491", "daisy", "daisy"),
    ("n12057211", "yellow lady's slipper", "yellow lady's slipper, yellow lady-slipper, Cypripedium calceolus, Cypripedium parviflorum"),
    ("n12144580", "corn", "corn"),
    ("n12267677", "acorn", "acorn"),
    ("n12620546", "hip", "hip, rose hip, rosehip"),
    ("n12768682", "buckeye", "buckeye, horse chestnut, conker"),
    ("n12985857", "coral fungus", "coral fungus"),
    ("n12998815", "agaric", "agaric"),
    ("n13037406", "gyromitra", "gyromitra"),
    ("n13040303", "stinkhorn", "stinkhorn, carrion fungus"),
    ("n13044778", "earthstar", "earthstar"),
    ("n13052670", "hen-of-the-woods", "hen-of-the-woods, hen of the woods, Polyporus frondosus, Grifola frondosa"),
    ("n13054560", "bolete", "bolete"),
    ("n13133613", "ear", "ear, spike, capitulum"),
    ("n15075141", "toilet tissue", "toilet tissue, toilet paper, bathroom tissue"),
]
