from typing import Any

from infrahub.core.constants import (
    DEFAULT_KIND_MAX_LENGTH,
    DEFAULT_KIND_MIN_LENGTH,
    NAMESPACE_REGEX,
    AccountStatus,
    AccountType,
    AllowOverrideType,
    ArtifactStatus,
    BranchConflictKeep,
    BranchSupportType,
    ContentType,
    GeneratorInstanceStatus,
    GlobalPermissions,
    InfrahubKind,
    PermissionAction,
    PermissionDecision,
    RelationshipDeleteBehavior,
    RepositoryInternalStatus,
    RepositoryOperationalStatus,
    RepositorySyncStatus,
    Severity,
    ValidatorConclusion,
    ValidatorState,
)
from infrahub.proposed_change.constants import ProposedChangeState

# pylint: disable=too-many-lines

core_profile_schema_definition: dict[str, Any] = {
    "name": "Profile",
    "namespace": "Core",
    "include_in_menu": False,
    "icon": "mdi:shape-plus-outline",
    "description": "Base Profile in Infrahub.",
    "label": "Profile",
    "display_labels": ["profile_name__value"],
    "default_filter": "profile_name__value",
    "uniqueness_constraints": [["profile_name__value"]],
    "attributes": [
        {
            "name": "profile_name",
            "kind": "Text",
            "min_length": 3,
            "max_length": 32,
            "optional": False,
            "unique": True,
        },
        {
            "name": "profile_priority",
            "kind": "Number",
            "default_value": 1000,
            "optional": True,
        },
    ],
}

# -----------------------------------------------
# Menu Items
# -----------------------------------------------
generic_menu_item: dict[str, Any] = {
    "name": "Menu",
    "namespace": "Core",
    "include_in_menu": False,
    "description": "Element of the Menu",
    "label": "Menu",
    "hierarchical": True,
    "human_friendly_id": ["namespace__value", "name__value"],
    "display_labels": ["label__value"],
    "generate_profile": False,
    "attributes": [
        {"name": "namespace", "kind": "Text", "regex": NAMESPACE_REGEX, "order_weight": 1000},
        {"name": "name", "kind": "Text", "order_weight": 1000},
        {"name": "label", "kind": "Text", "optional": True, "order_weight": 2000},
        {"name": "kind", "kind": "Text", "optional": True, "order_weight": 2500},
        {"name": "path", "kind": "Text", "optional": True, "order_weight": 2500},
        {"name": "description", "kind": "Text", "optional": True, "order_weight": 3000},
        {"name": "icon", "kind": "Text", "optional": True, "order_weight": 4000},
        {"name": "protected", "kind": "Boolean", "default_value": False, "read_only": True, "order_weight": 5000},
        {"name": "order_weight", "kind": "Number", "default_value": 2000, "order_weight": 6000},
        {"name": "required_permissions", "kind": "List", "optional": True, "order_weight": 7000},
        {
            "name": "section",
            "kind": "Text",
            "enum": ["object", "internal"],
            "default_value": "object",
            "order_weight": 8000,
        },
    ],
}

menu_item: dict[str, Any] = {
    "name": "MenuItem",
    "namespace": "Core",
    "include_in_menu": False,
    "description": "Menu Item",
    "label": "Menu Item",
    "inherit_from": ["CoreMenu"],
    "generate_profile": False,
}

core_models: dict[str, Any] = {
    "generics": [
        {
            "name": "Node",
            "namespace": "Core",
            "include_in_menu": False,
            "description": "Base Node in Infrahub.",
            "label": "Node",
        },
        {
            "name": "Owner",
            "namespace": "Lineage",
            "description": "Any Entities that is responsible for some data.",
            "label": "Owner",
            "include_in_menu": False,
            "documentation": "/topics/metadata",
        },
        core_profile_schema_definition,
        {
            "name": "Source",
            "namespace": "Lineage",
            "description": "Any Entities that stores or produces data.",
            "label": "Source",
            "include_in_menu": False,
            "documentation": "/topics/metadata",
        },
        {
            "name": "Comment",
            "namespace": "Core",
            "description": "A comment on a Proposed Change",
            "label": "Comment",
            "display_labels": ["text__value"],
            "order_by": ["created_at__value"],
            "include_in_menu": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "text", "kind": "TextArea", "unique": False, "optional": False},
                {"name": "created_at", "kind": "DateTime", "optional": True},
            ],
            "relationships": [
                {
                    "name": "created_by",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "cardinality": "one",
                    "identifier": "comment__account",
                },
            ],
        },
        {
            "name": "Thread",
            "namespace": "Core",
            "description": "A thread on a Proposed Change",
            "label": "Thread",
            "order_by": ["created_at__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "include_in_menu": False,
            "attributes": [
                {"name": "label", "kind": "Text", "optional": True},
                {"name": "resolved", "kind": "Boolean", "default_value": False},
                {"name": "created_at", "kind": "DateTime", "optional": True},
            ],
            "relationships": [
                {
                    "name": "change",
                    "peer": InfrahubKind.PROPOSEDCHANGE,
                    "identifier": "proposedchange__thread",
                    "kind": "Parent",
                    "optional": False,
                    "cardinality": "one",
                },
                {
                    "name": "comments",
                    "peer": InfrahubKind.THREADCOMMENT,
                    "identifier": "thread__threadcomment",
                    "kind": "Component",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                },
                {
                    "name": "created_by",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "identifier": "thread__account",
                    "optional": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "cardinality": "one",
                },
            ],
        },
        {
            "name": "Group",
            "namespace": "Core",
            "description": "Generic Group Object.",
            "label": "Group",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["label__value"],
            "include_in_menu": False,
            "icon": "mdi:group",
            "hierarchical": True,
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "label", "kind": "Text", "optional": True},
                {"name": "description", "kind": "Text", "optional": True},
                {
                    "name": "group_type",
                    "kind": "Text",
                    "enum": ["default", "internal"],
                    "default_value": "default",
                    "optional": False,
                },
            ],
            "relationships": [
                {
                    "name": "members",
                    "peer": InfrahubKind.NODE,
                    "optional": True,
                    "identifier": "group_member",
                    "cardinality": "many",
                },
                {
                    "name": "subscribers",
                    "peer": InfrahubKind.NODE,
                    "optional": True,
                    "identifier": "group_subscriber",
                    "cardinality": "many",
                },
            ],
        },
        {
            "name": "Validator",
            "namespace": "Core",
            "description": "",
            "include_in_menu": False,
            "label": "Validator",
            "order_by": ["started_at__value"],
            "display_labels": ["label__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "label", "kind": "Text", "optional": True},
                {
                    "name": "state",
                    "kind": "Text",
                    "enum": ValidatorState.available_types(),
                    "default_value": ValidatorState.QUEUED.value,
                },
                {
                    "name": "conclusion",
                    "kind": "Text",
                    "enum": ValidatorConclusion.available_types(),
                    "default_value": ValidatorConclusion.UNKNOWN.value,
                },
                {"name": "completed_at", "kind": "DateTime", "optional": True},
                {"name": "started_at", "kind": "DateTime", "optional": True},
            ],
            "relationships": [
                {
                    "name": "proposed_change",
                    "peer": InfrahubKind.PROPOSEDCHANGE,
                    "kind": "Parent",
                    "optional": False,
                    "cardinality": "one",
                    "identifier": "proposed_change__validator",
                },
                {
                    "name": "checks",
                    "peer": "CoreCheck",
                    "kind": "Component",
                    "optional": True,
                    "cardinality": "many",
                    "identifier": "validator__check",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                },
            ],
        },
        {
            "name": "Check",
            "namespace": "Core",
            "description": "",
            "display_labels": ["label__value"],
            "include_in_menu": False,
            "label": "Check",
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "name", "kind": "Text", "optional": True},
                {"name": "label", "kind": "Text", "optional": True},
                {"name": "origin", "kind": "Text", "optional": False},
                {
                    "name": "kind",
                    "kind": "Text",
                    "regex": "^[A-Z][a-zA-Z0-9]+$",
                    "optional": False,
                    "min_length": DEFAULT_KIND_MIN_LENGTH,
                    "max_length": DEFAULT_KIND_MAX_LENGTH,
                },
                {"name": "message", "kind": "TextArea", "optional": True},
                {
                    "name": "conclusion",
                    "kind": "Text",
                    "enum": ValidatorConclusion.available_types(),
                    "default_value": ValidatorConclusion.UNKNOWN.value,
                    "optional": True,
                },
                {
                    "name": "severity",
                    "kind": "Text",
                    "enum": Severity.available_types(),
                    "default_value": Severity.INFO.value,
                    "optional": True,
                },
                {"name": "created_at", "kind": "DateTime", "optional": True},
            ],
            "relationships": [
                {
                    "name": "validator",
                    "peer": InfrahubKind.VALIDATOR,
                    "identifier": "validator__check",
                    "kind": "Parent",
                    "optional": False,
                    "cardinality": "one",
                },
            ],
        },
        {
            "name": "Transformation",
            "namespace": "Core",
            "description": "Generic Transformation Object.",
            "include_in_menu": False,
            "icon": "mdi:cog-transfer",
            "label": "Transformation",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["label__value"],
            "branch": BranchSupportType.AWARE.value,
            "documentation": "/topics/proposed-change",
            "uniqueness_constraints": [["name__value"]],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "label", "kind": "Text", "optional": True},
                {"name": "description", "kind": "Text", "optional": True},
                {"name": "timeout", "kind": "Number", "default_value": 10},
            ],
            "relationships": [
                {
                    "name": "query",
                    "peer": InfrahubKind.GRAPHQLQUERY,
                    "identifier": "graphql_query__transformation",
                    "kind": "Attribute",
                    "cardinality": "one",
                    "optional": False,
                },
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "cardinality": "one",
                    "identifier": "repository__transformation",
                    "optional": False,
                },
                {
                    "name": "tags",
                    "peer": InfrahubKind.TAG,
                    "kind": "Attribute",
                    "optional": True,
                    "cardinality": "many",
                },
            ],
        },
        {
            "name": "ArtifactTarget",
            "include_in_menu": False,
            "namespace": "Core",
            "description": "Extend a node to be associated with artifacts",
            "label": "Artifact Target",
            "relationships": [
                {
                    "name": "artifacts",
                    "peer": InfrahubKind.ARTIFACT,
                    "optional": True,
                    "cardinality": "many",
                    "kind": "Generic",
                    "identifier": "artifact__node",
                },
            ],
        },
        {
            "name": "TaskTarget",
            "include_in_menu": False,
            "namespace": "Core",
            "description": "Extend a node to be associated with tasks",
            "label": "Task Target",
        },
        {
            "name": "Webhook",
            "namespace": "Core",
            "description": "A webhook that connects to an external integration",
            "label": "Webhook",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "include_in_menu": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "uniqueness_constraints": [["name__value"]],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True, "order_weight": 1000},
                {"name": "description", "kind": "Text", "optional": True, "order_weight": 2000},
                {"name": "url", "kind": "URL", "order_weight": 3000},
                {
                    "name": "validate_certificates",
                    "kind": "Boolean",
                    "default_value": True,
                    "optional": True,
                    "order_weight": 5000,
                },
            ],
        },
        {
            "name": "GenericRepository",
            "namespace": "Core",
            "label": "Git Repository",
            "description": "A Git Repository integrated with Infrahub",
            "include_in_menu": False,
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "icon": "mdi:source-repository",
            "branch": BranchSupportType.AGNOSTIC.value,
            "uniqueness_constraints": [["name__value"], ["location__value"]],
            "documentation": "/topics/repository",
            "attributes": [
                {
                    "name": "name",
                    "regex": "^[^/]*$",
                    "kind": "Text",
                    "unique": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "order_weight": 1000,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "description",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "order_weight": 2000,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "location",
                    "kind": "Text",
                    "unique": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "order_weight": 3000,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "internal_status",
                    "kind": "Dropdown",
                    "choices": [
                        {
                            "name": RepositoryInternalStatus.STAGING.value,
                            "label": "Staging",
                            "description": "Repository was recently added to this branch.",
                            "color": "#fef08a",
                        },
                        {
                            "name": RepositoryInternalStatus.ACTIVE.value,
                            "label": "Active",
                            "description": "Repository is actively being synced for this branch",
                            "color": "#86efac",
                        },
                        {
                            "name": RepositoryInternalStatus.INACTIVE.value,
                            "label": "Inactive",
                            "description": "Repository is not active on this branch.",
                            "color": "#e5e7eb",
                        },
                    ],
                    "default_value": "inactive",
                    "optional": False,
                    "branch": BranchSupportType.LOCAL.value,
                    "order_weight": 7000,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "operational_status",
                    "kind": "Dropdown",
                    "choices": [
                        {
                            "name": RepositoryOperationalStatus.UNKNOWN.value,
                            "label": "Unknown",
                            "description": "Status of the repository is unknown and mostlikely because it hasn't been synced yet",
                            "color": "#9ca3af",
                        },
                        {
                            "name": RepositoryOperationalStatus.ONLINE.value,
                            "label": "Online",
                            "description": "Repository connection is working",
                            "color": "#86efac",
                        },
                        {
                            "name": RepositoryOperationalStatus.ERROR_CRED.value,
                            "label": "Credential Error",
                            "description": "Repository can't be synced due to some credential error(s)",
                            "color": "#f87171",
                        },
                        {
                            "name": RepositoryOperationalStatus.ERROR_CONNECTION.value,
                            "label": "Connectivity Error",
                            "description": "Repository can't be synced due to some connectivity error(s)",
                            "color": "#f87171",
                        },
                        {
                            "name": RepositoryOperationalStatus.ERROR.value,
                            "label": "Error",
                            "description": "Repository can't be synced due to an unknown error",
                            "color": "#ef4444",
                        },
                    ],
                    "optional": False,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "default_value": RepositoryOperationalStatus.UNKNOWN.value,
                    "order_weight": 5000,
                },
                {
                    "name": "sync_status",
                    "kind": "Dropdown",
                    "choices": [
                        {
                            "name": RepositorySyncStatus.UNKNOWN.value,
                            "label": "Unknown",
                            "description": "Status of the repository is unknown and mostlikely because it hasn't been synced yet",
                            "color": "#9ca3af",
                        },
                        {
                            "name": RepositorySyncStatus.ERROR_IMPORT.value,
                            "label": "Import Error",
                            "description": "Repository import error observed",
                            "color": "#f87171",
                        },
                        {
                            "name": RepositorySyncStatus.IN_SYNC.value,
                            "label": "In Sync",
                            "description": "The repository is syncing correctly",
                            "color": "#60a5fa",
                        },
                        {
                            "name": RepositorySyncStatus.SYNCING.value,
                            "label": "Syncing",
                            "description": "A sync job is currently running against the repository.",
                            "color": "#a855f7",
                        },
                    ],
                    "optional": False,
                    "branch": BranchSupportType.LOCAL.value,
                    "default_value": RepositorySyncStatus.UNKNOWN.value,
                    "order_weight": 6000,
                },
            ],
            "relationships": [
                {
                    "name": "credential",
                    "peer": InfrahubKind.CREDENTIAL,
                    "identifier": "gitrepository__credential",
                    "kind": "Attribute",
                    "optional": True,
                    "cardinality": "one",
                    "order_weight": 4000,
                },
                {
                    "name": "tags",
                    "peer": InfrahubKind.TAG,
                    "kind": "Attribute",
                    "optional": True,
                    "cardinality": "many",
                    "order_weight": 8000,
                },
                {
                    "name": "transformations",
                    "peer": InfrahubKind.TRANSFORM,
                    "identifier": "repository__transformation",
                    "optional": True,
                    "cardinality": "many",
                    "order_weight": 10000,
                },
                {
                    "name": "queries",
                    "peer": InfrahubKind.GRAPHQLQUERY,
                    "identifier": "graphql_query__repository",
                    "optional": True,
                    "cardinality": "many",
                    "order_weight": 9000,
                },
                {
                    "name": "checks",
                    "peer": InfrahubKind.CHECKDEFINITION,
                    "identifier": "check_definition__repository",
                    "optional": True,
                    "cardinality": "many",
                    "order_weight": 11000,
                },
                {
                    "name": "generators",
                    "peer": InfrahubKind.GENERATORDEFINITION,
                    "identifier": "generator_definition__repository",
                    "optional": True,
                    "cardinality": "many",
                    "order_weight": 12000,
                },
            ],
        },
        {
            "name": "IPNamespace",
            "namespace": "Builtin",
            "label": "IP Namespace",
            "description": "A generic container for IP prefixes and IP addresses",
            "include_in_menu": False,
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "icon": "mdi:format-list-group",
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "generate_profile": False,
            "attributes": [
                {
                    "name": "name",
                    "kind": "Text",
                    "unique": True,
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 1000,
                },
                {
                    "name": "description",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 2000,
                },
            ],
            "relationships": [
                {
                    "name": "ip_prefixes",
                    "label": "IP Prefixes",
                    "peer": InfrahubKind.IPPREFIX,
                    "identifier": "ip_namespace__ip_prefix",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "ip_addresses",
                    "label": "IP Addresses",
                    "peer": InfrahubKind.IPADDRESS,
                    "identifier": "ip_namespace__ip_address",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                    "allow_override": AllowOverrideType.NONE,
                },
            ],
        },
        {
            "name": "IPPrefix",
            "label": "IP Prefix",
            "namespace": "Builtin",
            "description": "IPv6 or IPv4 prefix also referred as network",
            "include_in_menu": False,
            "default_filter": "prefix__value",
            "order_by": ["prefix__version", "prefix__binary_address", "prefix__prefixlen"],
            "display_labels": ["prefix__value"],
            "icon": "mdi:ip-network",
            "branch": BranchSupportType.AWARE.value,
            "hierarchical": True,
            "attributes": [
                {
                    "name": "prefix",
                    "kind": "IPNetwork",
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 1000,
                },
                {
                    "name": "description",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 2000,
                },
                {
                    "name": "member_type",
                    "kind": "Dropdown",
                    "choices": [
                        {
                            "name": "prefix",
                            "label": "Prefix",
                            "description": "Prefix serves as container for other prefixes",
                        },
                        {
                            "name": "address",
                            "label": "Address",
                            "description": "Prefix serves as subnet for IP addresses",
                        },
                    ],
                    "branch": BranchSupportType.AWARE.value,
                    "default_value": "address",
                    "order_weight": 3000,
                },
                {
                    "name": "is_pool",
                    "kind": "Boolean",
                    "branch": BranchSupportType.AWARE.value,
                    "default_value": False,
                    "order_weight": 4000,
                    "description": "All IP addresses within this prefix are considered usable",
                },
                {
                    "name": "is_top_level",
                    "kind": "Boolean",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "utilization",
                    "kind": "Number",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "netmask",
                    "kind": "Text",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "hostmask",
                    "kind": "Text",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "network_address",
                    "kind": "Text",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "broadcast_address",
                    "kind": "Text",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
            ],
            "relationships": [
                {
                    "name": "ip_namespace",
                    "label": "IP Namespace",
                    "peer": InfrahubKind.IPNAMESPACE,
                    "identifier": "ip_namespace__ip_prefix",
                    "optional": True,
                    "cardinality": "one",
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "ip_addresses",
                    "label": "IP Addresses",
                    "peer": InfrahubKind.IPADDRESS,
                    "identifier": "ip_prefix__ip_address",
                    "optional": True,
                    "cardinality": "many",
                    "allow_override": AllowOverrideType.NONE,
                    "read_only": True,
                },
                {
                    "name": "resource_pool",
                    "peer": "CoreIPAddressPool",
                    "identifier": "ipaddresspool__resource",
                    "cardinality": "many",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "optional": True,
                    "read_only": True,
                },
            ],
        },
        {
            "name": "IPAddress",
            "label": "IP Address",
            "namespace": "Builtin",
            "description": "IPv6 or IPv4 address",
            "include_in_menu": False,
            "default_filter": "address__value",
            "order_by": ["address__version", "address__binary_address"],
            "display_labels": ["address__value"],
            "icon": "mdi:ip-outline",
            "branch": BranchSupportType.AWARE.value,
            "attributes": [
                {
                    "name": "address",
                    "kind": "IPHost",
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 1000,
                },
                {
                    "name": "description",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 2000,
                },
            ],
            "relationships": [
                {
                    "name": "ip_namespace",
                    "label": "IP Namespace",
                    "peer": InfrahubKind.IPNAMESPACE,
                    "identifier": "ip_namespace__ip_address",
                    "optional": True,
                    "cardinality": "one",
                    "allow_override": AllowOverrideType.NONE,
                },
                {
                    "name": "ip_prefix",
                    "label": "IP Prefix",
                    "peer": InfrahubKind.IPPREFIX,
                    "identifier": "ip_prefix__ip_address",
                    "optional": True,
                    "cardinality": "one",
                    "allow_override": AllowOverrideType.NONE,
                    "read_only": True,
                },
            ],
        },
        {
            "name": "ResourcePool",
            "namespace": "Core",
            "label": "Resource Pool",
            "description": "The resource manager contains pools of resources to allow for automatic assignments.",
            "include_in_menu": False,
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "human_friendly_id": ["name__value"],
            "icon": "mdi:view-grid-outline",
            "branch": BranchSupportType.AGNOSTIC.value,
            "uniqueness_constraints": [["name__value"]],
            "generate_profile": False,
            "attributes": [
                {
                    "name": "name",
                    "kind": "Text",
                    "order_weight": 1000,
                    "unique": True,
                },
                {
                    "name": "description",
                    "kind": "Text",
                    "optional": True,
                    "order_weight": 2000,
                },
            ],
        },
        {
            "name": "GenericAccount",
            "namespace": "Core",
            "description": "User Account for Infrahub",
            "include_in_menu": False,
            "label": "Account",
            "icon": "mdi:account",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["label__value"],
            "human_friendly_id": ["name__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "documentation": "/topics/auth",
            "uniqueness_constraints": [["name__value"]],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "password", "kind": "HashedPassword", "unique": False},
                {"name": "label", "kind": "Text", "optional": True},
                {"name": "description", "kind": "Text", "optional": True},
                {
                    "name": "account_type",
                    "kind": "Text",
                    "default_value": AccountType.USER.value,
                    "enum": AccountType.available_types(),
                },
                {
                    "name": "status",
                    "kind": "Dropdown",
                    "choices": [
                        {
                            "name": AccountStatus.ACTIVE.value,
                            "label": "Active",
                            "description": "Account is allowed to login",
                            "color": "#52be80",
                        },
                        {
                            "name": AccountStatus.INACTIVE.value,
                            "label": "Inactive",
                            "description": "Account is not allowed to login",
                            "color": "#e74c3c",
                        },
                    ],
                    "default_value": AccountStatus.ACTIVE.value,
                },
            ],
            "relationships": [
                {"name": "tokens", "peer": InfrahubKind.ACCOUNTTOKEN, "optional": True, "cardinality": "many"}
            ],
        },
        {
            "name": "BasePermission",
            "namespace": "Core",
            "description": "A permission grants right to an account",
            "label": "Base permission",
            "icon": "mdi:user-key",
            "include_in_menu": False,
            "generate_profile": False,
            "attributes": [
                {"name": "description", "kind": "Text", "optional": True},
                {
                    "name": "identifier",
                    "kind": "Text",
                    "read_only": True,
                    "optional": True,
                    "allow_override": AllowOverrideType.NONE,
                },
            ],
            "relationships": [
                {
                    "name": "roles",
                    "peer": InfrahubKind.ACCOUNTROLE,
                    "optional": True,
                    "identifier": "role__permissions",
                    "cardinality": "many",
                    "kind": "Attribute",
                }
            ],
        },
        {
            "name": "Credential",
            "namespace": "Core",
            "description": "A credential that could be referenced to access external services.",
            "include_in_menu": False,
            "label": "Credential",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["label__value"],
            "icon": "mdi:key-variant",
            "human_friendly_id": ["name__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "uniqueness_constraints": [["name__value"]],
            "documentation": "/topics/auth",
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True, "order_weight": 1000},
                {"name": "label", "kind": "Text", "optional": True, "order_weight": 2000},
                {"name": "description", "kind": "Text", "optional": True, "order_weight": 3000},
            ],
        },
        generic_menu_item,
    ],
    "nodes": [
        menu_item,
        {
            "name": "StandardGroup",
            "namespace": "Core",
            "description": "Group of nodes of any kind.",
            "include_in_menu": False,
            "icon": "mdi:account-group",
            "label": "Standard Group",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.AWARE.value,
            "inherit_from": [InfrahubKind.GENERICGROUP],
            "generate_profile": False,
        },
        {
            "name": "GeneratorGroup",
            "namespace": "Core",
            "description": "Group of nodes that are created by a generator.",
            "include_in_menu": False,
            "icon": "mdi:state-machine",
            "label": "Generator Group",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.LOCAL.value,
            "inherit_from": [InfrahubKind.GENERICGROUP],
            "generate_profile": False,
        },
        {
            "name": "GraphQLQueryGroup",
            "namespace": "Core",
            "description": "Group of nodes associated with a given GraphQLQuery.",
            "include_in_menu": False,
            "icon": "mdi:account-group",
            "label": "GraphQL Query Group",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.LOCAL.value,
            "inherit_from": [InfrahubKind.GENERICGROUP],
            "generate_profile": False,
            "attributes": [
                {"name": "parameters", "kind": "JSON", "optional": True},
            ],
            "relationships": [
                {
                    "name": "query",
                    "peer": InfrahubKind.GRAPHQLQUERY,
                    "optional": False,
                    "cardinality": "one",
                    "kind": "Attribute",
                },
            ],
        },
        {
            "name": "Tag",
            "namespace": "Builtin",
            "description": "Standard Tag object to attached to other objects to provide some context.",
            "include_in_menu": True,
            "icon": "mdi:tag-multiple",
            "label": "Tag",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "description", "kind": "Text", "optional": True},
            ],
        },
        {
            "name": "Account",
            "namespace": "Core",
            "description": "User Account for Infrahub",
            "include_in_menu": False,
            "label": "Account",
            "icon": "mdi:account",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["label__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.LINEAGEOWNER, InfrahubKind.LINEAGESOURCE, InfrahubKind.GENERICACCOUNT],
        },
        {
            "name": "AccountToken",
            "namespace": "Internal",
            "description": "Token for User Account",
            "include_in_menu": False,
            "label": "Account Token",
            "default_filter": "token__value",
            "display_labels": ["token__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "uniqueness_constraints": [["token__value"]],
            "documentation": "/topics/auth",
            "attributes": [
                {"name": "name", "kind": "Text", "optional": True},
                {"name": "token", "kind": "Text", "unique": True},
                {"name": "expiration", "kind": "DateTime", "optional": True},
            ],
            "relationships": [
                {
                    "name": "account",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": False,
                    "cardinality": "one",
                    "identifier": "account__token",
                },
            ],
        },
        {
            "name": "PasswordCredential",
            "namespace": "Core",
            "description": "Username/Password based credential",
            "include_in_menu": False,
            "label": "Username / Password",
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.CREDENTIAL],
            "attributes": [
                {
                    "name": "username",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "order_weight": 6000,
                },
                {
                    "name": "password",
                    "kind": "Password",
                    "optional": True,
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "order_weight": 7000,
                },
            ],
        },
        {
            "name": "RefreshToken",
            "namespace": "Internal",
            "description": "Refresh Token",
            "include_in_menu": False,
            "label": "Refresh Token",
            "display_labels": [],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "expiration", "kind": "DateTime", "optional": False},
            ],
            "relationships": [
                {
                    "name": "account",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": False,
                    "cardinality": "one",
                    "identifier": "account__refreshtoken",
                },
            ],
        },
        {
            "name": "ProposedChange",
            "namespace": "Core",
            "description": "Metadata related to a proposed change",
            "include_in_menu": False,
            "icon": "mdi:file-replace-outline",
            "label": "Proposed Change",
            "default_filter": "name__value",
            "display_labels": ["name__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.TASKTARGET],
            "documentation": "/topics/proposed-change",
            "attributes": [
                {"name": "name", "kind": "Text", "optional": False},
                {"name": "description", "kind": "TextArea", "optional": True},
                {"name": "source_branch", "kind": "Text", "optional": False},
                {"name": "destination_branch", "kind": "Text", "optional": False},
                {
                    "name": "state",
                    "kind": "Text",
                    "enum": ProposedChangeState.available_types(),
                    "default_value": ProposedChangeState.OPEN.value,
                    "optional": True,
                },
            ],
            "relationships": [
                {
                    "name": "approved_by",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": True,
                    "cardinality": "many",
                    "kind": "Attribute",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "identifier": "coreaccount__proposedchange_approved_by",
                },
                {
                    "name": "reviewers",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": True,
                    "kind": "Attribute",
                    "cardinality": "many",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "identifier": "coreaccount__proposedchange_reviewed_by",
                },
                {
                    "name": "created_by",
                    "peer": InfrahubKind.GENERICACCOUNT,
                    "optional": True,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "identifier": "coreaccount__proposedchange_created_by",
                },
                {
                    "name": "comments",
                    "peer": InfrahubKind.CHANGECOMMENT,
                    "kind": "Component",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                },
                {
                    "name": "threads",
                    "peer": InfrahubKind.THREAD,
                    "identifier": "proposedchange__thread",
                    "kind": "Component",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                },
                {
                    "name": "validations",
                    "peer": InfrahubKind.VALIDATOR,
                    "kind": "Component",
                    "identifier": "proposed_change__validator",
                    "optional": True,
                    "cardinality": "many",
                    "on_delete": RelationshipDeleteBehavior.CASCADE,
                },
            ],
        },
        {
            "name": "ChangeThread",
            "namespace": "Core",
            "description": "A thread on proposed change",
            "include_in_menu": False,
            "label": "Change Thread",
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.THREAD],
            "generate_profile": False,
            "attributes": [],
            "relationships": [],
        },
        {
            "name": "FileThread",
            "namespace": "Core",
            "description": "A thread related to a file on a proposed change",
            "include_in_menu": False,
            "label": "Thread - File",
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.THREAD],
            "generate_profile": False,
            "attributes": [
                {"name": "file", "kind": "Text", "optional": True},
                {"name": "commit", "kind": "Text", "optional": True},
                {"name": "line_number", "kind": "Number", "optional": True},
            ],
            "relationships": [
                {
                    "name": "repository",
                    "peer": InfrahubKind.REPOSITORY,
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
            ],
        },
        {
            "name": "ArtifactThread",
            "namespace": "Core",
            "description": "A thread related to an artifact on a proposed change",
            "include_in_menu": False,
            "label": "Thread - Artifact",
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.THREAD],
            "generate_profile": False,
            "attributes": [
                {"name": "artifact_id", "kind": "Text", "optional": True},
                {"name": "storage_id", "kind": "Text", "optional": True},
                {"name": "line_number", "kind": "Number", "optional": True},
            ],
            "relationships": [],
        },
        {
            "name": "ObjectThread",
            "namespace": "Core",
            "description": "A thread related to an object on a proposed change",
            "include_in_menu": False,
            "label": "Thread - Object",
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.THREAD],
            "generate_profile": False,
            "attributes": [
                {"name": "object_path", "kind": "Text", "optional": False},
            ],
            "relationships": [],
        },
        {
            "name": "ChangeComment",
            "namespace": "Core",
            "description": "A comment on proposed change",
            "include_in_menu": False,
            "label": "Change Comment",
            "default_filter": "text__value",
            "display_labels": ["text__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.COMMENT],
            "generate_profile": False,
            "relationships": [
                {
                    "name": "change",
                    "kind": "Parent",
                    "peer": InfrahubKind.PROPOSEDCHANGE,
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "ThreadComment",
            "namespace": "Core",
            "description": "A comment on thread within a Proposed Change",
            "include_in_menu": False,
            "label": "Thread Comment",
            "default_filter": "text__value",
            "display_labels": ["text__value"],
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [InfrahubKind.COMMENT],
            "generate_profile": False,
            "attributes": [],
            "relationships": [
                {
                    "name": "thread",
                    "peer": InfrahubKind.THREAD,
                    "kind": "Parent",
                    "identifier": "thread__threadcomment",
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "Repository",
            "namespace": "Core",
            "description": "A Git Repository integrated with Infrahub",
            "include_in_menu": False,
            "icon": "mdi:source-repository",
            "label": "Repository",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [
                InfrahubKind.LINEAGEOWNER,
                InfrahubKind.LINEAGESOURCE,
                InfrahubKind.GENERICREPOSITORY,
                InfrahubKind.TASKTARGET,
            ],
            "documentation": "/topics/repository",
            "attributes": [
                {"name": "default_branch", "kind": "Text", "default_value": "main", "order_weight": 6000},
                {
                    "name": "commit",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.LOCAL.value,
                    "order_weight": 7000,
                },
            ],
        },
        {
            "name": "ReadOnlyRepository",
            "namespace": "Core",
            "description": "A Git Repository integrated with Infrahub, Git-side will not be updated",
            "include_in_menu": False,
            "label": "Read-Only Repository",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "inherit_from": [
                InfrahubKind.LINEAGEOWNER,
                InfrahubKind.LINEAGESOURCE,
                InfrahubKind.GENERICREPOSITORY,
                InfrahubKind.TASKTARGET,
            ],
            "documentation": "/topics/repository",
            "attributes": [
                {
                    "name": "ref",
                    "kind": "Text",
                    "default_value": "main",
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 6000,
                },
                {
                    "name": "commit",
                    "kind": "Text",
                    "optional": True,
                    "branch": BranchSupportType.AWARE.value,
                    "order_weight": 7000,
                },
            ],
        },
        {
            "name": "TransformJinja2",
            "namespace": "Core",
            "description": "A file rendered from a Jinja2 template",
            "include_in_menu": False,
            "label": "Transform Jinja2",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "inherit_from": [InfrahubKind.TRANSFORM],
            "generate_profile": False,
            "branch": BranchSupportType.AWARE.value,
            "documentation": "/topics/transformation",
            "attributes": [
                {"name": "template_path", "kind": "Text"},
            ],
        },
        {
            "name": "DataCheck",
            "namespace": "Core",
            "description": "A check related to some Data",
            "include_in_menu": False,
            "label": "Data Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "conflicts", "kind": "JSON"},
                {"name": "keep_branch", "enum": BranchConflictKeep.available_types(), "kind": "Text", "optional": True},
                {"name": "enriched_conflict_id", "kind": "Text", "optional": True},
            ],
        },
        {
            "name": "StandardCheck",
            "namespace": "Core",
            "description": "A standard check",
            "include_in_menu": False,
            "label": "Standard Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
        },
        {
            "name": "SchemaCheck",
            "namespace": "Core",
            "description": "A check related to the schema",
            "include_in_menu": False,
            "label": "Schema Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "conflicts", "kind": "JSON"},
                {"name": "enriched_conflict_id", "kind": "Text", "optional": True},
            ],
        },
        {
            "name": "FileCheck",
            "namespace": "Core",
            "description": "A check related to a file in a Git Repository",
            "include_in_menu": False,
            "label": "File Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "files", "kind": "List", "optional": True},
                {"name": "commit", "kind": "Text", "optional": True},
            ],
        },
        {
            "name": "ArtifactCheck",
            "namespace": "Core",
            "description": "A check related to an artifact",
            "include_in_menu": False,
            "label": "Artifact Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {"name": "changed", "kind": "Boolean", "optional": True},
                {"name": "checksum", "kind": "Text", "optional": True},
                {"name": "artifact_id", "kind": "Text", "optional": True},
                {"name": "storage_id", "kind": "Text", "optional": True},
                {"name": "line_number", "kind": "Number", "optional": True},
            ],
        },
        {
            "name": "GeneratorCheck",
            "namespace": "Core",
            "description": "A check related to a Generator instance",
            "include_in_menu": False,
            "label": "Generator Check",
            "display_labels": ["label__value"],
            "inherit_from": ["CoreCheck"],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {
                    "name": "instance",
                    "kind": "Text",
                    "optional": False,
                },
            ],
        },
        {
            "name": "DataValidator",
            "namespace": "Core",
            "description": "A check to validate the data integrity between two branches",
            "include_in_menu": False,
            "label": "Data Validator",
            "display_labels": ["label__value"],
            "inherit_from": [InfrahubKind.VALIDATOR],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
        },
        {
            "name": "RepositoryValidator",
            "namespace": "Core",
            "description": "A Validator related to a specific repository",
            "include_in_menu": False,
            "label": "Repository Validator",
            "display_labels": ["label__value"],
            "inherit_from": [InfrahubKind.VALIDATOR],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "relationships": [
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
            ],
        },
        {
            "name": "UserValidator",
            "namespace": "Core",
            "description": "A Validator related to a user defined checks in a repository",
            "include_in_menu": False,
            "label": "User Validator",
            "display_labels": ["label__value"],
            "inherit_from": [InfrahubKind.VALIDATOR],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "relationships": [
                {
                    "name": "check_definition",
                    "peer": InfrahubKind.CHECKDEFINITION,
                    "kind": "Attribute",
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
            ],
        },
        {
            "name": "SchemaValidator",
            "namespace": "Core",
            "description": "A validator related to the schema",
            "include_in_menu": False,
            "label": "Schema Validator",
            "display_labels": ["label__value"],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.VALIDATOR],
            "branch": BranchSupportType.AGNOSTIC.value,
        },
        {
            "name": "ArtifactValidator",
            "namespace": "Core",
            "description": "A validator related to the artifacts",
            "include_in_menu": False,
            "label": "Artifact Validator",
            "display_labels": ["label__value"],
            "inherit_from": [InfrahubKind.VALIDATOR],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "relationships": [
                {
                    "name": "definition",
                    "peer": InfrahubKind.ARTIFACTDEFINITION,
                    "kind": "Attribute",
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
            ],
        },
        {
            "name": "GeneratorValidator",
            "namespace": "Core",
            "description": "A validator related to generators",
            "include_in_menu": False,
            "label": "Generator Validator",
            "display_labels": ["label__value"],
            "inherit_from": [InfrahubKind.VALIDATOR],
            "generate_profile": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "relationships": [
                {
                    "name": "definition",
                    "peer": InfrahubKind.GENERATORDEFINITION,
                    "kind": "Attribute",
                    "optional": False,
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                },
            ],
        },
        {
            "name": "CheckDefinition",
            "namespace": "Core",
            "include_in_menu": False,
            "icon": "mdi:check-all",
            "label": "Check Definition",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.TASKTARGET],
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "description", "kind": "Text", "optional": True},
                {"name": "file_path", "kind": "Text"},
                {"name": "class_name", "kind": "Text"},
                {"name": "timeout", "kind": "Number", "default_value": 10},
                {"name": "parameters", "kind": "JSON", "optional": True},
            ],
            "relationships": [
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "cardinality": "one",
                    "identifier": "check_definition__repository",
                    "optional": False,
                },
                {
                    "name": "query",
                    "peer": InfrahubKind.GRAPHQLQUERY,
                    "kind": "Attribute",
                    "identifier": "check_definition__graphql_query",
                    "cardinality": "one",
                    "optional": True,
                },
                {
                    "name": "targets",
                    "peer": InfrahubKind.GENERICGROUP,
                    "kind": "Attribute",
                    "identifier": "check_definition___group",
                    "cardinality": "one",
                    "optional": True,
                },
                {
                    "name": "tags",
                    "peer": InfrahubKind.TAG,
                    "kind": "Attribute",
                    "optional": True,
                    "cardinality": "many",
                },
            ],
        },
        {
            "name": "TransformPython",
            "namespace": "Core",
            "description": "A transform function written in Python",
            "include_in_menu": False,
            "label": "Transform Python",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "inherit_from": [InfrahubKind.TRANSFORM],
            "generate_profile": False,
            "branch": BranchSupportType.AWARE.value,
            "documentation": "/topics/transformation",
            "attributes": [
                {"name": "file_path", "kind": "Text"},
                {"name": "class_name", "kind": "Text"},
            ],
        },
        {
            "name": "GraphQLQuery",
            "namespace": "Core",
            "description": "A pre-defined GraphQL Query",
            "include_in_menu": False,
            "icon": "mdi:graphql",
            "label": "GraphQL Query",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "generate_profile": False,
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "documentation": "/topics/graphql",
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "description", "kind": "Text", "optional": True},
                {"name": "query", "kind": "TextArea"},
                {
                    "name": "variables",
                    "kind": "JSON",
                    "description": "variables in use in the query",
                    "optional": True,
                    "read_only": True,
                },
                {
                    "name": "operations",
                    "kind": "List",
                    "description": "Operations in use in the query, valid operations: 'query', 'mutation' or 'subscription'",
                    "read_only": True,
                    "optional": True,
                },
                {
                    "name": "models",
                    "kind": "List",
                    "description": "List of models associated with this query",
                    "read_only": True,
                    "optional": True,
                },
                {
                    "name": "depth",
                    "kind": "Number",
                    "description": "number of nested levels in the query",
                    "read_only": True,
                    "optional": True,
                },
                {
                    "name": "height",
                    "kind": "Number",
                    "description": "total number of fields requested in the query",
                    "read_only": True,
                    "optional": True,
                },
            ],
            "relationships": [
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "identifier": "graphql_query__repository",
                    "cardinality": "one",
                    "optional": True,
                },
                {
                    "name": "tags",
                    "peer": InfrahubKind.TAG,
                    "kind": "Attribute",
                    "optional": True,
                    "cardinality": "many",
                },
            ],
        },
        {
            "name": "Artifact",
            "namespace": "Core",
            "label": "Artifact",
            "include_in_menu": False,
            "icon": "mdi:file-document-outline",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.LOCAL.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.TASKTARGET],
            "documentation": "/topics/artifact",
            "attributes": [
                {"name": "name", "kind": "Text"},
                {
                    "name": "status",
                    "kind": "Text",
                    "enum": ArtifactStatus.available_types(),
                },
                {
                    "name": "content_type",
                    "kind": "Text",
                    "enum": ContentType.available_types(),
                },
                {
                    "name": "checksum",
                    "kind": "Text",
                    "optional": True,
                },
                {
                    "name": "storage_id",
                    "kind": "Text",
                    "optional": True,
                    "description": "ID of the file in the object store",
                },
                {"name": "parameters", "kind": "JSON", "optional": True},
            ],
            "relationships": [
                {
                    "name": "object",
                    "peer": InfrahubKind.ARTIFACTTARGET,
                    "kind": "Attribute",
                    "identifier": "artifact__node",
                    "cardinality": "one",
                    "optional": False,
                },
                {
                    "name": "definition",
                    "peer": InfrahubKind.ARTIFACTDEFINITION,
                    "kind": "Attribute",
                    "identifier": "artifact__artifact_definition",
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "ArtifactDefinition",
            "namespace": "Core",
            "include_in_menu": False,
            "icon": "mdi:file-document-multiple-outline",
            "label": "Artifact Definition",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.AWARE.value,
            "generate_profile": False,
            "uniqueness_constraints": [["name__value"]],
            "inherit_from": [InfrahubKind.TASKTARGET],
            "documentation": "/topics/artifact",
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "artifact_name", "kind": "Text"},
                {"name": "description", "kind": "Text", "optional": True},
                {"name": "parameters", "kind": "JSON"},
                {
                    "name": "content_type",
                    "kind": "Text",
                    "enum": ContentType.available_types(),
                },
            ],
            "relationships": [
                {
                    "name": "targets",
                    "peer": InfrahubKind.GENERICGROUP,
                    "kind": "Attribute",
                    "identifier": "artifact_definition___group",
                    "cardinality": "one",
                    "optional": False,
                },
                {
                    "name": "transformation",
                    "peer": InfrahubKind.TRANSFORM,
                    "kind": "Attribute",
                    "identifier": "artifact_definition___transformation",
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "GeneratorDefinition",
            "namespace": "Core",
            "include_in_menu": False,
            "icon": "mdi:state-machine",
            "label": "Generator Definition",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.AWARE.value,
            "uniqueness_constraints": [["name__value"]],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.TASKTARGET],
            "documentation": "/topics/generator",
            "attributes": [
                {"name": "name", "kind": "Text", "unique": True},
                {"name": "description", "kind": "Text", "optional": True},
                {"name": "parameters", "kind": "JSON"},
                {"name": "file_path", "kind": "Text"},
                {"name": "class_name", "kind": "Text"},
                {"name": "convert_query_response", "kind": "Boolean", "optional": True, "default_value": False},
            ],
            "relationships": [
                {
                    "name": "query",
                    "peer": InfrahubKind.GRAPHQLQUERY,
                    "identifier": "generator_definition__graphql_query",
                    "kind": "Attribute",
                    "cardinality": "one",
                    "optional": False,
                },
                {
                    "name": "repository",
                    "peer": InfrahubKind.GENERICREPOSITORY,
                    "kind": "Attribute",
                    "cardinality": "one",
                    "identifier": "generator_definition__repository",
                    "optional": False,
                },
                {
                    "name": "targets",
                    "peer": InfrahubKind.GENERICGROUP,
                    "kind": "Attribute",
                    "identifier": "generator_definition___group",
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "GeneratorInstance",
            "namespace": "Core",
            "label": "Generator Instance",
            "include_in_menu": False,
            "icon": "mdi:file-document-outline",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "branch": BranchSupportType.LOCAL.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.TASKTARGET],
            "documentation": "/topics/generator",
            "attributes": [
                {"name": "name", "kind": "Text"},
                {
                    "name": "status",
                    "kind": "Text",
                    "enum": GeneratorInstanceStatus.available_types(),
                },
            ],
            "relationships": [
                {
                    "name": "object",
                    "peer": InfrahubKind.NODE,
                    "kind": "Attribute",
                    "identifier": "generator__node",
                    "cardinality": "one",
                    "optional": False,
                },
                {
                    "name": "definition",
                    "peer": InfrahubKind.GENERATORDEFINITION,
                    "kind": "Attribute",
                    "identifier": "generator__generator_definition",
                    "cardinality": "one",
                    "optional": False,
                },
            ],
        },
        {
            "name": "StandardWebhook",
            "namespace": "Core",
            "description": "A webhook that connects to an external integration",
            "label": "Standard Webhook",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "include_in_menu": False,
            "icon": "mdi:webhook",
            "branch": BranchSupportType.AGNOSTIC.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.WEBHOOK, InfrahubKind.TASKTARGET],
            "attributes": [
                {"name": "shared_key", "kind": "Password", "unique": False, "order_weight": 4000},
            ],
        },
        {
            "name": "CustomWebhook",
            "namespace": "Core",
            "description": "A webhook that connects to an external integration",
            "label": "Custom Webhook",
            "default_filter": "name__value",
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "include_in_menu": False,
            "icon": "mdi:cog-outline",
            "branch": BranchSupportType.AGNOSTIC.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.WEBHOOK, InfrahubKind.TASKTARGET],
            "attributes": [],
            "relationships": [
                {
                    "name": "transformation",
                    "peer": InfrahubKind.TRANSFORMPYTHON,
                    "kind": "Attribute",
                    "identifier": "webhook___transformation",
                    "cardinality": "one",
                    "optional": True,
                    "order_weight": 7000,
                },
            ],
        },
        {
            "name": "Namespace",
            "namespace": "Ipam",
            "description": "A namespace that segments IPAM",
            "label": "IPAM Namespace",
            "default_filter": "name__value",
            "human_friendly_id": ["name__value"],
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "include_in_menu": False,
            "icon": "mdi:format-list-group",
            "branch": BranchSupportType.AWARE.value,
            "inherit_from": [InfrahubKind.IPNAMESPACE],
            "attributes": [
                {"name": "default", "kind": "Boolean", "optional": True, "read_only": True, "order_weight": 9000}
            ],
        },
        {
            "name": "IPPrefixPool",
            "namespace": "Core",
            "description": "A pool of IP prefix resources",
            "label": "IP Prefix Pool",
            "include_in_menu": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.RESOURCEPOOL, InfrahubKind.LINEAGESOURCE],
            "attributes": [
                {
                    "name": "default_prefix_length",
                    "kind": "Number",
                    "description": "The default prefix length as an integer for prefixes allocated from this pool.",
                    "optional": True,
                    "order_weight": 5000,
                },
                {
                    "name": "default_member_type",
                    "kind": "Text",
                    "enum": ["prefix", "address"],
                    "default_value": "prefix",
                    "optional": True,
                    "order_weight": 3000,
                },
                {
                    "name": "default_prefix_type",
                    "kind": "Text",
                    "optional": True,
                    "order_weight": 4000,
                },
            ],
            "relationships": [
                {
                    "name": "resources",
                    "peer": "BuiltinIPPrefix",
                    "kind": "Attribute",
                    "identifier": "prefixpool__resource",
                    "cardinality": "many",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "optional": False,
                    "order_weight": 6000,
                },
                {
                    "name": "ip_namespace",
                    "peer": "BuiltinIPNamespace",
                    "kind": "Attribute",
                    "identifier": "prefixpool__ipnamespace",
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "optional": False,
                    "order_weight": 7000,
                },
            ],
        },
        {
            "name": "IPAddressPool",
            "namespace": "Core",
            "description": "A pool of IP address resources",
            "label": "IP Address Pool",
            "include_in_menu": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.RESOURCEPOOL, InfrahubKind.LINEAGESOURCE],
            "attributes": [
                {
                    "name": "default_address_type",
                    "kind": "Text",
                    "optional": False,
                    "description": "The object type to create when reserving a resource in the pool",
                    "order_weight": 3000,
                },
                {
                    "name": "default_prefix_length",
                    "kind": "Number",
                    "description": "The default prefix length as an integer for addresses allocated from this pool.",
                    "optional": True,
                    "order_weight": 4000,
                },
            ],
            "relationships": [
                {
                    "name": "resources",
                    "peer": "BuiltinIPPrefix",
                    "kind": "Attribute",
                    "identifier": "ipaddresspool__resource",
                    "cardinality": "many",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "optional": False,
                    "order_weight": 5000,
                },
                {
                    "name": "ip_namespace",
                    "peer": "BuiltinIPNamespace",
                    "kind": "Attribute",
                    "identifier": "ipaddresspool__ipnamespace",
                    "cardinality": "one",
                    "branch": BranchSupportType.AGNOSTIC.value,
                    "optional": False,
                    "order_weight": 6000,
                },
            ],
        },
        {
            "name": "NumberPool",
            "namespace": "Core",
            "description": "A pool of number resources",
            "label": "Number Pool",
            "include_in_menu": False,
            "branch": BranchSupportType.AGNOSTIC.value,
            "generate_profile": False,
            "inherit_from": [InfrahubKind.RESOURCEPOOL, InfrahubKind.LINEAGESOURCE],
            "attributes": [
                {
                    "name": "node",
                    "kind": "Text",
                    "optional": False,
                    "description": "The model of the object that requires integers to be allocated",
                    "order_weight": 3000,
                },
                {
                    "name": "node_attribute",
                    "kind": "Text",
                    "description": "The attribute of the selected model",
                    "optional": False,
                    "order_weight": 4000,
                },
                {
                    "name": "start_range",
                    "kind": "Number",
                    "optional": False,
                    "description": "The start range for the pool",
                    "order_weight": 5000,
                },
                {
                    "name": "end_range",
                    "kind": "Number",
                    "optional": False,
                    "description": "The end range for the pool",
                    "order_weight": 6000,
                },
            ],
        },
        {
            "name": "GlobalPermission",
            "namespace": "Core",
            "description": "A permission that grants global rights to perform actions in Infrahub",
            "label": "Global permission",
            "include_in_menu": False,
            "order_by": ["action__value", "decision__value"],
            "display_labels": ["action__value", "decision__value"],
            "human_friendly_id": ["action__value", "decision__value"],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.BASEPERMISSION],
            "branch": BranchSupportType.AGNOSTIC.value,
            "attributes": [
                {
                    "name": "action",
                    "kind": "Dropdown",
                    "choices": [{"name": permission.value} for permission in GlobalPermissions],
                    "order_weight": 2000,
                },
                {
                    "name": "decision",
                    "kind": "Number",
                    "enum": PermissionDecision.available_types(),
                    "default_value": PermissionDecision.ALLOW_ALL.value,
                    "order_weight": 3000,
                    "description": "Decide to deny or allow the action at a global level",
                },
            ],
        },
        {
            "name": "ObjectPermission",
            "namespace": "Core",
            "description": "A permission that grants rights to perform actions on objects",
            "label": "Object permission",
            "include_in_menu": False,
            "order_by": ["namespace__value", "name__value", "action__value", "decision__value"],
            "display_labels": ["namespace__value", "name__value", "action__value", "decision__value"],
            "human_friendly_id": ["namespace__value", "name__value", "action__value", "decision__value"],
            "uniqueness_constraints": [["namespace__value", "name__value", "action__value", "decision__value"]],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.BASEPERMISSION],
            "attributes": [
                {"name": "namespace", "kind": "Text", "order_weight": 2000},
                {"name": "name", "kind": "Text", "order_weight": 3000},
                {
                    "name": "action",
                    "kind": "Text",
                    "enum": PermissionAction.available_types(),
                    "default_value": PermissionAction.ANY.value,
                    "order_weight": 4000,
                },
                {
                    "name": "decision",
                    "kind": "Number",
                    "enum": PermissionDecision.available_types(),
                    "default_value": PermissionDecision.ALLOW_ALL.value,
                    "order_weight": 5000,
                    "description": (
                        "Decide to deny or allow the action. If allowed, it can be configured for the default branch, any other branches or all "
                        "branches"
                    ),
                },
            ],
        },
        {
            "name": "AccountRole",
            "namespace": "Core",
            "description": "A role defines a set of permissions to grant to a group of accounts",
            "label": "Account role",
            "icon": "mdi:user-badge",
            "include_in_menu": False,
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "human_friendly_id": ["name__value"],
            "generate_profile": False,
            "attributes": [{"name": "name", "kind": "Text", "unique": True}],
            "relationships": [
                {
                    "name": "groups",
                    "peer": InfrahubKind.ACCOUNTGROUP,
                    "optional": True,
                    "identifier": "role__accountgroups",
                    "cardinality": "many",
                    "kind": "Attribute",
                },
                {
                    "name": "permissions",
                    "peer": InfrahubKind.BASEPERMISSION,
                    "optional": True,
                    "identifier": "role__permissions",
                    "cardinality": "many",
                    "kind": "Attribute",
                },
            ],
        },
        {
            "name": "AccountGroup",
            "namespace": "Core",
            "description": "A group of users to manage common permissions",
            "label": "Account group",
            "icon": "mdi:account-group",
            "include_in_menu": False,
            "order_by": ["name__value"],
            "display_labels": ["name__value"],
            "human_friendly_id": ["name__value"],
            "generate_profile": False,
            "inherit_from": [InfrahubKind.LINEAGEOWNER, InfrahubKind.LINEAGESOURCE, InfrahubKind.GENERICGROUP],
            "branch": BranchSupportType.AGNOSTIC.value,
            "relationships": [
                {
                    "name": "roles",
                    "peer": InfrahubKind.ACCOUNTROLE,
                    "optional": True,
                    "identifier": "role__accountgroups",
                    "cardinality": "many",
                    "kind": "Attribute",
                }
            ],
        },
    ],
}
