"""Django management command to disassemble items from ROM and generate Python implementations."""

from django.core.management.base import BaseCommand
from pathlib import Path
from smrpgpatchbuilder.datatypes.items.encoding import decode_item_description


class Command(BaseCommand):
    help = "Disassemble items from ROM and generate Python item definitions"

    def add_arguments(self, parser):
        parser.add_argument(
            "-r",
            "--rom",
            dest="rom",
            required=True,
            help="Path to a Mario RPG ROM file",
        )
        parser.add_argument(
            "-o",
            "--output",
            dest="output",
            default="src/disassembler_output/items/items.py",
            help="Output file path",
        )

    def handle(self, *args, **options):
        rom_path = options["rom"]
        output_path = options["output"]

        # read rom
        with open(rom_path, "rb") as f:
            rom = bytearray(f.read())

        # constants from items/constants.py
        ITEMS_BASE_ADDRESS = 0x3A014D
        ITEMS_BASE_PRICE_ADDRESS = 0x3A40F2
        ITEMS_BASE_TIMING_ADDRESS = 0x3A438A
        ITEMS_BASE_NAME_ADDRESS = 0x3A46EF
        ITEMS_BASE_DESC_POINTER_ADDRESS = 0x3A2F20
        ITEMS_DESC_DATA_POINTER_OFFSET = 0x3A0000
        NUM_ITEMS = 256

        # generate output
        output_lines = []
        output_lines.append("# AUTOGENERATED DO NOT EDIT!!")
        output_lines.append("# Run the following command to rebuild:")
        output_lines.append(f"# python manage.py itemdisassembler --rom <ROM_PATH>")
        output_lines.append("")
        output_lines.append("from typing import List")
        output_lines.append("from smrpgpatchbuilder.datatypes.items.classes import (")
        output_lines.append("    RegularItem,")
        output_lines.append("    Weapon,")
        output_lines.append("    Armor,")
        output_lines.append("    Accessory,")
        output_lines.append("    ItemCollection,")
        output_lines.append(")")
        output_lines.append("from smrpgpatchbuilder.datatypes.items.enums import (")
        output_lines.append("    EffectType,")
        output_lines.append("    InflictFunction,")
        output_lines.append("    OverworldMenuBehaviour,")
        output_lines.append("    ItemPrefix,")
        output_lines.append(")")
        output_lines.append(
            "from smrpgpatchbuilder.datatypes.numbers.classes import UInt8"
        )
        output_lines.append(
            "from smrpgpatchbuilder.datatypes.overworld_scripts.arguments.types.party_character import PartyCharacter"
        )
        output_lines.append(
            "from smrpgpatchbuilder.datatypes.overworld_scripts.arguments.area_objects import ("
        )
        output_lines.append("    MARIO,")
        output_lines.append("    TOADSTOOL,")
        output_lines.append("    BOWSER,")
        output_lines.append("    GENO,")
        output_lines.append("    MALLOW,")
        output_lines.append(")")
        output_lines.append(
            "from smrpgpatchbuilder.datatypes.spells.enums import Element, Status, TempStatBuff"
        )
        output_lines.append("")
        output_lines.append("")

        # read item descriptions from rom
        descriptions = self._read_item_descriptions(
            rom,
            ITEMS_BASE_DESC_POINTER_ADDRESS,
            ITEMS_DESC_DATA_POINTER_OFFSET,
            NUM_ITEMS
        )

        # process each item
        used_class_names = {}
        class_names = []  # store class names for itemcollection
        for item_id in range(NUM_ITEMS):
            item_data = self._parse_item(
                rom,
                item_id,
                ITEMS_BASE_ADDRESS,
                ITEMS_BASE_PRICE_ADDRESS,
                ITEMS_BASE_TIMING_ADDRESS,
                ITEMS_BASE_NAME_ADDRESS
            )

            # add description to item data
            item_data["description"] = descriptions[item_id]

            # generate unique class name
            class_name = self._generate_class_name(item_data["name"], used_class_names)
            class_names.append(class_name)

            class_def = self._generate_class_definition(item_id, item_data, class_name)
            output_lines.extend(class_def)
            output_lines.append("")
            output_lines.append("")

        # generate itemcollection
        output_lines.append("")
        output_lines.append("# Item collection containing all items")
        output_lines.append("ALL_ITEMS = ItemCollection([")
        for class_name in class_names:
            output_lines.append(f"    {class_name}(),")
        output_lines.append("])")
        output_lines.append("")

        # write output file
        output_file = Path(output_path)
        output_file.parent.mkdir(parents=True, exist_ok=True)
        output_file.write_text("\n".join(output_lines))

        # generate __init__.py with all class imports
        init_lines = []
        init_lines.append("# AUTOGENERATED DO NOT EDIT!!")
        init_lines.append("# This file imports all item classes from items.py")
        init_lines.append("")
        init_lines.append("from .items import (")
        for class_name in class_names:
            init_lines.append(f"    {class_name},")
        init_lines.append("    ALL_ITEMS,")
        init_lines.append(")")
        init_lines.append("")

        init_file = output_file.parent / "__init__.py"
        init_file.write_text("\n".join(init_lines))

        self.stdout.write(
            self.style.SUCCESS(f"Successfully generated {NUM_ITEMS} item definitions to {output_path}")
        )
        self.stdout.write(
            self.style.SUCCESS(f"Successfully generated __init__.py with {len(class_names)} class imports")
        )

    def _parse_item(self, rom, item_id, base_addr, price_addr, timing_addr, name_addr):
        """Parse item data from ROM."""
        # read item name and prefix
        name_offset = name_addr + (item_id * 15)
        name, prefix_byte = self._parse_item_name(rom, name_offset)

        offset = base_addr + (item_id * 18)

        # read 18 bytes of item data
        data = rom[offset : offset + 18]

        # byte 0: type and flags
        byte0 = data[0]
        type_value = byte0 & 0x03
        usable_battle = bool(byte0 & 0x08)
        usable_overworld = bool(byte0 & 0x10)
        reusable = bool(byte0 & 0x20)
        prevent_ko = bool(byte0 & 0x80)

        # byte 1: effect type and overworld menu flags
        byte1 = data[1]
        effect_type = byte1 & 0x07
        overworld_menu_fp = bool(byte1 & 0x20)
        overworld_menu_fill_hp = bool(byte1 & 0x40)
        overworld_menu_fill_fp = bool(byte1 & 0x80)

        # byte 2: equip chars
        byte2 = data[2]
        equip_chars = []
        if byte2 & 0x01:
            equip_chars.append("MARIO")
        if byte2 & 0x02:
            equip_chars.append("TOADSTOOL")
        if byte2 & 0x04:
            equip_chars.append("BOWSER")
        if byte2 & 0x08:
            equip_chars.append("GENO")
        if byte2 & 0x10:
            equip_chars.append("MALLOW")

        # byte 3: targeting
        byte3 = data[3]
        can_target_others = bool(byte3 & 0x02)
        target_enemies = bool(byte3 & 0x04)
        target_all = bool(byte3 & 0x10)
        koed_target_only = bool(byte3 & 0x20)
        one_side_only = bool(byte3 & 0x40)
        can_target_self = not bool(byte3 & 0x80)

        # byte 4: inflict element
        inflict_element = data[4]

        # byte 5: elemental immunities
        elemental_immunities = self._parse_bitfield_to_elements(data[5])

        # byte 6: elemental resistances
        elemental_resistances = self._parse_bitfield_to_elements(data[6])

        # byte 7: status immunities
        status_immunities = self._parse_bitfield_to_statuses(data[7])

        # byte 8: temp buffs
        temp_buffs = self._parse_bitfield_to_buffs(data[8])

        # bytes 9-15: stats
        speed = self._signed_byte(data[9])
        attack = self._signed_byte(data[10])
        defense = self._signed_byte(data[11])
        magic_attack = self._signed_byte(data[12])
        magic_defense = self._signed_byte(data[13])
        variance = data[14]
        inflict = data[15]

        # byte 16: inflict type and hide damage
        byte16 = data[16]
        inflict_type = byte16  # remove hide_damage bit

        # byte 17: hide damage
        byte17 = data[17]
        hide_damage = bool(byte17 & 0x04)

        # read price (2 bytes)
        price_offset = price_addr + (item_id * 2)
        price = rom[price_offset] + (rom[price_offset + 1] << 8)

        # read weapon timing if this is a weapon (item_id 0-40)
        timing_data = None
        if item_id < 39 and type_value == 0:  # weapon
            timing_offset = timing_addr + (item_id * 4)
            timing_data = {
                "half_time_begins": rom[timing_offset],
                "perfect_begins": rom[timing_offset + 1],
                "perfect_ends": rom[timing_offset + 2],
                "half_time_ends": rom[timing_offset + 3],
            }

        return {
            "name": name,
            "prefix_byte": prefix_byte,
            "type_value": type_value,
            "usable_battle": usable_battle,
            "usable_overworld": usable_overworld,
            "reusable": reusable,
            "prevent_ko": prevent_ko,
            "effect_type": effect_type,
            "overworld_menu_fp": overworld_menu_fp,
            "overworld_menu_fill_hp": overworld_menu_fill_hp,
            "overworld_menu_fill_fp": overworld_menu_fill_fp,
            "equip_chars": equip_chars,
            "can_target_others": can_target_others,
            "target_enemies": target_enemies,
            "target_all": target_all,
            "koed_target_only": koed_target_only,
            "one_side_only": one_side_only,
            "can_target_self": can_target_self,
            "inflict_element": inflict_element,
            "elemental_immunities": elemental_immunities,
            "elemental_resistances": elemental_resistances,
            "status_immunities": status_immunities,
            "temp_buffs": temp_buffs,
            "speed": speed,
            "attack": attack,
            "defense": defense,
            "magic_attack": magic_attack,
            "magic_defense": magic_defense,
            "variance": variance,
            "inflict": inflict,
            "hide_damage": hide_damage,
            "inflict_type": inflict_type,
            "price": price,
            "timing_data": timing_data,
        }

    def _signed_byte(self, byte):
        """Convert unsigned byte to signed."""
        if byte > 127:
            return byte - 256
        return byte

    def _read_item_descriptions(self, rom, pointer_addr, pointer_offset, num_items):
        """read all item descriptions from rom using the pointer table.

        args:
            rom: rom bytearray
            pointer_addr: address of the pointer table
            pointer_offset: offset to add to pointers to get actual address
            num_items: number of items to read

        returns:
            list of description strings
        """
        descriptions = []

        for item_id in range(num_items):
            # read pointer (little-endian 2 bytes)
            ptr_offset = pointer_addr + (item_id * 2)
            pointer_value = rom[ptr_offset] | (rom[ptr_offset + 1] << 8)

            # calculate actual address
            desc_addr = pointer_offset + pointer_value

            # read description (null-terminated)
            desc_bytes = []
            idx = desc_addr
            while idx < len(rom):
                byte = rom[idx]
                if byte == 0x00:  # null terminator
                    break
                desc_bytes.append(byte)
                idx += 1

            # decode description using special character mapping
            desc = decode_item_description(bytes(desc_bytes))

            descriptions.append(desc)

        return descriptions

    def _parse_item_name(self, rom, offset):
        """parse item name from rom (15 bytes, detect prefix, strip trailing 0x20).

        returns tuple of (name, prefix_byte)
        """
        name_bytes = rom[offset : offset + 15]

        # check if first byte is a prefix (non-ascii or special character)
        first_byte = name_bytes[0]
        prefix_byte = None
        start_idx = 0

        # known prefix values that should not be treated as ascii
        known_prefixes = {0x7F, 0x22, 0x28, 0x23, 0x21, 0x29, 0x25, 0x27, 0x26, 0x3C, 0x3D, 0x2D, 0x2E, 0x3B, 0x3F}

        if first_byte in known_prefixes:
            prefix_byte = first_byte
            start_idx = 1

        # extract the name portion and strip trailing spaces (0x20)
        name_chars = list(name_bytes[start_idx:])
        while name_chars and name_chars[-1] == 0x20:
            name_chars.pop()

        # convert to string (assuming ascii/latin-1 encoding)
        # special character mapping: 0x3A -> ’ (closing single quote)
        try:
            name = ""
            for byte_val in name_chars:
                if byte_val == 0x7E:
                    name += "’"
                else:
                    name += bytes([byte_val]).decode('latin-1')
        except:
            name = f"Item{offset}"

        return (name if name else "UnnamedItem", prefix_byte)

    def _generate_class_name(self, item_name, used_names):
        """Generate a valid Python class name from item name, handling duplicates."""
        # clean up the name for use as a class name
        base_name = item_name.replace(" ", "").replace("-", "").replace("'", "")
        # remove any non-alphanumeric characters
        base_name = ''.join(c for c in base_name if c.isalnum())

        # ensure it starts with a letter
        if not base_name or not base_name[0].isalpha():
            base_name = "Unnamed"

        # track occurrence for duplicates
        if base_name not in used_names:
            used_names[base_name] = 0

        used_names[base_name] += 1
        occurrence = used_names[base_name]

        # generate class name with "Item" suffix
        if occurrence > 1:
            class_name = f"{base_name}Item{occurrence}"
        else:
            class_name = f"{base_name}Item"

        return class_name

    def _parse_bitfield_to_elements(self, bitfield):
        """Parse bitfield to Element enum list."""
        elements = []
        element_map = {
            4: "Element.ICE",
            5: "Element.THUNDER",
            6: "Element.FIRE",
            7: "Element.JUMP",
        }
        for bit_pos, element in element_map.items():
            if bitfield & (1 << bit_pos):
                elements.append(element)
        return elements

    def _parse_bitfield_to_statuses(self, bitfield):
        """Parse bitfield to Status enum list."""
        statuses = []
        status_map = {
            0: "Status.MUTE",
            1: "Status.SLEEP",
            2: "Status.POISON",
            3: "Status.FEAR",
            4: "Status.BERSERK",
            5: "Status.MUSHROOM",
            6: "Status.SCARECROW",
            7: "Status.INVINCIBLE",
        }
        for bit_pos, status in status_map.items():
            if bitfield & (1 << bit_pos):
                statuses.append(status)
        return statuses

    def _parse_bitfield_to_buffs(self, bitfield):
        """Parse bitfield to TempStatBuff enum list."""
        buffs = []
        buff_map = {
            3: "TempStatBuff.MAGIC_ATTACK",
            4: "TempStatBuff.ATTACK",
            5: "TempStatBuff.MAGIC_DEFENSE",
            6: "TempStatBuff.DEFENSE",
        }
        for bit_pos, buff in buff_map.items():
            if bitfield & (1 << bit_pos):
                buffs.append(buff)
        return buffs

    def _generate_class_definition(self, item_id, data, class_name):
        """Generate Python class definition for an item."""
        lines = []

        # determine base class based on item_id ranges
        # 0-36: weapon, 37-73: armor, 74-95: accessory, 96+: regularitem
        if data["type_value"] == 0:
            base_class = "Weapon"
        elif data["type_value"] == 1:
            base_class = "Armor"
        elif data["type_value"] == 2:
            base_class = "Accessory"
        else:
            base_class = "RegularItem"

        lines.append(f"class {class_name}({base_class}):")
        lines.append(f'    """{data["name"]} item class"""')
        lines.append(f'    _item_name: str = "{data["name"]}"')

        # add prefix if present
        if data["prefix_byte"] is not None:
            prefix_map = {
                0x7F: "ItemPrefix.EMPTY_SPACE",
                0x22: "ItemPrefix.HAMMER",
                0x28: "ItemPrefix.WAND",
                0x23: "ItemPrefix.SHELL",
                0x21: "ItemPrefix.GLOVE",
                0x29: "ItemPrefix.GUN",
                0x25: "ItemPrefix.MUSIC",
                0x27: "ItemPrefix.CHOMP",
                0x26: "ItemPrefix.FAN",
                0x3C: "ItemPrefix.SHIRT",
                0x3D: "ItemPrefix.RING",
                0x2D: "ItemPrefix.CONSUMABLE",
                0x2E: "ItemPrefix.DOT",
                0x3B: "ItemPrefix.BOMB",
                0x3F: "ItemPrefix.QUESTION",
            }
            if data["prefix_byte"] in prefix_map:
                lines.append(f"    _prefix = {prefix_map[data['prefix_byte']]}")

        lines.append("")
        lines.append(f"    _item_id: int = {item_id}")

        # add description (escape backslashes, newlines, and quotes)
        desc = data.get("description", "")
        desc_escaped = desc.replace("\\", "\\\\").replace('\n', '\\n').replace('"', '\\"')
        lines.append(f'    _description: str = "{desc_escaped}"')

        # equip chars
        if data["equip_chars"]:
            chars_str = ", ".join(data["equip_chars"])
            lines.append(f"    _equip_chars: List[PartyCharacter] = [{chars_str}]")

        # stats
        if data["speed"] != 0:
            lines.append(f"    _speed: int = {data['speed']}")
        if data["attack"] != 0:
            lines.append(f"    _attack: int = {data['attack']}")
        if data["defense"] != 0:
            lines.append(f"    _defense: int = {data['defense']}")
        if data["magic_attack"] != 0:
            lines.append(f"    _magic_attack: int = {data['magic_attack']}")
        if data["magic_defense"] != 0:
            lines.append(f"    _magic_defense: int = {data['magic_defense']}")
        if data["variance"] != 0:
            lines.append(f"    _variance: int = {data['variance']}")
        if data["inflict"] != 0:
            lines.append(f"    _inflict: int = {data['inflict']}")

        # price
        lines.append(f"    _price: int = {data['price']}")

        # effect type
        if data["effect_type"] != 0:
            effect_map = {
                0x01: "EffectType.PROTECTION",
                0x02: "EffectType.INFLICTION",
                0x04: "EffectType.NULLIFICATION",
            }
            if data["effect_type"] in effect_map:
                lines.append(f"    _effect_type = {effect_map[data['effect_type']]}")

        # inflict type
        if 0 <= data["inflict_type"] < 8:
            inflict_map = {
                0: "InflictFunction.ITEM_MORPH",
                1: "InflictFunction.REVIVE",
                2: "InflictFunction.RESTORE_FP",
                3: "InflictFunction.INCREASE_STATS_ITEM",
                4: "InflictFunction.RESTORE_HP",
                5: "InflictFunction.RESTORE_ALL_HP_FP",
                6: "InflictFunction.RAISE_MAX_FP",
                7: "InflictFunction.INSTANT_DEATH",
            }
            lines.append(f"    _inflict_type = {inflict_map[data['inflict_type']]}")
        else:
            lines.append(f"    _inflict_type = None")

        # inflict element
        if data["inflict_element"] != 0:
            element_map = {
                0x10: "Element.ICE",
                0x20: "Element.THUNDER",
                0x40: "Element.FIRE",
                0x80: "Element.JUMP",
            }
            if data["inflict_element"] in element_map:
                lines.append(f"    _inflict_element = {element_map[data['inflict_element']]}")

        # flags
        if data["prevent_ko"]:
            lines.append("    _prevent_ko: bool = True")
        if data["hide_damage"]:
            lines.append("    _hide_damage: bool = True")
        if data["usable_battle"]:
            lines.append("    _usable_battle: bool = True")
        if data["usable_overworld"]:
            lines.append("    _usable_overworld: bool = True")
        if data["reusable"]:
            lines.append("    _reusable: bool = True")

        # overworld menu behavior
        if data["overworld_menu_fp"]:
            lines.append("    _overworld_menu_behaviour: OverworldMenuBehaviour = OverworldMenuBehaviour.LEAD_TO_FP")
        if data["overworld_menu_fill_hp"]:
            lines.append("    _overworld_menu_fill_hp: bool = True")
        if data["overworld_menu_fill_fp"]:
            lines.append("    _overworld_menu_fill_fp: bool = True")

        # targeting
        if data["can_target_others"]:
            lines.append("    _can_target_others: bool = True")
        if not data["can_target_self"]:
            lines.append("    _can_target_self: bool = False")
        if data["target_enemies"]:
            lines.append("    _target_enemies: bool = True")
        if data["target_all"]:
            lines.append("    _target_all: bool = True")
        if data["koed_target_only"]:
            lines.append("    _koed_target_only: bool = True")
        if data["one_side_only"]:
            lines.append("    _one_side_only: bool = True")

        # elemental immunities
        if data["elemental_immunities"]:
            immunities_str = ", ".join(data["elemental_immunities"])
            lines.append(f"    _elemental_immunities: List[Element] = [{immunities_str}]")

        # elemental resistances
        if data["elemental_resistances"]:
            resistances_str = ", ".join(data["elemental_resistances"])
            lines.append(f"    _elemental_resistances: List[Element] = [{resistances_str}]")

        # status immunities
        if data["status_immunities"]:
            immunities_str = ", ".join(data["status_immunities"])
            lines.append(f"    _status_immunities: List[Status] = [{immunities_str}]")

        # temp buffs
        if data["temp_buffs"]:
            buffs_str = ", ".join(data["temp_buffs"])
            lines.append(f"    _temp_buffs: List[TempStatBuff] = [{buffs_str}]")

        # weapon timing data
        if data["timing_data"]:
            timing = data["timing_data"]
            lines.append(f"    _half_time_window_begins = UInt8({timing['half_time_begins']})")
            lines.append(f"    _perfect_window_begins = UInt8({timing['perfect_begins']})")
            lines.append(f"    _perfect_window_ends = UInt8({timing['perfect_ends']})")
            lines.append(f"    _half_time_window_ends = UInt8({timing['half_time_ends']})")

        return lines
