import * as cdk from 'aws-cdk-lib';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as apigateway from 'aws-cdk-lib/aws-apigateway';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
import { CloudFrontDistribution } from 'must-cdk';

export class CloudFrontDistributionStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Create S3 bucket for static assets
    const staticAssetsBucket = new s3.Bucket(this, 'StaticAssetsBucket', {
      bucketName: 'my-app-static-assets',
      versioned: true,
      encryption: s3.BucketEncryption.S3_MANAGED,
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
      removalPolicy: cdk.RemovalPolicy.DESTROY // Use RETAIN for production
    });

    // Create Lambda function for API
    const apiLambda = new lambda.Function(this, 'ApiLambda', {
      runtime: lambda.Runtime.NODEJS_18_X,
      handler: 'index.handler',
      code: lambda.Code.fromInline(`
        exports.handler = async (event) => {
          return {
            statusCode: 200,
            headers: {
              'Content-Type': 'application/json',
              'Access-Control-Allow-Origin': '*'
            },
            body: JSON.stringify({
              message: 'Hello from API!',
              timestamp: new Date().toISOString()
            })
          };
        };
      `),
      timeout: cdk.Duration.seconds(30)
    });

    // Create API Gateway
    const api = new apigateway.RestApi(this, 'Api', {
      restApiName: 'My App API',
      description: 'API for my application',
      defaultCorsPreflightOptions: {
        allowOrigins: apigateway.Cors.ALL_ORIGINS,
        allowMethods: apigateway.Cors.ALL_METHODS,
        allowHeaders: ['Content-Type', 'Authorization']
      }
    });

    const apiIntegration = new apigateway.LambdaIntegration(apiLambda);
    api.root.addResource('api').addMethod('GET', apiIntegration);

    // Create CloudFront distribution with multiple origins
    const distribution = new CloudFrontDistribution(this, 'AppDistribution', {
      comment: 'Multi-origin CloudFront distribution for web app',
      
      // Primary origin (S3 for static assets)
      defaultOrigin: {
        domainName: staticAssetsBucket.bucketDomainName,
        originPath: '/static',
        customOriginConfig: {
          httpPort: 80,
          httpsPort: 443,
          originProtocolPolicy: 'https-only'
        }
      },

      // Additional origins
      additionalOrigins: [
        {
          id: 'api-origin',
          domainName: `${api.restApiId}.execute-api.${this.region}.amazonaws.com`,
          originPath: `/${api.deploymentStage.stageName}`,
          customOriginConfig: {
            httpPort: 80,
            httpsPort: 443,
            originProtocolPolicy: 'https-only',
            originSslProtocols: ['TLSv1.2']
          }
        }
      ],

      // Cache behaviors
      defaultCacheBehavior: {
        targetOriginId: 'default',
        viewerProtocolPolicy: 'redirect-to-https',
        allowedMethods: ['GET', 'HEAD', 'OPTIONS'],
        cachedMethods: ['GET', 'HEAD'],
        compress: true,
        cachePolicyId: '4135ea2d-6df8-44a3-9df3-4b5a84be39ad', // CachingOptimized
        originRequestPolicyId: '88a5eaf4-2fd4-4709-b370-b4c650ea3fcf', // CORS-S3Origin
        responseHeadersPolicyId: '5cc3b908-e619-4b99-88e5-2cf7f45965bd' // SecurityHeadersPolicy
      },

      additionalCacheBehaviors: [
        {
          pathPattern: '/api/*',
          targetOriginId: 'api-origin',
          viewerProtocolPolicy: 'redirect-to-https',
          allowedMethods: ['GET', 'HEAD', 'OPTIONS', 'PUT', 'POST', 'PATCH', 'DELETE'],
          cachedMethods: ['GET', 'HEAD'],
          compress: true,
          cachePolicyId: '4135ea2d-6df8-44a3-9df3-4b5a84be39ad',
          originRequestPolicyId: 'b689b0a8-53d0-40ab-baf2-68738e2966ac', // AllViewer
          responseHeadersPolicyId: '5cc3b908-e619-4b99-88e5-2cf7f45965bd'
        }
      ],

      // Custom domain configuration
      customDomain: {
        domainName: 'app.example.com',
        certificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
        aliases: ['www.app.example.com']
      },

      // Security configuration
      webAclId: 'arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a',
      
      // Geographic restrictions
      geoRestriction: {
        restrictionType: 'whitelist',
        locations: ['US', 'CA', 'GB', 'DE', 'FR']
      },

      // Error pages
      customErrorResponses: [
        {
          errorCode: 404,
          responseCode: 200,
          responsePagePath: '/index.html',
          errorCachingMinTtl: 300
        },
        {
          errorCode: 403,
          responseCode: 200,
          responsePagePath: '/index.html',
          errorCachingMinTtl: 300
        }
      ],

      // Logging configuration
      enableLogging: true,
      logBucket: staticAssetsBucket,
      logFilePrefix: 'cloudfront-logs/',
      logIncludeCookies: false,

      // Performance settings
      priceClass: 'PriceClass_100', // Use only North America and Europe edge locations
      httpVersion: 'http2',
      enableIpv6: true,

      // Additional security headers
      responseHeadersPolicy: {
        securityHeadersConfig: {
          strictTransportSecurity: {
            accessControlMaxAgeSec: 31536000,
            includeSubdomains: true,
            override: true
          },
          contentTypeOptions: {
            override: true
          },
          frameOptions: {
            frameOption: 'DENY',
            override: true
          },
          referrerPolicy: {
            referrerPolicy: 'strict-origin-when-cross-origin',
            override: true
          }
        }
      }
    });

    // Grant CloudFront access to S3 bucket
    staticAssetsBucket.grantRead(distribution.originAccessIdentity);

    // Output important values
    new cdk.CfnOutput(this, 'DistributionDomainName', {
      value: distribution.distributionDomainName,
      description: 'CloudFront Distribution Domain Name'
    });

    new cdk.CfnOutput(this, 'DistributionId', {
      value: distribution.distributionId,
      description: 'CloudFront Distribution ID'
    });

    new cdk.CfnOutput(this, 'S3BucketName', {
      value: staticAssetsBucket.bucketName,
      description: 'S3 Bucket Name for Static Assets'
    });
  }
}
