"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmplifyApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const amplify = require("aws-cdk-lib/aws-amplify");
const constructs_1 = require("constructs");
class AmplifyApp extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        // Get GitHub authentication
        const accessToken = this.getGitHubAuthentication(props);
        if (!accessToken) {
            throw new Error(`
❌ GitHub authentication is required for Amplify App deployment!

Please provide authentication using one of these methods:

1. 🔑 Personal Access Token (Current approach):
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: 'your-github-token'
   })

2. 📱 Use GitHub CLI (for development):
   gh auth login
   
3. 🌍 Set environment variable:
   export MUFIN_PUSH_TOKEN=your-github-token

4. 🔒 Use AWS Secrets Manager:
   AmplifyApp(this, 'MyApp', {
     appName: 'my-app',
     repository: 'https://github.com/user/repo',
     accessToken: SecretValue.secretsManager('github-token').unsafeUnwrap()
   })

💡 Production Recommendation:
   After your project is set up and working, consider migrating to GitHub Apps
   for better security, organization support, and higher rate limits.
   See: https://docs.github.com/en/apps/creating-github-apps
      `);
        }
        console.log("🚀 Creating Amplify App with authenticated GitHub access...");
        // Create Amplify App
        this.app = new amplify.CfnApp(this, "App", {
            name: props.appName,
            repository: props.repository,
            accessToken: accessToken,
            environmentVariables: this.formatEnvVars(props.environmentVariables),
            buildSpec: props.buildSpec || this.getDefaultBuildSpec(props.platform),
            platform: props.platform || "WEB_COMPUTE",
            customRules: props.customRules || this.getDefaultCustomRules(),
            autoBranchCreationConfig: {
                enableAutoBuild: props.buildSettings?.enableBranchAutoBuild ?? false,
                enableAutoBranchCreation: props.buildSettings?.enableAutoBranchCreation ?? false,
                enablePullRequestPreview: true,
            },
            basicAuthConfig: props.basicAuth
                ? {
                    enableBasicAuth: props.basicAuth.enableBasicAuth ?? true,
                    username: props.basicAuth.username,
                    password: props.basicAuth.password,
                }
                : undefined,
            tags: props.tags
                ? Object.entries(props.tags).map(([key, value]) => ({ key, value }))
                : undefined,
        });
        // Create branches
        if (props.branches) {
            props.branches.forEach((branch, index) => {
                const cfnBranch = new amplify.CfnBranch(this, `Branch${index}`, {
                    appId: this.app.attrAppId,
                    branchName: branch.branchName,
                    stage: branch.stage || "PRODUCTION",
                    environmentVariables: this.formatEnvVars(branch.environmentVariables),
                    enableAutoBuild: branch.enableAutoBuild ?? true,
                    framework: branch.framework,
                    buildSpec: branch.buildSpec,
                });
                this.branches.push(cfnBranch);
            });
        }
        else {
            // Default main branch
            const mainBranch = new amplify.CfnBranch(this, "MainBranch", {
                appId: this.app.attrAppId,
                branchName: "main",
                stage: "PRODUCTION",
                enableAutoBuild: true,
            });
            this.branches.push(mainBranch);
        }
        // Custom domain setup
        if (props.customDomain) {
            this.domain = new amplify.CfnDomain(this, "Domain", {
                appId: this.app.attrAppId,
                domainName: props.customDomain.domainName,
                enableAutoSubDomain: props.customDomain.enableAutoSubdomain,
                autoSubDomainCreationPatterns: props.customDomain.autoSubdomainCreationPatterns,
                subDomainSettings: props.customDomain.subDomains?.map((sub) => ({
                    branchName: sub.branchName,
                    prefix: sub.prefix,
                })) || [
                    {
                        branchName: "main",
                        prefix: "",
                    },
                ],
            });
        }
        // Outputs
        new aws_cdk_lib_1.CfnOutput(this, "AppId", {
            value: this.app.attrAppId,
            description: "Amplify App ID",
        });
        new aws_cdk_lib_1.CfnOutput(this, "DefaultDomain", {
            value: this.app.attrDefaultDomain,
            description: "Amplify Default Domain",
        });
        if (this.domain) {
            new aws_cdk_lib_1.CfnOutput(this, "CustomDomain", {
                value: props.customDomain.domainName,
                description: "Amplify Custom Domain",
            });
        }
    }
    formatEnvVars(envVars) {
        if (!envVars)
            return undefined;
        return Object.entries(envVars).map(([name, value]) => ({ name, value }));
    }
    getDefaultCustomRules() {
        return [
            {
                source: "/<*>",
                target: "/index.html",
                status: "404-200",
            },
        ];
    }
    getGitHubAuthentication(props) {
        console.log("🔍 Attempting to retrieve GitHub authentication...");
        // Priority 1: Direct access token
        if (props.accessToken) {
            console.log("🔑 Using provided access token...");
            console.log(`🔑 Token length: ${props.accessToken.length} characters`);
            console.log(`🔑 Token prefix: ${props.accessToken.substring(0, 7)}...`);
            return props.accessToken;
        }
        // Priority 2: Fallback to GitHub CLI or environment variable
        console.log("🔄 Falling back to GitHub CLI or environment variable...");
        return this.getGitHubToken();
    }
    getGitHubToken() {
        console.log("🔍 Attempting to retrieve GitHub token...");
        // Check if we're in a test environment
        const isTestEnvironment = process.env.NODE_ENV === "test" ||
            process.env.JEST_WORKER_ID !== undefined ||
            process.env.CI === "true";
        if (isTestEnvironment) {
            console.log("🧪 Test environment detected, using mock token");
            return process.env.MUFIN_PUSH_TOKEN || "mock-github-token-for-testing";
        }
        try {
            // Try to get token from gh CLI first
            console.log("📱 Trying to get token from GitHub CLI (gh auth token)...");
            const token = (0, child_process_1.execSync)("gh auth token", {
                encoding: "utf8",
                stdio: "pipe",
            })
                .toString()
                .trim();
            if (token && token.length > 0) {
                console.log("✅ Successfully retrieved GitHub token from gh CLI");
                console.log("💡 Tip: For production deployments, consider migrating to GitHub Apps");
                console.log("   for better security and organization support after setup is complete.");
                return token;
            }
            else {
                console.log("⚠️  gh CLI returned empty token");
            }
        }
        catch (error) {
            console.log("❌ Failed to get token from gh CLI:");
            console.log(`   Error: ${error.message}`);
            console.log("   This might mean:");
            console.log("   - GitHub CLI is not installed");
            console.log("   - You're not authenticated with gh CLI");
            console.log("   - Run 'gh auth login' to authenticate");
        }
        // Fallback to environment variable
        console.log("🔄 Falling back to environment variable MUFIN_PUSH_TOKEN...");
        const envToken = process.env.MUFIN_PUSH_TOKEN;
        if (envToken && envToken.length > 0) {
            console.log("✅ Successfully retrieved GitHub token from environment variable");
            console.log("💡 Tip: For production deployments, consider migrating to GitHub Apps");
            console.log("   for better security and organization support after setup is complete.");
            return envToken;
        }
        else {
            console.log("⚠️  Environment variable MUFIN_PUSH_TOKEN is not set or empty");
        }
        console.log("❌ No GitHub token found! Please either:");
        console.log("   1. Run 'gh auth login' to authenticate with GitHub CLI, or");
        console.log("   2. Set the MUFIN_PUSH_TOKEN environment variable");
        console.log("   3. Pass accessToken directly in AmplifyAppProps");
        console.log("");
        console.log("💡 After setup: Consider GitHub Apps for production deployments");
        return undefined;
    }
    getDefaultBuildSpec(platform) {
        if (platform === "WEB_COMPUTE") {
            return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - yarn install
    build:
      commands:
        - yarn run build
  artifacts:
    baseDirectory: .next
    files:
      - '**/*'
  cache:
    paths:
      - .next/cache/**/*
      - node_modules/**/*`;
        }
        return `version: 1
frontend:
  phases:
    preBuild:
      commands:
        - npm ci
    build:
      commands:
        - npm run build
  artifacts:
    baseDirectory: dist
    files:
      - '**/*'
  cache:
    paths:
      - node_modules/**/*`;
    }
}
exports.AmplifyApp = AmplifyApp;
_a = JSII_RTTI_SYMBOL_1;
AmplifyApp[_a] = { fqn: "must-cdk.AmplifyApp", version: "0.0.85" };
//# sourceMappingURL=data:application/json;base64,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