"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigatewayv2 = require("aws-cdk-lib/aws-apigatewayv2");
const apigatewayv2_integrations = require("aws-cdk-lib/aws-apigatewayv2-integrations");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
class WebSocketApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = props.lambdaFunction;
        // Create WebSocket API
        this.webSocketApi = new apigatewayv2.WebSocketApi(this, "WebSocketApi", {
            apiName: props.apiName,
            description: `WebSocket API for ${props.apiName}`,
            ...props.apiProps,
        });
        // Create default integration for the primary lambda
        const defaultIntegration = new apigatewayv2_integrations.WebSocketLambdaIntegration("DefaultIntegration", this.lambdaFunction);
        // Add default route
        this.webSocketApi.addRoute("$default", {
            integration: defaultIntegration,
        });
        // Add custom routes if provided
        if (props.customRoutes) {
            this.addCustomRoutes(props.customRoutes);
        }
        // Create stage
        const stageName = props.stageName || "dev";
        this.webSocketStage = new apigatewayv2.WebSocketStage(this, "WebSocketStage", {
            webSocketApi: this.webSocketApi,
            stageName: stageName,
            autoDeploy: true,
        });
        // Setup CloudWatch logging
        if (props.enableLogging) {
            const loggingResources = (0, utils_1.setupLogging)(this, props.apiName, props.logGroupProps);
            this.apiGatewayLogGroup = loggingResources.logGroup;
        }
        // Setup custom domain
        if (props.customDomainName) {
            const domainResources = this.setupCustomDomain(props);
            this.certificate = domainResources.certificate;
            this.domain = domainResources.domain;
            this.aRecord = domainResources.aRecord;
        }
    }
    /**
     * Add custom routes to the WebSocket API
     */
    addCustomRoutes(routes) {
        routes.forEach((route) => {
            const integration = new apigatewayv2_integrations.WebSocketLambdaIntegration(`${route.routeKey}Integration`, route.handler);
            this.webSocketApi.addRoute(route.routeKey, {
                integration: integration,
                returnResponse: route.routeResponseSelectionExpression !== undefined,
            });
        });
    }
    /**
     * Setup custom domain with certificate and Route53 record
     */
    setupCustomDomain(props) {
        let certificate;
        // Use existing certificate or create new one
        if (props.existingCertificate) {
            certificate = props.existingCertificate;
        }
        else if (props.hostedZone) {
            certificate = new acm.Certificate(this, "Certificate", {
                domainName: props.customDomainName,
                validation: acm.CertificateValidation.fromDns(props.hostedZone),
            });
        }
        else {
            throw new Error("Either certificateArn or hostedZone must be provided for custom domain");
        }
        // Create custom domain for WebSocket API
        const domain = new apigatewayv2.DomainName(this, "CustomDomain", {
            domainName: props.customDomainName,
            certificate: certificate,
        });
        // Create API mapping
        new apigatewayv2.ApiMapping(this, "ApiMapping", {
            api: this.webSocketApi,
            domainName: domain,
            stage: this.webSocketStage,
        });
        let aRecord;
        // Create Route53 alias record if hosted zone provided
        if (props.hostedZone) {
            aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
                zone: props.hostedZone,
                recordName: props.customDomainName,
                target: route53.RecordTarget.fromAlias(new targets.ApiGatewayv2DomainProperties(domain.regionalDomainName, domain.regionalHostedZoneId)),
            });
        }
        return { certificate, domain, aRecord };
    }
    /**
     * Add a custom route after construction (for dynamic route addition)
     */
    addRoute(route) {
        const integration = new apigatewayv2_integrations.WebSocketLambdaIntegration(`${route.routeKey}Integration`, route.handler);
        return this.webSocketApi.addRoute(route.routeKey, {
            integration: integration,
            returnResponse: route.routeResponseSelectionExpression !== undefined,
        });
    }
    /**
     * Get the WebSocket API URL (useful for outputs)
     */
    get webSocketUrl() {
        if (this.domain) {
            return `wss://${this.domain.name}`;
        }
        return this.webSocketStage.url;
    }
}
exports.WebSocketApiGatewayToLambda = WebSocketApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
WebSocketApiGatewayToLambda[_a] = { fqn: "must-cdk.WebSocketApiGatewayToLambda", version: "0.0.85" };
//# sourceMappingURL=data:application/json;base64,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