import React, { useState, useEffect } from "react";
import "./App.css";
import "@aws-amplify/ui-react/styles.css";
import { generateClient } from "aws-amplify/data";
import { uploadData, getUrl, remove } from "aws-amplify/storage";
import {
  Button,
  Flex,
  Heading,
  Image,
  Text,
  TextField,
  View,
  withAuthenticator,
} from "@aws-amplify/ui-react";
import type { Schema } from "./types";

const client = generateClient<Schema>();

interface Note {
  id: string;
  name: string;
  description?: string | null;
  image?: string | null;
}

interface AppProps {
  signOut?: () => void;
}

const App: React.FC<AppProps> = ({ signOut }) => {
  const [notes, setNotes] = useState<Note[]>([]);

  useEffect(() => {
    fetchNotes();
  }, []);

  async function fetchNotes() {
    const { data: notesFromAPI } = await client.models.Note.list();
    
    // Process images
    const notesWithImages = await Promise.all(
      notesFromAPI.map(async (note) => {
        if (note.image) {
          try {
            const { url } = await getUrl({ key: note.name });
            return { ...note, image: url.toString() };
          } catch (error) {
            console.error("Error getting image URL:", error);
            return note;
          }
        }
        return note;
      })
    );
    
    setNotes(notesWithImages);
  }

  async function createNote(event: React.FormEvent<HTMLFormElement>) {
    event.preventDefault();
    const form = new FormData(event.target as HTMLFormElement);
    const image = form.get("image") as File;
    const name = form.get("name") as string;
    const description = form.get("description") as string;
    
    const data = {
      name,
      description,
      image: image?.name || null,
    };

    // Upload image if provided
    if (image && image.size > 0) {
      try {
        await uploadData({
          key: name,
          data: image,
        });
      } catch (error) {
        console.error("Error uploading image:", error);
      }
    }

    // Create note in database
    await client.models.Note.create(data);
    
    // Refresh notes list
    fetchNotes();
    
    // Reset form
    (event.target as HTMLFormElement).reset();
  }

  async function deleteNote(note: Note) {
    const newNotes = notes.filter((n) => n.id !== note.id);
    setNotes(newNotes);
    
    // Remove image from storage if it exists
    if (note.image) {
      try {
        await remove({ key: note.name });
      } catch (error) {
        console.error("Error removing image:", error);
      }
    }
    
    // Delete note from database
    await client.models.Note.delete({ id: note.id });
  }

  return (
    <View className="App">
      <Heading level={1}>My Notes App</Heading>
      <View as="form" margin="3rem 0" onSubmit={createNote}>
        <Flex direction="row" justifyContent="center">
          <TextField
            name="name"
            placeholder="Note Name"
            label="Note Name"
            labelHidden
            variation="quiet"
            required
          />
          <TextField
            name="description"
            placeholder="Note Description"
            label="Note Description"
            labelHidden
            variation="quiet"
            required
          />
          <View
            name="image"
            as="input"
            type="file"
            style={{ alignSelf: "end" }}
          />
          <Button type="submit" variation="primary">
            Create Note
          </Button>
        </Flex>
      </View>
      <Heading level={2}>Current Notes</Heading>
      <View margin="3rem 0">
        {notes.map((note) => (
          <Flex
            key={note.id}
            direction="row"
            justifyContent="center"
            alignItems="center"
          >
            <Text as="strong" fontWeight={700}>
              {note.name}
            </Text>
            <Text as="span">{note.description}</Text>
            {note.image && (
              <Image
                src={note.image}
                alt={`visual aid for ${note.name}`}
                style={{ maxWidth: "100px", maxHeight: "100px" }}
              />
            )}
            <Button variation="link" onClick={() => deleteNote(note)}>
              Delete note
            </Button>
          </Flex>
        ))}
      </View>
      <Button onClick={signOut}>Sign Out</Button>
    </View>
  );
};

export default withAuthenticator(App);
